<?php declare(strict_types = 1);

namespace EshopCatalog\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\ControllerPath;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;

/**
 * Class ProductsController
 * @ControllerPath("/product")
 */
class ProductsController extends BaseController
{
	protected Products $productsDaoService;

	protected ProductsFacade $productsFacade;

	public function __construct(Products $productsDaoService, ProductsFacade $productsFacade)
	{
		$this->productsDaoService = $productsDaoService;
		$this->productsFacade     = $productsFacade;

		ProductsFacade::setMode(ProductsFacade::MODE_CHECKOUT);
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return array
	 * @Path("/")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="start", type="int", description="start", required=false, in="query"),
	 *      @RequestParameter(name="limit", type="int", description="limit", required=false, in="query"),
	 *      @RequestParameter(name="filter", type="string", description="filter", required=false, in="query")
	 * })
	 */
	public function index(ApiRequest $request, ApiResponse $response)
	{
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', 0);
		$sort   = $request->getQueryParam('sort', 'ASC');

		$productIds = $this->productsDaoService->getProductsIdAll($offset, $limit, $sort);

		return $productIds ? array_values($this->productsFacade->getProducts($productIds)) : [];
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return Product[]
	 * @Path("/get/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int")
	 * })
	 */
	public function product(ApiRequest $request, ApiResponse $response)
	{
		$id      = $request->getParameter('id');
		$product = $this->productsFacade->getProduct($id);

		if (!$product) {
			$this->sendError($response, "Product with this id doesn´t exist");
		}

		return [$product];
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return array
	 * @Path("/search/{query}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="query", type="string", required=true),
	 *      @RequestParameter(name="start", type="int", description="start", required=false, in="query"),
	 *      @RequestParameter(name="limit", type="int", description="limit", required=false, in="query"),
	 *      @RequestParameter(name="filter", type="string", description="filter", required=false, in="query"),
	 *      @RequestParameter(name="sort", type="string", description="sort", required=false, in="query")
	 * })
	 */
	public function search(ApiRequest $request, ApiResponse $response)
	{
		$query  = $request->getParameter('query');
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', null);
		$filter = $request->getQueryParam('filter', null);
		$sort   = $request->getQueryParam('sort', null);

		$productIds = $this->productsDaoService->getProductsIdBySearch($query, $offset, $limit, $sort, $filter);

		return $productIds ? array_values($this->productsFacade->getProducts($productIds)) : [];
	}
}
