<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Dao;

use Core\Model\Entities\TSeo;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopGifts\FrontModule\Model\Dao\Gift;
use Gallery\FrontModule\Model\Dao\Album;
use Nette\Utils\DateTime;

class Product
{
	use TSeo;

	/** @var int */
	public $id;

	/** @var int */
	public $quantity;

	/**
	 * Základní cena bez slev
	 *
	 * @var float
	 */
	public $basePrice;

	/** @var float */
	public $price;

	/** @var float */
	public $retailPrice;

	/** @var float */
	public $basePriceInBaseCurrency;

	/** @var float */
	public $priceInBaseCurrency;

	/** @var float */
	public $retailPriceInBaseCurrency;

	/** @var int */
	public $manufacturerId;

	/** @var Manufacturer */
	private $manufacturer;

	/** @var string */
	public $name;

	/** @var string */
	public $name2;

	/** @var string */
	public $shortDescription;

	/** @var string */
	public $description;

	/** @var DateTime */
	public $created;

	/** @var DateTime */
	public $modified;

	/** @var int */
	public $galleryId;

	/** @var Album */
	public $gallery;

	/** @var FeatureProduct[] */
	public $features;

	/** @var self[] */
	protected $related;

	/** @var self[] */
	public $alternative;

	/** @var int */
	public $defaultCategoryId;

	/** @var Category */
	public $defaultCategory;

	/** @var int[] */
	public $categories;

	/** @var Tag[] */
	public $tags;

	/** @var string */
	public $link = '#';

	/** @var int */
	public $vatRate;

	/** @var string */
	protected $ean;

	/** @var string */
	public $code1;

	/** @var string */
	protected $code2;

	/** @var Availability */
	protected $availability;

	/** @var int */
	public $unlimitedQuantity = 0;

	/** @var array */
	protected $extraFields = [];

	/** @var bool */
	public $discountDisabled = false;

	/** @var bool */
	public $categoryGiftsAllowed = true;

	/** @var Gift[] */
	public $gifts = [];

	/** @var bool */
	public $isActive = true;

	/** @var bool */
	public $canAddToCart = true;

	/** @var array|self[] */
	public array $variants = [];

	public ?int $variantId = null;

	public ?int $variantOf = null;

	public array $variantProductIds = [];

	public array $helperData = [];

	/** @var FeatureProduct[] */
	protected array $cVariantDifferences = [];

	public function __construct($id)
	{
		$this->id = $id;
	}

	/**
	 * @return int
	 */
	public function getId()
	{
		return $this->id;
	}

	/*******
	 * == Quantity
	 */

	public function getQuantity(): int
	{
		return (int) $this->quantity;
	}

	public function setQuantity(int $quantity): self
	{
		$this->quantity = $quantity;

		return $this;
	}

	/**
	 * Vrátí text pro překlad s počtem ks skladem
	 *
	 * @return string
	 */
	public function getQuantityText(): string
	{
		$q = $this->getQuantity();

		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1)
			$text = 'inStock';
		else if ($q <= 0)
			$text = 'soldOut';
		else if ($q <= 4)
			$text = 'lastInStock1';
		else if ($q <= 15)
			$text = 'lastInStock2';
		else
			$text = 'inStock';

		return 'eshopCatalogFront.product.' . $text;
	}

	/**
	 * Vrátí stav poštu ks skladem
	 *
	 * @return string
	 */
	public function getQuantityColor(): string
	{
		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1)
			return 'green';

		$q = $this->getQuantity();
		if ($q <= 0)
			return 'red';

		if ($q <= 15)
			return 'yellow';

		return 'green';
	}

	/*******
	 * == Price
	 */

	public function getPrice(): float { return (float) $this->price; }

	public function setPrice(float $price): self
	{
		$this->price = $price;

		return $this;
	}

	public function getBasePrice(): float { return (float) $this->basePrice; }

	public function setBasePrice(float $price): self
	{
		$this->basePrice = $price;

		return $this;
	}

	public function getPriceWithoutVat(): float { return ($this->getPrice() / (1 + ($this->vatRate / 100))); }

	public function getRetailPrice(): float
	{
		return (float) $this->retailPrice;
	}

	/**
	 * @param float $retailPrice
	 */
	public function setRetailPrice(float $retailPrice)
	{
		$this->retailPrice = $retailPrice;
	}

	/**
	 * @return Manufacturer
	 */
	public function getManufacturer()
	{
		return $this->manufacturer;
	}

	/**
	 * @param mixed $manufacturer
	 */
	public function setManufacturer($manufacturer)
	{
		$this->manufacturer = $manufacturer;
	}

	/**
	 * @return string
	 */
	public function getName()
	{
		return $this->name;
	}

	/**
	 * @param string $name
	 */
	public function setName($name)
	{
		$this->name = $name;
	}

	public function setName2($name)
	{
		$this->name2 = $name;
	}

	public function getName2(): ?string { return Config::load('enableProductName2') ? $this->name2 : $this->name; }

	/**
	 * @return string
	 */
	public function getDescription()
	{
		return $this->description;
	}

	/**
	 * @param string $description
	 */
	public function setDescription($description)
	{
		$this->description = $description;
	}

	/**
	 * @return ?Album
	 */
	public function getGallery()
	{
		return $this->gallery;
	}

	/**
	 * @param mixed $gallery
	 */
	public function setGallery($gallery)
	{
		$this->gallery = $gallery;
	}

	/**
	 * @param FeatureProduct[] $features
	 */
	public function setFeatures($features)
	{
		$this->features = $features;
	}

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeatures() { return $this->features; }

	/**
	 * @param string $type
	 *
	 * @return Tag|null
	 */
	public function getTag($type) { return $this->tags[$type] ?? null; }

	/**
	 * @param self[] $array
	 *
	 * @return $this
	 */
	public function setAlternatives($array)
	{
		$this->alternative = $array;

		return $this;
	}

	/** @return int */
	public function getVatRate()
	{
		return $this->vatRate;
	}

	/**
	 * @param int $vatRate
	 *
	 * @return Product
	 */
	public function setVatRate($vatRate): Product
	{
		$this->vatRate = $vatRate;

		return $this;
	}

	/** @return string */
	public function getEan()
	{
		return $this->ean;
	}

	/**
	 * @param string $ean
	 *
	 * @return Product
	 */
	public function setEan($ean): Product
	{
		$this->ean = $ean;

		return $this;
	}

	/**
	 * @return string
	 */
	public function getCode1()
	{
		return $this->code1;
	}

	/**
	 * @param $code1
	 *
	 * @return Product
	 */
	public function setCode1($code1): Product
	{
		$this->code1 = $code1;

		return $this;
	}

	/**
	 * @return string
	 */
	public function getCode2()
	{
		return $this->code2;
	}

	/**
	 * @param $code2
	 *
	 * @return Product
	 */
	public function setCode2($code2): Product
	{
		$this->code2 = $code2;

		return $this;
	}

	public function setExtraFields(array $array): self
	{
		$this->extraFields = $array;

		return $this;
	}

	public function getExtraFields(): array { return $this->extraFields; }

	/**
	 * @param string $key
	 * @param mixed  $default
	 *
	 * @return mixed|null
	 */
	public function getExtraField(string $key, $default = null) { return $this->extraFields[$key] ?? $default; }

	/** @return Gift[] */
	public function getGifts(): array { return $this->gifts; }

	public function getAvailability(): Availability { return $this->availability; }

	public function setAvailability(?Availability $availability = null): self
	{
		$this->availability = $availability;

		return $this;
	}

	/**
	 * ====== Variants
	 */

	public function hasVariants(): bool { return $this->variants ? true : false; }

	public function getVariantDifferences(): array
	{
		if (empty($this->cVariantDifferences)) {
			$used     = [];
			$features = [];
			foreach ($this->features as $v)
				$features[$v->getKey()] = $v;

			foreach ($this->variants as $v)
				foreach ($v->features as $f)
					$used[$f->getKey()]++;

			foreach ($used as $k => $v)
				if ($v === count($this->variants))
					unset($features[$k]);

			if (count($features) === 1) {
				$fKey = explode('-', key($features))[0];
				uksort($this->variants, function($a, $b) use ($fKey) {
					return $this->variants[$b]->features[$fKey]->position <=> $this->variants[$a]->features[$fKey]->position;
				});
			}

			$this->cVariantDifferences = $features;
		}

		return $this->cVariantDifferences;
	}

	public function getFromPrice(): float
	{
		$min = $this->price;

		foreach ($this->variants as $v)
			if ($v->price < $min)
				$min = $v->price;

		return $min;
	}

	/**
	 * @return self[]
	 */
	public function getRelated(): array
	{
		return is_array($this->related) ? $this->related : [];
	}

	/**
	 * @param Product[] $products
	 */
	public function setRelated(array $products): void
	{
		$this->related = $products;
	}
}
