<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Export;

use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryExport;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductExport;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\Model\PaymentSpeditions;
use Nette\Localization\ITranslator;

class Data
{
	protected EntityManagerDecorator $em;

	protected ITranslator $translator;

	protected Categories $categories;

	protected ProductsFacade $productsFacade;

	protected PaymentSpeditions $paymentSpeditions;

	protected ?array $cCategories = null;

	public function __construct(EntityManagerDecorator $em, ITranslator $translator,
	                            Categories $categories, ProductsFacade $productsFacade, PaymentSpeditions $paymentSpeditions)
	{
		$this->em                = $em;
		$this->translator        = $translator;
		$this->categories        = $categories;
		$this->productsFacade    = $productsFacade;
		$this->paymentSpeditions = $paymentSpeditions;
	}

	public function getProductsData(int $start = 0, int $limit = 1000): array
	{
		$ids               = [];
		$exports           = [];
		$categoryExports   = [];
		$result            = [];
		$minimumPrice      = Config::load('exportMinimumPrice', 0);

		foreach ((new ProductQuery($this->translator->getLocale()))
			         ->onlyInStockOrSupplier()
			         ->selectIds()
			         ->getQueryBuilder($this->em->getRepository(Product::class))
			         ->setMaxResults($limit)->getQuery()->setFirstResult($start)->getScalarResult() as $row)
			$ids[] = $row['id'];

		foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('pe')
			         ->where('pe.id IN (:ids)')->andWhere('pe.lang = :lang')
			         ->setParameters(['ids' => $ids, 'lang' => $this->translator->getLocale()])
			         ->getQuery()->getArrayResult() as $row)
			$exports[$row['id']][$row['service']] = $row;

		foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('ce')
			         ->where('ce.lang = :lang')->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getArrayResult() as $row)
			$categoryExports[$row['id']][$row['service']] = $row;

		if ($ids) {
			foreach ($this->productsFacade->getProducts($ids) as $product) {
				if ($product->getQuantity() <= 0 || $product->getPrice() < $minimumPrice || !$product->canAddToCart)
					continue;

				$category = $product->defaultCategory;

				$catExports = $categoryExports[$product->defaultCategoryId];
				$needLoop   = [];
				$cat        = $category;
				while (count($needLoop) !== 3 && $cat) {
					if (isset($categoryExports[$cat->id])) {
						foreach ($categoryExports[$cat->id] as $service => $data) {
							if (!isset($needLoop[$service]))
								$catExports[$service] = $data;
							if ($data['status'] !== 2)
								$needLoop[$service] = false;
						}
					}
					$cat = $cat->getParent();
				}

				if (!$category)
					continue;

				$product->setExtraFields([
					'export'          => $exports[$product->getId()],
					'category'        => $category,
					'categoryExports' => $catExports,
				]);

				$result[] = $product;
			}
		}

		return $result;
	}

	public function getCategories(): array
	{
		if ($this->cCategories === null) {
			$this->cCategories = [];

			$exports = [];
			foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('ce')
				         ->where('ce.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->getQuery()->getArrayResult() as $row)
				$exports[$row['id']][$row['service']] = $row;

			$categories = [];
			foreach ($this->em->getRepository(Category::class)->createQueryBuilder('c')
				         ->select('c.id, IDENTITY(c.parent) as parent, c.isPublished, ct.name')
				         ->innerJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->getQuery()->getArrayResult() as $row) {
				$categories[$row['id']] = $row;
			}

			foreach ($categories as $category) {
				$name      = $category['name'];
				$export    = [];
				$parent    = $categories[$category['parent']];
				$parentIds = [];

				if ($exports[$category['id']])
					$export[] = $exports[$category['id']];

				$i = 0;
				while ($parent && $i < 20) {
					$parentIds[] = $parent['id'];
					$name        = $parent['name'] . ' | ' . $name;

					$parent = $categories[$parent['parent']];
					$i++;
				}

				$data = [
					'categoryText' => $name,
					'status'       => $category['isPublished'] ? 1 : 0,
					'exports'      => [],
				];

				foreach ($parentIds as $id) {
					if ($exports[$id])
						$export[] = $exports[$id];
				}

				foreach ($export as $y => $vals) {
					foreach ($vals as $k => $v) {
						$data['exports'][$y][$k] = [
							'categoryText' => $v['categoryText'] ?: $data['categoryText'],
							'status'       => $v['status'],
						];
					}
				}

				$this->cCategories[$category['id']] = $data;
			}
		}

		return $this->cCategories;
	}
}
