<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Export;

use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Provider\IXmlExportService;
use Nette\Utils\Strings;

class HeurekaXmlService extends XmlService implements IXmlExportService
{
	public string $name = 'heureka';

	public function startExport(): void
	{
		parent::startExport();

		$this->writeToFile('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<SHOP>' . PHP_EOL);

		$this->startExportStock();
	}

	public function startExportStock(): void
	{
		parent::startExportStock();

		$this->writeToFileStock('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<item_list>' . PHP_EOL);
	}

	public function writeItem(Product $product)
	{
		$export = $this->prepareExportData($product);

		if ($export['status'] == 0)
			return;

		$dom               = new \DOMDocument();
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('SHOPITEM');

		$child = $dom->createElement('ITEM_ID', (string) $product->getId());
		$shopitem->appendChild($child);

		if ($product->variantId) {
			$child = $dom->createElement('ITEMGROUP_ID', (string) $product->variantId);
			$shopitem->appendChild($child);
		}

		$child     = $dom->createElement('PRODUCTNAME');
		$childText = $dom->createTextNode((string) $export['productName']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('PRODUCT');
		$childText = $dom->createTextNode((string) $export['product']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('DESCRIPTION');
		$childText = $dom->createCDATASection(Strings::normalize(strip_tags((string) $product->getDescription())));
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child = $dom->createElement('URL', (string) $product->link);
		$shopitem->appendChild($child);

		$child = $dom->createElement('PRICE_VAT', number_format($product->getPrice(), 2, ".", ""));
		$shopitem->appendChild($child);

		if ($product->getVatRate()) {
			$child = $dom->createElement('VAT', $product->getVatRate() . '%');
			$shopitem->appendChild($child);
		}

		$child = $dom->createElement('DELIVERY_DATE', (string) $product->getAvailability()->getDelay());
		$shopitem->appendChild($child);

		foreach ($this->getSpeditionsForPrice($product->getPrice()) as $country => $speditions) {
			if ($country !== 'cz')
				continue;

			foreach ($speditions as $spedName => $spedition) {
				$delivery = $dom->createElement('DELIVERY');

				$child = $dom->createElement('DELIVERY_ID', (string) $spedName);
				$delivery->appendChild($child);
				$child = $dom->createElement('DELIVERY_PRICE', (string) $spedition['price']);
				$delivery->appendChild($child);

				if ($spedition['cod']) {
					$child = $dom->createElement('DELIVERY_PRICE_COD', (string) $spedition['cod']);
					$delivery->appendChild($child);
				}

				$shopitem->appendChild($delivery);
			}
		}

		if ($product->getGallery()) {
			$coverId = null;
			if ($product->getGallery()->getCover()) {
				$coverId = $product->getGallery()->getCover()->id;
				$child   = $dom->createElement('IMGURL', (string) ($this->baseUrl . $product->getGallery()->getCover()->getFilePath()));
				$shopitem->appendChild($child);
			}

			foreach ($product->getGallery()->getImages() as $img) {
				if ($img->id == $coverId)
					continue;

				$child = $dom->createElement('IMGURL_ALTERNATIVE', (string) ($this->baseUrl . $img->getFilePath()));
				$shopitem->appendChild($child);
			}
		}

		if ($product->getEan()) {
			$child = $dom->createElement('EAN', (string) $product->getEan());
			$shopitem->appendChild($child);
		}

		if ($export['categoryText']) {
			$child = $dom->createElement('CATEGORYTEXT', (string) $export['categoryText']);
			$shopitem->appendChild($child);
		}

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('MANUFACTURER', (string) $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		foreach ($product->getFeatures() as $feature) {
			$param = $dom->createElement('PARAM');

			if (!$this->canExportFeature((int) $feature->idFeature))
				continue;

			$child = $dom->createElement('PARAM_NAME', $feature->name);
			$param->appendChild($child);
			$child = $dom->createElement('VAL', $feature->value);
			$param->appendChild($child);

			$shopitem->appendChild($param);
		}

		if ($export['bidCpc']) {
			$child = $dom->createElement('HEUREKA_CPC', $export['bidCpc']);
			$shopitem->appendChild($child);
		}

		foreach ($product->getGifts() as $gift) {
			$child = $dom->createElement('GIFT', Strings::truncate($gift->getName(), 250, ''));
			$child->setAttribute('ID', (string) $gift->getProduct()->getId());
			$shopitem->appendChild($child);
		}

		$dom->appendChild($shopitem);

		$this->writeToFile($dom->saveXML($dom->documentElement) . PHP_EOL);

		// Stock
		if ($product->getQuantity() > 0) {
			$dom               = new \DOMDocument();
			$dom->formatOutput = true;
			$dom->encoding     = 'utf-8';

			$shopitem = $dom->createElement('item');
			$shopitem->setAttribute('id', (string) $product->getId());

			$child = $dom->createElement('stock_quantity', (string) $product->getQuantity());
			$shopitem->appendChild($child);

			$dom->appendChild($shopitem);

			$this->writeToFileStock($dom->saveXML($dom->documentElement) . PHP_EOL);
		}
	}

	public function endExport()
	{
		$this->writeToFile('</SHOP>');

		parent::endExport();

		$this->endExportStock();
	}

	public function endExportStock()
	{
		$this->writeToFileStock('</item_list>');

		parent::endExportStock();
	}
}
