<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use EshopCatalog\Model\Entities\ProductVariant;
use Nette\Application\LinkGenerator;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use EshopCatalog\Model\Entities\ProductInSite;
use Core\Model\Event\EventDispatcher;
use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Customer;
use Gallery\FrontModule\Model\Albums;
use Nette\Caching\Cache;
use Users\Model\Http\UserStorage;

/**
 * TODO možnost přesunutí do fasády
 *
 * Class Products
 * @package EshopCatalog\FrontModule\Model
 */
class Products extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'eshopCatalogProducts';

	const MODE_ESHOP    = 'eshop';
	const MODE_CHECKOUT = 'checkout';

	protected static string $mode = self::MODE_ESHOP;

	protected $entityClass = Product::class;

	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var Albums @inject */
	public $albumsService;

	/** @var Manufacturers @inject */
	public $manufacturersService;

	/** @var Sites @inject */
	public $sitesService;

	/** @var FeatureProducts */
	protected $featureProductsService;

	/** @var CacheService */
	protected $cacheService;

	/** @var UserStorage */
	protected $userStorage;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var Config */
	protected $config;

	/** @var AvailabilityService */
	protected $availabilityService;

	/** @var callable[] */
	public $customQB = [];

	/** @var array */
	protected $cProductsIdInCategory = [], $cSearch, $cProductsIdBySearch = [], $cProductsIdAll = [], $cGroupCustomer = [],
		$cPrices = [], $cExtraFields, $cQuantity = [];

	/** @var callable[] */
	public $onAfterProductsGet = [];

	/** @var array */
	public $cacheDep = [
		Cache::TAGS   => ['products'],
		Cache::EXPIRE => '15 minutes',
	];

	public function __construct(FeatureProducts $featurePorducts, CacheService $cacheService,
	                            UserStorage $userStorage, Config $config, EventDispatcher $eventDispatcher, AvailabilityService $availabilityService)
	{
		$this->featureProductsService = $featurePorducts;
		$this->cacheService           = $cacheService;
		$this->userStorage            = $userStorage;
		$this->config                 = $config;
		$this->eventDispatcher        = $eventDispatcher;
		$this->availabilityService    = $availabilityService;

		$this->afterConstruct();
		$this->onAfterProductsGet[] = function($products) {
			/** @var Dao\Product[] $products */
			$this->loadPriceLevels($products,
				$this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);
		};
	}

	protected function afterConstruct() { }

	/**
	 * @param int|array $id
	 *
	 * @return array|mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getForNavigation($id)
	{
		$qb = $this->getEr()->createQueryBuilder('p')->select('p.id, pt.name')
			->andWhere(is_array($id) ? 'p.id IN (:id)' : 'p.id = :id')->setParameter('id', $id)
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->setParameter('lang', $this->defaultLang->locale)
			->getQuery();

		return is_array($id) ? $qb->getArrayResult() : $qb->getOneOrNullResult(Query::HYDRATE_ARRAY);
	}

	/**
	 * @param int $categoryId
	 *
	 * @return int[]
	 */
	public function getProductsIdInCategory($categoryId, $start = 0, $limit = null, $sort = null, $filter = [])
	{
		$lang    = $this->translator->getLocale();
		$site    = $this->sitesService->getCurrentSite();
		$keyBase = $site->getIdent() . '/' . md5(serialize([$categoryId, $sort, $filter])) . '/' . $lang;

		if (isset($this->cProductsIdInCategory[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);

			return $this->cProductsIdInCategory[$keyBase];
		} else {
			//			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($categoryId, $sort, $filter, $lang, $site) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'listing/' . $categoryId;
			$dep[Cache::TAGS][] = 'listing';
			$dep[Cache::EXPIRE] = '5 minutes';

			if (isset($filter['manu']))
				$dep[Cache::TAGS][] = 'manufacturers';
			if (isset($filter['features']))
				$dep[Cache::TAGS][] = 'features';

			$query = (new ProductQuery($lang))
				->withTexts()
				->selectIds();

			if (self::$mode == self::MODE_ESHOP)
				$query->inSite($site->getIdent());
			$query->inCategory($categoryId);

			if (Config::load('onlyStockProductsInList', true) == true)
				$query->onlyInStockOrSupplier();

			if ($filter)
				$query->useFilter($this->prepareFilter($filter));
			$this->prepareSort($query, $sort);

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			$ids           = [];
			$groupVariants = Config::load('groupVariantsInList', false);
			if (self::$mode == self::MODE_CHECKOUT)
				$groupVariants = false;

			if (!$groupVariants) {
				$rawIds     = [];
				$variantIds = [];
				$prepared   = [];
				$qb->leftJoin('p.isVariant', 'pv')
					->addSelect('pv.variantId');
				foreach ($qb->getQuery()->getArrayResult() as $v) {
					$rawIds[] = [
						'variantId' => $v['variantId'],
						'id'        => $v['id'],
					];
					if ($v['variantId'])
						$variantIds[] = $v['variantId'];
				}

				if (!empty($variantIds)) {
					foreach ($this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
						         ->select('IDENTITY(pv.product) as product, pv.variantId')
						         ->where('pv.variantId IN (:ids)')->setParameter('ids', array_unique($variantIds))
						         ->getQuery()->getArrayResult() as $row)
						$prepared[$row['variantId']][] = $row['product'];

					foreach ($rawIds as $v) {
						if (isset($prepared[$v['variantId']]))
							foreach ($prepared[$v['variantId']] as $v2)
								$ids[] = $v2;
						else
							$ids[] = $v['id'];
					}
				} else {
					foreach ($rawIds as $v)
						$ids[] = $v['id'];
				}
			} else {
				foreach ($qb->getQuery()->getArrayResult() as $v)
					$ids[] = $v['id'];
			}

			if (is_null($sort)) {
				$this->cProductsIdInCategory[$keyBase] = array_unique(array_merge($this->getRecommendedIds($categoryId, 0, null, null, $filter), $ids));
			} else
				$this->cProductsIdInCategory[$keyBase] = $ids;
		}

		if ($limit)
			return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);

		return $this->cProductsIdInCategory[$keyBase];
	}

	/**
	 * @param int        $categoryId
	 * @param array|null $filter
	 *
	 * @return int
	 */
	public function getProductsInCategoryCount($categoryId, $filter = [])
	{
		return count($this->getProductsIdInCategory($categoryId, 0, null, null, $filter));
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return int[]
	 */
	public function getProductsIdBySearch($q, $start = 0, $limit = null, $sort = null, $filter = [])
	{
		$keyBase = md5(serialize([$q, $sort, $filter]));

		if (isset($this->cProductsIdBySearch[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);

			return $this->cProductsIdBySearch[$keyBase];
		} else {
			$query = (new ProductQuery($this->defaultLang->locale))
				->search($q)
				->selectIds()
				->availableInSearch()
				->selectInStockBool();

			$emptySort = is_null($sort);
			if (is_string($sort)) {
				$tmp  = explode(' ', $sort, 2);
				$sort = ['inStockBool' => 'DESC'] + [$tmp[0] => ($tmp[1] ?? 'ASC')];
			} else if (is_array($sort))
				$sort = ['inStockBool' => 'DESC'] + $sort;
			else
				$sort = ['inStockBool' => 'DESC', 'p.price' => 'ASC'];

			if ($filter)
				$query->useFilter($this->prepareFilter($filter));
			$this->prepareSort($query, $sort);

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			$ids = [];
			foreach ($qb->getQuery()->getScalarResult() as $v)
				$ids[] = $v['id'];

			if (is_null($sort)) {
				$this->cProductsIdBySearch[$keyBase] = array_unique(array_merge($this->getRecommendedIds($q, 0, null, null, $filter, 'search'), $ids));
			} else
				$this->cProductsIdBySearch[$keyBase] = $ids;
		}

		if ($limit)
			return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);

		return $this->cProductsIdBySearch[$keyBase];
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return int[]
	 */
	public function getProductsIdAll($start = 0, $limit = null, $sort = null, $filter = [])
	{
		$keyBase = md5(serialize([$sort, $filter]));

		if (isset($this->cProductsIdAll[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);

			return $this->cProductsIdAll[$keyBase];
		} else {
			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($sort, $filter) {
				$dep = $this->cacheDep;

				if (isset($filter['manu']))
					$dep[Cache::TAGS][] = 'manufacturers';
				if (isset($filter['features']))
					$dep[Cache::TAGS][] = 'features';

				$query = (new ProductQuery($this->defaultLang->locale))
					->selectIds();

				if ($filter)
					$query->useFilter($this->prepareFilter($filter));
				$this->prepareSort($query, $sort);

				$qb = $query->getQueryBuilder($this->getEr());
				foreach ($this->customQB as $cqb)
					$cqb($qb);

				$ids = [];
				foreach ($qb->getQuery()->getScalarResult() as $v)
					$ids[] = $v['id'];

				return $ids;
			});

			if (is_null($sort)) {
				$this->cProductsIdAll[$keyBase] = array_unique(array_merge($this->getRecommendedIds(null, 0, null, null, $filter), $ids));
			} else
				$this->cProductsIdAll[$keyBase] = $ids;;
		}

		if ($limit)
			return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);

		return $this->cProductsIdAll[$keyBase];
	}

	/**
	 * @param int   $start
	 * @param null  $limit
	 * @param null  $sort
	 * @param array $filter
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getProductsAll($start = 0, $limit = null, $sort = null, $filter = [])
	{
		$products = [];
		foreach ($this->getProductsIdAll($start, $limit, $sort, $filter) as $id) {
			$product       = $this->get($id, ['loadPrice' => false]);
			$products[$id] = $product;
		}

		$this->loadFresh($products, $this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);

		foreach ($this->onAfterProductsGet as $c)
			$c($products);

		foreach ($products as $k => $product)
			if (!$product->isActive)
				unset($products[$k]);

		return $products;
	}

	/**
	 * @param      $ids
	 * @param int  $start
	 * @param null $limit
	 * @param null $sort
	 * @param null $filter
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getProductsByIds($ids, $start = 0, $limit = null, $sort = null, $filter = null)
	{
		$products = [];

		$ids = array_slice($ids, $start, $limit);

		foreach ($ids as $id) {
			$product = $this->get($id, ['loadPrice' => false]);
			if ($product)
				$products[$id] = $product;
		}

		foreach ($this->onAfterProductsGet as $c)
			$c($products);

		return $products;
	}

	public function getRecommendedIds($q = null, $start = 0, $limit = null, $sort = null, $filter = [], $type = 'category')
	{
		$site    = $this->sitesService->getCurrentSite();
		$keyBase = $site->getIdent() . '-' . md5('recommended_' . serialize([$q, $sort, $filter, $type]));
		$key     = "$keyBase-$start-$limit";

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($q, $start, $limit, $sort, $filter, $type, $site) {
			$dep                = $this->cacheDep;
			$dep[Cache::EXPIRE] = '5 minutes';

			if (isset($filter['manu']))
				$dep[Cache::TAGS][] = 'manufacturers';
			if (isset($filter['features']))
				$dep[Cache::TAGS][] = 'features';

			$query = (new ProductQuery($this->defaultLang->locale))
				->withTag('tip')
				->inSite($site->getIdent())
				->selectIds();

			if (Config::load('onlyStockProductsInList', true) == true)
				$query->onlyInStockOrSupplier();

			switch ($type) {
				case 'category':
					$dep[Cache::TAGS][] = 'listing/' . $q;
					if ($q)
						$query->inCategory($q);
					break;
				case 'search':
					$dep[Cache::TAGS][] = 'listing/search';
					$query->search($q);
					break;
			}

			if ($filter)
				$query->useFilter($this->prepareFilter($filter));
			$sort = 'p.price ASC';
			$this->prepareSort($query, $sort);

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			$ids = [];
			foreach ($qb->getQuery()->setMaxResults($limit)->setFirstResult($start)->getScalarResult() as $v)
				$ids[] = $v['id'];

			return $ids;
		});
	}

	/**
	 * @param       $id
	 * @param array $params
	 *
	 * @return Dao\Product|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 *
	 * @deprecated Použijte getProducts nebo getProduct z productsFacade
	 *
	 */
	public function get($id, $params = [])
	{
		$locale = $params['lang'] ?? $this->translator->getLocale();
		$key    = 'product_' . $locale . '_' . $id;

		/** @var Dao\Product $product */
		$product = $this->cacheService->productCache->load($key, function(&$dep) use ($id, $locale) {
			$dep                 = $this->cacheDep;
			$dep[Cache::TAGS][]  = 'product/' . $id;
			$dep[Cache::EXPIRE]  = '1 month';
			$dep[Cache::SLIDING] = true;

			$query = (new ProductQuery($locale))
				->withCategories()
				->withTexts()
				->withVatRate()
				->withVariantParent()
				->byId($id);

			$qb = $query->getQueryBuilder($this->getEr());
			$qb->groupBy('p.id');

			foreach ($this->customQB as $cqb)
				$cqb($qb);

			$result = $qb->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

			if ($result) {
				$result                = $this->normalizeHydrateArray($result);
				$result['extraFields'] = $this->getAllExtraFields()[$result['id']] ?? [];
				$dao                   = $this->fillDao($result);

				if ($this->linkGenerator)
					$dao->link = $this->linkGenerator->link('EshopCatalog:Front:Default:product', [
						'id'     => $dao->getId(),
						'locale' => $locale,
					]);

				return $dao;
			}

			return null;
		});

		if ($product) {
			if ($product->galleryId)
				$product->setGallery($this->albumsService->get($product->galleryId));

			if ($product->manufacturerId)
				$product->setManufacturer($this->manufacturersService->get((int) $product->manufacturerId));

			$products = [$product->getId() => &$product];
			$this->loadFresh($products, $this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);

			$this->eventDispatcher->dispatch(new ProductsEvent($product), Products::class . '::afterFillDao');

			return $product->isActive ? $product : null;
		}

		return null;
	}

	/**
	 * @param Dao\Product[] $products
	 * @param null          $userId
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadFresh(&$products, $userId = null): void
	{
		$ids = array_map(function($p) { return $p->getId(); }, $products);
		foreach ($this->getEr()->createQueryBuilder('p')
			         ->select('p.id, p.price, p.retailPrice, IDENTITY(p.availability) as availability')
			         ->andWhere('p.id IN (:ids)')->setParameter('ids', $ids)
			         ->getQuery()->getArrayResult() as $row) {
			$products[$row['id']]->setPrice((float) $row['price']);
			$products[$row['id']]->setBasePrice((float) $row['price']);
			$products[$row['id']]->setRetailPrice((float) $row['retailPrice']);
			$products[$row['id']]->priceInBaseCurrency       = (float) $row['price'];
			$products[$row['id']]->basePriceInBaseCurrency   = (float) $row['price'];
			$products[$row['id']]->retailPriceInBaseCurrency = (float) $row['retailPrice'];
			if ($products[$row['id']]->unlimitedQuantity)
				$products[$row['id']]->setAvailability($this->availabilityService->getByIdent('inStock'));
			else if ($row['availability'])
				$products[$row['id']]->setAvailability($this->availabilityService->get((int) $row['availability']));
		}

		$this->loadPriceLevels($products, $userId);
		$this->loadQuantity($products);
		$this->loadSite($products);

		$this->eventDispatcher->dispatch(new ProductsEvent($products), Products::class . '::afterLoadFresh');
	}

	/**
	 * @param Dao\Product[] $products
	 */
	public function loadSite(&$products)
	{
		$ids      = array_map(function($product) { return $product->getId(); }, $products);
		$variants = [];

		$qb = $this->em->getRepository(ProductInSite::class)->createQueryBuilder('ps')
			->select('IDENTITY(ps.product) as prod, IDENTITY(ps.site) as site, IDENTITY(ps.category) as cat, ps.isActive')
			->where('ps.product IN (:ids)')
			->andWhere('ps.isActive = 1')
			->setParameters([
				'ids' => $ids,
			]);

		if (self::$mode == self::MODE_ESHOP)
			$qb->andWhere('ps.site = :site')
				->setParameter('site', $this->sitesService->getCurrentSite()->getIdent());

		foreach ($qb->getQuery()->getScalarResult() as $row) {
			$products[$row['prod']]->defaultCategoryId = (int) $row['cat'];
			$products[$row['prod']]->isActive          = (bool) $row['isActive'];

			if ($products[$row['prod']]->variantOf)
				$variants[$products[$row['prod']]->variantOf][] = $row['prod'];
		}

		// Projduti variant
		foreach ($variants as $k => $vals) {
			if (!isset($products[$k]))
				continue;

			foreach ($vals as $v) {
				$products[$v]->defaultCategoryId = $products[$k]->defaultCategoryId;
				$products[$v]->isActive          = $products[$k]->isActive;
			}
		}
	}

	/**
	 * @param Dao\Product[] $products
	 * @param int|null      $userId
	 */
	public function loadPriceLevels(&$products, $userId = null): void
	{
		if (!$userId || !Config::load('enablePriceLevels', false))
			return;

		$ids    = [];
		$prices = [];
		foreach ($products as $product) {
			$ids[$product->getId()]    = $product->getId();
			$prices[$product->getId()] = $product->getPrice();
		}

		// Pokud je uživatel ve skupině a je u produktu nastavena cenová hladina, aplikujeme ji
		if (!isset($this->cGroupCustomer[$userId]))
			$this->cGroupCustomer[$userId] = $this->em->getRepository(Customer::class)->createQueryBuilder('c', 'c.id')
					->select('cg.id, cg.productsSale')
					->innerJoin('c.groupCustomers', 'cg')
					->where('c.user = :user')
					->setParameter('user', $userId)
					->setMaxResults(1)
					->getQuery()->getArrayResult()[0] ?? [];

		$group = $this->cGroupCustomer[$userId] ?? null;
		if ($group['id']) {
			foreach ($this->em->getRepository(ProductPriceLevel::class)->createQueryBuilder('ppl')
				         ->select('IDENTITY(ppl.productId) as product, ppl.price')
				         ->where(count($ids) == 1 ? 'ppl.productId = :ids' : 'ppl.productId IN (:ids)')
				         ->andWhere('ppl.groupId = :groupId')
				         ->setParameters([
					         'ids'     => array_values($ids),
					         'groupId' => $group['id'],
				         ])->getQuery()->getArrayResult() as $row) {
				$prices[$row['product']] = $row['price'];
				unset($ids[$row['product']]);
			}
		}

		// Pokud je ve skupině nastavená sleva na všechny produkty, aplikuejeme ji na produkty, které ještě nemají upravenou cenu
		if ($group['productsSale'] > 0) {
			$tmp = 1 - ((float) $group['productsSale'] / 100);

			foreach ($ids as $id) {
				$prices[$id] *= $tmp;
				unset($ids[$id]);
			}
		}

		// Na neupravené produkty aplikujeme slevu hromadnou
		$registerUserSale = $this->settings->get('eshopCatalogRegisterUserSale', 0);
		if ($registerUserSale > 0) {
			$tmp = 1 - ((float) $registerUserSale / 100);

			foreach ($ids as $id) {
				$prices[$id] *= $tmp;
				unset($ids[$id]);
			}
		}

		// Aplikujeme slevu na produkty
		foreach ($prices as $product => $price) {
			$products[$product]->price               = $price;
			$products[$product]->priceInBaseCurrency = $price;
		}
	}

	/**
	 * @param Dao\Product[] $product
	 */
	public function loadFeatures(&$products)
	{
		if ($products instanceof Dao\Product)
			$products->setFeatures($this->featureProductsService->getFeaturesForProduct([$products->getId()])[$products->getId()] ?? []);
		else {
			$ids = [];
			foreach ($products as $p)
				$ids[] = $p->getId();
			foreach ($this->featureProductsService->getFeaturesForProduct($ids) as $k => $v)
				$products[$k]->setFeatures($v);
		}
	}

	/**
	 * @param Dao\Product|Dao\Product[] $product
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadQuantity(&$product)
	{
		if (is_array($product)) {
			foreach ($product as $p) {
				if (isset($this->cQuantity[$p->getId()]))
					$product[$p->getId()]->setQuantity($this->cQuantity[$p->getId()]);
				else
					$id[] = $p->getId();
			}
			$id = array_map(function($row) { return $row->getId(); }, $product);
		} else {
			if (isset($this->cQuantity[$product->getId()]))
				$product->setQuantity($this->cQuantity[$product->getId()]);
			else
				$id = $product->getId();
		}

		if (!isset($id))
			return;

		$query = (new ProductQuery($this->defaultLang->locale))
			->byId($id)
			->selectQuantity();

		$data = $query->getQueryBuilder($this->getEr())->groupBy('p.id')->getQuery()->getArrayResult();

		if (is_array($product)) {
			foreach ($data as $id => $row) {
				$q                    = intval($row['productQ']) + intval($row['suppliersQ']);
				$this->cQuantity[$id] = $q;
				$product[$id]->setQuantity($q);
			}
		} else if (isset($data[$id])) {
			$q                    = intval($data[$id]['productQ']) + intval($data[$id]['suppliersQ']);
			$this->cQuantity[$id] = $q;
			$product->setQuantity($q);
		}
	}

	/**
	 * @param array $product
	 *
	 * @return Dao\Product
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function fillDao($product): Dao\Product
	{
		$p = new Dao\Product($product['id']);
		$p->setQuantity((int) $product['quantity']);

		$p->setPrice((float) $product['price']);
		$p->setBasePrice((float) $product['price']);
		$p->setRetailPrice((float) $product['retailPrice']);

		$p->priceInBaseCurrency       = $p->getPrice();
		$p->basePriceInBaseCurrency   = $p->basePrice;
		$p->retailPriceInBaseCurrency = $p->retailPrice;

		$p->setName($product['productTexts']['name']);
		$p->setName2($product['productTexts']['name2']);
		$p->setDescription($product['productTexts']['description']);
		$p->setSeo($product['productTexts']['seo']);
		$p->setVatRate($product['vatRate']);
		$p->setEan($product['ean']);
		$p->setCode1($product['code1']);
		$p->setCode2($product['code2']);
		$p->setExtraFields($product['extraFields']);
		$p->unlimitedQuantity = $product['unlimitedQuantity'] ? 1 : 0;
		$p->shortDescription  = $product['productTexts']['shortDescription'];
		$p->created           = $product['created'];
		$p->modified          = $product['modified'];
		$p->categories        = $product['categories'];
		$p->variantId         = $product['variantId'];
		$p->helperData        = [
			'variantImage' => $product['variantImage'],
		];

		$p->defaultCategoryId    = (int) $product['defaultCategory'];
		$p->canAddToCart         = $product['canProductsAddToCart'] ? true : false;
		$p->discountDisabled     = $product['discountDisabled'] ? true : false;
		$p->categoryGiftsAllowed = $product['categoryGiftsAllowed'] ? true : false;

		if ($product['gallery'])
			$p->galleryId = $product['gallery'];

		if ($product['manufacturer'])
			$p->manufacturerId = $product['manufacturer'];

		return $p;
	}

	/**
	 * @param array $product
	 *
	 * @return array
	 */
	public function normalizeHydrateArray($product)
	{
		$result                 = $product[0];
		$result['variantId']    = (int) $product['variantId'];
		$result['variantImage'] = $product['variantImage'] ? (int) $product['variantImage'] : null;
		$result['productTexts'] = $product[0]['productTexts'][$this->defaultLang->locale];
		foreach (['gallery', 'manufacturer', 'vatRate', 'defaultCategory', 'availability'] as $v)
			$result[$v] = $product[$v];

		$result['categories'] = array_map(function($r) { return $r['id_category']; }, $product[0]['categoryProducts']);
		unset($result['categoryProducts']);

		return $result;
	}

	protected function prepareFilter($filter)
	{
		if (isset($filter['features'])) {
			$count = count($filter['features']);
			$ids   = [];
			$qb    = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')->select('IDENTITY(fp.product) as product');
			$tmp   = [];

			foreach ($filter['features'] as $group => $vals) {
				$or  = [];
				$qb2 = clone $qb;
				foreach ($vals as $k => $v) {
					$or[] = "IDENTITY(fp.featureValue) = :fpfv_{$i}_{$k}";
					$qb2->setParameter("fpfv_{$i}_{$k}", $v);
				}
				$qb2->andWhere(implode(' OR ', $or));
				foreach ($qb2->getQuery()->useResultCache(true, 60)->getArrayResult() as $row)
					$tmp[$row['product']] = isset($tmp[$row['product']]) ? $tmp[$row['product']] + 1 : 1;
			}

			foreach ($tmp as $k => $v) {
				if ($v == $count)
					$ids[] = $k;
			}
			$tmp = null;

			$filter['features'] = empty($ids) ? 'empty' : $ids;
		}

		return $filter;
	}

	public function prepareSort(ProductQuery &$query, $sort)
	{
		if ($sort == 'recommended' || is_null($sort)) {
			$sort = 'p.price ASC';
		}

		if (is_string($sort)) {
			$sort = explode(' ', $sort, 2);
			$query->sortBy($sort[0], $sort[1] ?? null);
		} else if (is_array($sort)) {
			foreach ($sort as $k => $v)
				$query->addSortBy($k, $v);
		}
	}

	public function getAllExtraFields(): array
	{
		if ($this->cExtraFields === null) {
			$this->cExtraFields = [];
			foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
				         ->select('ef.sectionKey, ef.key, ef.value')
				         ->where('ef.sectionName = :secName')
				         ->setParameter('secName', Product::EXTRA_FIELD_SECTION)->getQuery()->getScalarResult() as $row)
				$this->cExtraFields[$row['sectionKey']][$row['key']] = $row['value'];
		}

		return $this->cExtraFields;
	}

	public static function setMode(string $mode): void { self::$mode = $mode; }
}
