<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Categories;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\SmartObject;

class CategoryListener implements EventSubscriberInterface
{
	use SmartObject;

	/** @var CacheService */
	protected $cacheService;

	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PostUpdate
	 *
	 * @param Category           $category
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\ORM\ORMException
	 */
	public function postUpdateHandler(Category $category, LifecycleEventArgs $event)
	{
		bdump("category '{$category->getId()}' post update");
		$em        = $event->getEntityManager();
		$changeSet = $em->getUnitOfWork()->getEntityChangeSet($category);

		if (isset($changeSet['isPublished'])) {
			$this->cacheService->clean('default', [
				Cache::TAGS => ['listing/' . $category->getId(), 'listing/search'],
			]);
		}

		$this->cacheService->clean('category', [Cache::TAGS => ['categories']]);
	}

	/**
	 * @ORM\PostRemove
	 *
	 * @param Category           $category
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\ORM\ORMException
	 */
	public function postRemoveHandler(Category $category, LifecycleEventArgs $event)
	{
		bdump("category '{$category->getId()}' remove");

		$tags1 = [
			'listing/' . $category->getId(),
			'listing/search',
		];

		$tags2 = [
			'category/' . $category->getId(),
			Categories::CACHE_NAMESPACE,
		];

		if ($category->getParent()) {
			$tags1[] = 'listing/' . $category->getParent()->getId();
			$tags2[] = 'category/' . $category->getParent()->getId();
		}

		$this->cacheService->clean('default', [
			Cache::TAGS => $tags1,
		]);
		$this->cacheService->clean('category', [Cache::TAGS => $tags2]);
	}
}
