<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Nette\Caching\Cache;
use Nette\SmartObject;

class CategoryProductListener implements EventSubscriberInterface
{
	use SmartObject;

	/** @var array */
	protected static $cleared = [];

	/** @var CacheService */
	protected $cacheService;

	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PostUpdate
	 * @ORM\PostRemove
	 *
	 * @param CategoryProduct    $categoryProduct
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\ORM\ORMException
	 */
	public function postHandler(CategoryProduct $categoryProduct, LifecycleEventArgs $event)
	{
		if (!self::$cleared[$categoryProduct->getIdCategory()]) {
			bdump("category product '{$categoryProduct->getIdCategory()}' post update / remove");
			$this->cacheService->clean('default', [
				Cache::TAGS => ['listing/' . $categoryProduct->getIdCategory(), 'listing/search'],
			]);
		}

		self::$cleared[$categoryProduct->getIdCategory()] = true;
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param CategoryProduct    $categoryProduct
	 * @param LifecycleEventArgs $args
	 */
	public function onPreFlush($categoryProduct, $args)
	{
		foreach ($args->getEntityManager()->getUnitOfWork()->getScheduledEntityInsertions() as $entity) {
			if ($entity instanceof CategoryProduct && !self::$cleared[$entity->getIdCategory()]) {
				bdump("category product {$entity->getIdCategory()} pre flush");
				$this->cacheService->clean('default', [
					Cache::TAGS => ['listing/' . $entity->getIdCategory(), 'listing/search'],
				]);

				self::$cleared[$entity->getIdCategory()] = true;
			}
		}
	}
}
