<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Navigation;

use Nette\Application\LinkGenerator;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\Products;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Pages\AdminModule\Model\TemplatePages;

class Home extends BaseNavigation implements INavigationItem
{
	protected $title = 'home';

	protected $presenter     = 'EshopCatalog:Front:Default';
	protected $action        = 'default';
	protected $actionProduct = 'product';

	protected $componentChildStart = 0;
	protected $componentChildLimit = 1000;
	protected $cComponentChildGalleries;

	const CACHE_PRODUCT = 'eshopCatalogProduct';

	/** @var Products */
	protected $productsService;

	/** @var TemplatePages */
	protected $templatePagesService;

	/** @var LinkGenerator */
	public $linkGenerator;

	public function __construct(Products $products, TemplatePages $templatePages)
	{
		$this->productsService      = $products;
		$this->templatePagesService = $templatePages;
	}

	/**
	 * @param BaseForm $baseForm
	 *
	 * @return BaseContainer
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer();

		$site      = $this->navFormSite ?: $baseForm->getCustomData('loadComponent')['site'];
		$pages     = $this->templatePagesService->getBySiteForSelectOption($site ?? $baseForm['site']->getValue());

		$container->addSelect('templateHome', 'eshopCatalog.navigationBuilder.templateHome', $pages);
		$container->addSelect('templateProduct', 'eshopCatalog.navigationBuilder.templateDetail', $pages);

		if ($baseForm->getParent()->lookup(null, false)) {
			$componentId = $baseForm->getParent()->getParameter('componentId');
			if ($componentId)
				$container['templateHome']->setAttribute('data-custom-link', $baseForm->getParent()->link('componentCustom!', [$componentId]));
		} else {
			$baseForm->onAnchor[] = function () use ($container, $baseForm) {
				$container['templateHome']->setAttribute('data-custom-link', $baseForm->getParent()->link('componentCustom!', [$baseForm->getValues()->componentType]));
			};
		}

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{

	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		$cache[Cache::TAGS][] = self::CACHE_PRODUCT . '/' . $urlParams['id'];

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		return $this->presenter == $presenter && in_array($action, [$this->action, $this->actionProduct]);
	}

	/**
	 * @param Navigation $navigation
	 * @param array      $urlParams
	 *
	 * @return array|mixed
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$returnDefault = [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => null,
		];

		if (isset($urlParams['getHomepage']))
			return $returnDefault;

		if (isset($urlParams['path']) && $navigation->alias != $urlParams['path']) {
			$params    = explode('/', $urlParams['path']);
			$lastParam = end($params);
			$id        = explode('-', $lastParam, 2);

			if (is_numeric($id[0])) {
				return [
					'presenter' => $this->presenter,
					'action'    => $this->actionProduct,
					'id'        => $id[0],
				];
			}
		}

		return [];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		if ($urlParams['action'] == $this->actionProduct && isset($urlParams['id'])) {
			$product = $this->productsService->getForNavigation($urlParams['id']);

			if ($product)
				return ($navigation->isHomepage ? '' : '/' . $navigation->alias) . '/' . $product['id'] . '-' . Strings::webalize($product['name']);
		}

		return $navigation->isHomepage ? '/' : '/' . $navigation->alias;
	}

	/**
	 * @param Navigation|null $navigation
	 * @param bool            $siteMapData
	 *
	 * @return array
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];
		$i   = 0;

		//		if ($this->cComponentChildGalleries === null) {
		//			if ($siteMapData) {
		//				foreach ($this->productsService->em->getRepository(Image::class)->createQueryBuilder('i')
		//					         ->select('i.path, i.filename, it.title, p.id as productId')
		//					         ->leftJoin('i.texts', 'it', 'WITH', 'it.lang = :lang')
		//					         ->join('i.album', 'a')->leftJoin(Product::class, 'p', 'WITH', 'p.gallery = a.id')
		//					         ->setParameter('lang', $lang)
		//					         ->getQuery()->getScalarResult() as $row) {
		//					$this->cComponentChildGalleries[$row['productId']][] = $row;
		//				}
		//
		//				$this->productsService->em->clear(Image::class);
		//			}
		//		}
		//		$galleries = $this->cComponentChildGalleries;

		foreach ((new ProductQuery($lang))
			         ->withTexts()
			         ->getQueryBuilder($this->productsService->getEr())
			         ->select('p.id, pt.name, p.modified, pt.seo')
			         ->join('p.idCategoryDefault', 'defaultCategory', 'WITH', 'defaultCategory.isPublished = 1')
			         ->where('p.isPublished = 1')
			         ->setFirstResult($this->componentChildStart)->setMaxResults($this->componentChildLimit)
			         ->getQuery()->getScalarResult() as $row) {
			$id = $row['id'];

			$data = [
				'action'    => $this->actionProduct,
				'id'        => $id,
				'presenter' => $this->presenter,
				'locale'    => $lang,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($row['modified'])
					$siteMapUrl->setModified($row['modified']);

				$seo = $row['seo'] ? @unserialize($row['seo']) : null;
				if (is_array($seo)) {
					if ($seo['addToSiteMap'] === '0' || is_string($seo['robots']) && strpos('noindex', $seo['robots']) !== false)
						continue;

					$siteMapUrl->setSeo($seo, null, true);
				}

				//				if (isset($galleries[$id])) {
				//					foreach ($galleries[$id] as $k => $rowImg) {
				//						$img           = new SiteMapImage();
				//						$img->title    = $rowImg['title'];
				//						$img->filePath = $rowImg['path'] . '/' . $rowImg['filename'];
				//						$siteMapUrl->addImage($img);
				//					}
				//				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			$arr[] = $data;
			$i++;
		}

		$this->productsService->em->clear();

		$this->componentChildStart += $this->componentChildLimit;

		if (!empty($arr))
			$arr['continue'] = true;

		return $arr;
	}
}
