<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Tags;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\Tag;
use EshopCatalog\Model\Entities\TagText;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\AdminModule\Model\Tags;

class TagForm extends BaseControl
{
	/** @var Tag */
	public $tag;

	/** @var \EshopCatalog\AdminModule\Model\Tags */
	protected $tagServices;

	public function __construct(Tags $tags)
	{
		$this->tagServices = $tags;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('type', 'eshopCatalog.tagForm.type')->setMaxLength(255);
		$form->addText('name', 'eshopCatalog.tagForm.name')
			->setIsMultilanguage()->setMaxLength(255);
		$form->addFilesManager('image', 'default.image');

		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$nameFilled = false;
		foreach ($values->name as $v)
			if ($v)
				$nameFilled = true;

		if (!$nameFilled)
			$form->addError('eshopCatalog.tagForm.nameMissing');
		if ($form->hasErrors())
			$this->redrawControl('form');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TagText[] $tagTexts */
			$tagTexts = [];

			if ($this->tag) {
				$tag      = $this->tag;
				$tagTexts = $tag->getTexts()->toArray();
			} else {
				$tag = new Tag();
			}

			$tag->type  = $values->type;
			$tag->image = $values->image;

			foreach ($langValues as $l => $v) {
				if ($v['name'] === '')
					continue;

				if (!$tagTexts[$l])
					$tagTexts[$l] = new TagText($tag, $l);

				$tagTexts[$l]->name = $v['name'];
				$this->em->persist($tagTexts[$l]);
			}
			$tag->setTexts($tagTexts);

			$this->em->persist($tag);
			$this->em->flush();
			$form->addCustomData('tagType', $tag->type);
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.tagForm.tagSaved', Flashes::FLASH_SUCCESS);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setTag($type)
	{
		$this->tag = $this->tagServices->getByType($type);

		if ($this->tag) {
			$form     = $this['form'];
			$defaults = [
				'type'  => $this->tag->type,
				'image' => $this->tag->image,
			];

			foreach ($this->langsService->getLangs(false) as $lang) {
				foreach (['name'] as $name) {
					$defaults[$name][$lang->getTag()] = $this->tag->getText($lang->getTag())->$name;
				}
			}

			$form->setDefaults($defaults);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
