<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Variants;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\VariantValue;
use EshopCatalog\Model\Entities\VariantValueText;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\AdminModule\Model\Variants;
use EshopCatalog\AdminModule\Model\VariantValues;

/**
 * Class VariantValueForm
 * @package EshopCatalog\AdminModule\Components\Variants
 */
class VariantValueForm extends BaseControl
{
	/** @var VariantValue */
	protected $variantValue;

	/** @var Variants */
	protected $variantsService;

	/** @var VariantValues */
	protected $variantValuesService;

	public function __construct(Variants $variants, VariantValues $variantValues)
	{
		$this->variantsService      = $variants;
		$this->variantValuesService = $variantValues;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$variants = [];
		foreach ($this->variantsService->getEr()->createQueryBuilder('v')->select('v.id, vt.name')
			         ->join('v.texts', 'vt', 'WITH', 'vt.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->orderBy('v.position')->getQuery()->getArrayResult() as $v) {
			$variants[$v['id']] = $v['name'];
		}

		$form->addSelect('variant', 'eshopCatalog.variantValueForm.variant', $variants)->setRequired();
		$form->addText('name', 'eshopCatalog.variantValueForm.value')->setIsMultilanguage()
			->setMaxLength(255)->setRequired();
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);

		$form->addSubmit('saveAndNew', null ?: 'eshopCatalog.defaultForm.saveAndNew')->getControlPrototype()->addHtml('<i class="fa fa-save"></i>');
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$variant      = $this->variantsService->getReference($values->variant);
			$variantValue = $this->variantValue ?: new VariantValue($variant);

			$mTexts = $form->prepareMultilangTextsForEntity(['name'], $values);
			foreach ($mTexts as $lang => $vals) {
				$valueText = $variantValue->getText($lang) ?: new VariantValueText($variantValue, $lang);
				foreach ($vals as $k => $v)
					$valueText->$k = $v;

				$this->em->persist($valueText);
			}
			$variantValue->setVariant($variant);
			$variantValue->isPublished = $values->isPublished;

			$this->em->persist($variantValue);
			$this->em->flush();
			$form->addCustomData('variantValueId', $variantValue->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.variantValueForm.variantValueSaved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setVariantValue($id)
	{
		$this->variantValue = $this->variantValuesService->get($id);

		if ($this->variantValue) {
			$form = $this['form'];
			$form->setDefaults([
				'variant'     => $this->variantValue->getVariant()->getId(),
				'isPublished' => $this->variantValue->isPublished,
			]);

			$mTexts = $form->prepareMultilangTextsForForm(['name'], $this->variantValue->getTexts()->toArray());
			$form->setDefaults($mTexts['texts']);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
