<?php declare(strict_types = 1);


namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseService;
use EshopCatalog\FrontModule\Model\Dao\Product;

/**
 * Class GoogleXml
 * @package EshopCatalog\Model
 */
class GoogleXmlService extends BaseService
{
	/** @var string */
	protected $siteName;

	/** @var string */
	public $baseUrl;

	/** @var Categories */
	protected $categoriesService;

	public function __construct($siteName, Categories $categories)
	{
		$this->siteName          = $siteName;
		$this->categoriesService = $categories;
	}

	/** @param string $baseUrl */
	public function setBaseUrl($baseUrl)
	{
		$this->baseUrl = $baseUrl;
	}

	/**
	 * @return string
	 */
	public function getSiteName()
	{
		return $this->siteName;
	}

	/**
	 * @param Product $product
	 *
	 * @return \DOMDocument xml SHOPITEM
	 */
	public function createXmlShopitem($product)
	{
		$dom               = new \DOMDocument();
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('item');

		$child = $dom->createElement('g:id', (string) $product->getId());
		$shopitem->appendChild($child);

		$child     = $dom->createElement('title');
		$childText = $dom->createTextNode((string) $product->getName());
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('description');
		$childText = $dom->createCDATASection((string) $product->getDescription());
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child = $dom->createElement('link', (string) $product->link);
		$shopitem->appendChild($child);

		$child = $dom->createElement('g:price', number_format($product->getPrice(), 2, ".", ""));
		$shopitem->appendChild($child);

		if ($product->quantity > 0) {
			$child = $dom->createElement('g:availability', 'in stock');
			$shopitem->appendChild($child);
		}

		if ($product->getGallery() && $product->getGallery()->cover) {
			$child = $dom->createElement('g:image_link', (string) ($this->baseUrl . $product->getGallery()->cover->file));
			$shopitem->appendChild($child);
		}

		if ($product->getEan()) {
			$child = $dom->createElement('g:gtin', (string) $product->getEan());
			$shopitem->appendChild($child);
		}

		if ($product->getDefaultCategory()) {
			$category = $this->categoriesService->get($product->getDefaultCategory());
			$child    = $dom->createElement('g:google_product_category', (string) $category->name);
			$shopitem->appendChild($child);
		}

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('g:brand', (string) $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		$dom->appendChild($shopitem);

		return $dom;
	}
}
