<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Categories;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Model\Products;
use Navigations\Model\Event\RouteInEvent;
use EshopCatalog\Model\Event\RouteInFindAliasEvent;
use Navigations\Model\Event\RouteOutEvent;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Event\RouteUpdateCacheDep;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Pages\FrontModule\Model\TemplatePages;
use EshopCatalog\FrontModule\Model\Dao;
use Tracy\Debugger;

class Category extends BaseNavigation implements INavigationItem
{
	protected $title = 'category';

	protected $presenter = 'EshopCatalog:Front:Default';
	protected $action    = 'category';

	const CACHE_CATEGORY = 'eshopCatalogCategory';

	/** @var Categories */
	protected $categoriesService;

	/** @var Products */
	protected $productsService;

	/** @var TemplatePages */
	protected $templatePagesService;

	public function __construct(Categories $categories, Products $products, TemplatePages $templatePages)
	{
		$this->categoriesService    = $categories;
		$this->productsService      = $products;
		$this->templatePagesService = $templatePages;
	}

	/**
	 * @return BaseContainer
	 */
	public function getFormContainer()
	{
		$container = new BaseContainer();

		$categories = [];
		foreach ($this->categoriesService->getEr()->createQueryBuilder('c')->select('c.id, c.lvl, ct.name')
			         ->where('c.isPublished = 1')
			         ->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->orderBy('c.root')->addOrderBy('c.lft')->getQuery()->getArrayResult() as $category) {
			$title = ' ' . $category['name'];
			for ($i = 1; $i < $category['lvl']; $i++)
				$title = '---' . $title;
			$categories[$category['id']] = $title;
		}

		$templates = [];
		foreach ($this->templatePagesService->getEr()->findBy([]) as $t)
			$templates[$t->getId()] = $t->title;

		$container->addSelect('category', $this->t('eshopCatalog.navigationBuilder.category'), $categories)
			->setTranslator(null);
		$container->addSelect('templatePage', $this->t('eshopCatalog.navigationBuilder.template'), $templates)
			->setTranslator(null);

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{

	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		// Přidat klíče pouze pokud se nejedná o filtr, jinak se to zasekne a nepůjdou upravit kategorie
		if (!isset($urlParams['productsFilter-filter']) && !isset($urlParams['productsFilter-sort'])) {
			$cache[Cache::TAGS][] = 'category/' . $urlParams['id'];
			$cache[Cache::TAGS][] = 'categories';
		}

		$this->eventDispatcher->dispatch(BaseNavigation::class . '::routeUpdateCacheDep', new RouteUpdateCacheDep($cache, $urlParams));

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		return $this->presenter == $presenter && $this->action == $action;
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 *
	 * @return array|mixed
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$parts    = explode('/', $urlParams['path']);
		$category = null;

		$return = [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => null,
		];

		if (count($parts) > 1) {
			foreach (array_reverse($parts) as $alias) {
				foreach ($this->categoriesService->findByAlias($alias) as $c) {
					$error = false;

					$path = [$c];
					$p    = $c->getParent();

					while ($p) {
						$path[] = $p;
						$p      = $p->getParent();
					}
					$path = array_reverse($path);

					foreach ($parts as $k => $part) {
						if (!isset($path[$k]) || $part != $path[$k]->alias && $k > 0) {
							$error = true;
							break;
						}
					}

					if (!$error && $parts[0] == $navigation->alias) {
						$category = $c->id;
						break 2;
					}
				}

				$this->eventDispatcher->dispatch(BaseNavigation::class . '::routeInFindAlias', new RouteInFindAliasEvent($alias, $urlParams, $return));
				if (isset($return['id'])) {
					$category = $return['id'];
					break;
				}
			}
		}

		if ($category)
			$return['id'] = $category;
		else
			$return['id'] = $navigation->componentParams['category'];

		$this->eventDispatcher->dispatch(BaseNavigation::class . '::routeIn', new RouteInEvent($navigation, $urlParams, $return));

		return $return;
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		$cfUrl       = '';
		$cfUrlParams = [];

		$categoryId = $urlParams['id'] ?? $navigation->componentParams['category'];
		$path       = $this->categoriesService->getQueryPath((int) $categoryId);
		$tmp        = [];

		if ($path && !$path[0])
			array_shift($path);

		if (count($path) > 1) {
			$parentFound = false;
			foreach (array_reverse($path) as $k => $c) {
				if ($c['id'] == $navigation->componentParams['category']) {
					$tmp[]       = $navigation->alias;
					$parentFound = true;
					break;
				} else {
					$tmp[] = $c['alias'];
				}
			}
			if (!$parentFound)
				return null;

			$tmp = array_reverse($tmp);
		} else {
			if (isset($urlParams['id']) && $navigation->componentParams['category'] != $urlParams['id'])
				return null;

			$tmp[] = $navigation->alias;
		}

		$cfUrl = implode('/', $tmp);

		if ($cfUrl) {
			$cfUrl = '/' . ltrim($cfUrl, '-');
		}
		$this->eventDispatcher->dispatch(BaseNavigation::class . '::routeOut', new RouteOutEvent($navigation, $urlParams, $cfUrl));

		return $cfUrl;
	}

	/**
	 * @param Navigation|null $navigation
	 * @param bool            $siteMapData
	 *
	 * @return array
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		$category = $this->categoriesService->getEr()->createQueryBuilder('c')->select('c.lft, c.gt')
			->where('c.id = :id')
			->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			->setParameters([
				'lang' => $lang,
				'id'   => $navigation->componentParams['category'],
			])
			->getQuery()->getOneOrNullResult(Query::HYDRATE_SCALAR);

		foreach ($this->categoriesService->getEr()->createQueryBuilder('c')->select('c.id, ct.name, ct.seo, c.modified')
			         ->andWhere('c.isPublished = 1')->andWhere('c.lft >= :lft')->andWhere('c.lft <= :gt')
			         ->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			         ->setParameters([
				         'lang' => $lang,
				         'lft'  => $category['lft'],
				         'gt'   => $category['gt'],
			         ])
			         ->getQuery()->getScalarResult() as $row) {
			$id = $row['id'];

			$data = [
				'action'    => $this->action,
				'id'        => $id,
				'presenter' => $this->presenter,
				'locale'    => $lang,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($row['modified'])
					$siteMapUrl->setModified($row['modified']);

				$seo = $row['seo'] ? unserialize($row['seo']) : null;
				if ($seo)
					$siteMapUrl->setSeo($seo, null, true);

				$data['siteMapData']['default'] = $siteMapUrl;
			}
			$arr[] = $data;
		}

		return $arr;
	}
}
