<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\DateTime;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="eshop_catalog__category", indexes={
 *     @ORM\Index(name="publish_id_id_lft", columns={"is_published", "id", "parent_id", "lft"}),
 *     @ORM\Index(name="root_lft_id", columns={"tree_root", "lft", "id"}),
 * })
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\NestedTreeRepository")
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "CategoryListener"})
 */
class Category
{
	use Identifier;
	use MagicAccessors;
	use TTranslateListener;

	/**
	 * @var string
	 * @ORM\Column(name="image", type="string", nullable=true, length=255)
	 */
	public $image;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", options={"default": 0})
	 */
	public $isPublished;

	/**
	 * @var Category[]
	 * @ORM\OneToMany(targetEntity="Category", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	protected $children;

	/**
	 * @var Category
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Category", inversedBy="childrens", cascade={"persist"})
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $parent;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(name="lft", type="integer")
	 */
	private $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(name="lvl", type="integer")
	 */
	protected $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	private $gt;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Category")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", onDelete="CASCADE")
	 */
	private $root;

	/**
	 * @ORM\OneToMany(targetEntity="CategoryTexts", mappedBy="id", indexBy="lang", cascade={"persist", "remove"})
	 */
	protected $categoryTexts;

	/**
	 * @ORM\OneToMany(targetEntity="CategoryProduct", mappedBy="category", indexBy="category", cascade={"all"})
	 */
	protected $categoryProducts;

	/**
	 * @var CategoryFilter[]
	 * @ORM\OneToMany(targetEntity="CategoryFilter", mappedBy="category", indexBy="feature_id")
	 * @ORM\OrderBy({"position" = "ASC"})
	 */
	public $filters;

	/**
	 * @var int
	 * @ORM\Column(name="filters_from_parent", type="smallint", options={"default": 1})
	 */
	public $filtersFromParent;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(name="created", type="datetime", options={"default":"CURRENT_TIMESTAMP"})
	 */
	private $created;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(name="modified", type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * Vlastní hodnoty třeba z rozšířeních
	 * @var array
	 * @ORM\Column(name="attrs", type="array", nullable=true)
	 */
	protected $attrs;

	/**
	 * @var string
	 * @ORM\Column(name="rod", type="string", nullable=true, length=1)
	 */
	public $rod;

	/**
	 * @var CategoryExport
	 * @ORM\OneToMany(targetEntity="CategoryExport", mappedBy="id", indexBy="service")
	 */
	public $exports;

	public function __construct()
	{
		$this->categoryTexts     = new ArrayCollection();
		$this->categoryProducts  = new ArrayCollection();
		$this->children          = new ArrayCollection();
		$this->filters           = new ArrayCollection();
		$this->exports           = new ArrayCollection();
		$this->isPublished       = 0;
		$this->created           = new DateTime();
		$this->filtersFromParent = 1;
		$this->attrs             = [];
	}

	/**
	 * @return int
	 */
	public function getLvl() { return $this->lvl; }

	public function getRoot() { return $this->root; }

	public function addChildren($child)
	{
		$this->children->add($child);
	}

	public function addCategoryText($lang)
	{
		$this->categoryTexts[$lang] = new CategoryTexts($this, $lang);
	}

	/**
	 * @param null|string $lang
	 *
	 * @return CategoryTexts|null
	 */
	public function getCategoryText($lang = null)
	{
		return $this->categoryTexts[$lang ?: $this->locale];
	}

	public function getCategoryTexts() { return $this->categoryTexts; }

	public function setCategoryTexts(array $categoryTexts): self
	{
		$this->categoryTexts = new ArrayCollection($categoryTexts);

		return $this;
	}

	public function setCategoryText(CategoryTexts $categoryTexts)
	{
		return $this->categoryTexts->set($categoryTexts->getLang(), $categoryTexts);
	}

	/**
	 * @return Category
	 */
	public function getParent()
	{
		return $this->parent;
	}

	/**
	 * @param Category $parent
	 */
	public function setParent($parent)
	{
		$this->parent = $parent;
	}

	public function addProduct(CategoryProduct $categoryProduct)
	{
		$this->categoryProducts->add($categoryProduct);
	}

	/**
	 * @return DateTime
	 */
	public function getCreated() { return $this->created; }

	/**
	 * @return DateTime
	 */
	public function getModified() { return $this->modified ?: $this->getCreated(); }

	public function getAttrs(): array { return $this->attrs ?: []; }

	public function getAttr(string $key) { return $this->attrs[$key] ?? null; }

	public function setAttr(string $key, string $value): self
	{
		$this->attrs[$key] = $value;

		return $this;
	}

	public function removeAttr(string $key): self
	{
		unset($this->attrs[$key]);

		return $this;
	}
}
