<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Helpers\Traits\TPublish;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Doctrine\Common\Collections\ArrayCollection;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;

/**
 * Class VariantValue
 * @package EshopCatalog\Model\Entities
 *
 * @ORM\Table(name="eshop_catalog__variant_value")
 * @ORM\Entity
 * @ORM\EntityListeners({"VariantValueListener"})
 */
class VariantValue
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var Variant
	 * @Gedmo\SortableGroup
	 * @ORM\ManyToOne(targetEntity="Variant", inversedBy="values")
	 * @ORM\JoinColumn(name="variant_id", referencedColumnName="id")
	 */
	protected $variant;

	/**
	 * @var VariantValueText[]
	 * @ORM\OneToMany(targetEntity="VariantValueText", mappedBy="variantValue", indexBy="lang", cascade={"persist", "remove"})
	 */
	protected $texts;

	/**
	 * @var int
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	private $position;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint")
	 */
	public $isPublished;

	public function __construct(Variant $variant)
	{
		$this->variant     = $variant;
		$this->texts       = new ArrayCollection();
		$this->isPublished = 0;
		$this->setPosition(-1);
	}

	/**
	 * @param string $lang
	 *
	 * @return $this
	 */
	public function addText(string $lang)
	{
		$this->texts->set($lang, new VariantValueText($this, $lang));

		return $this;
	}

	/**
	 * @param string $lang
	 *
	 * @return VariantValueText|null
	 */
	public function getText($lang) { return $this->texts->get($lang); }

	/**
	 * @param Variant $variant
	 *
	 * @return $this
	 */
	public function setVariant(Variant $variant)
	{
		$this->variant = $variant;

		return $this;
	}

	/**
	 * @return Variant
	 */
	public function getVariant() { return $this->variant; }

	/**
	 * @param VariantValueText $variantValueText
	 *
	 * @return $this
	 */
	public function setText(VariantValueText $variantValueText)
	{
		$this->texts->set($variantValueText->getLang(), $variantValueText);

		return $this;
	}

	/******
	 * === Position
	 */

	/**
	 * @param int $position
	 *
	 * @return $this
	 */
	public function setPosition(int $position)
	{
		$this->position = $position;

		return $this;
	}

	/**
	 * @return int
	 */
	public function getPosition() { return $this->position; }
}
