<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Categories;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\FrontModule\Model\SeoContainer;
use EshopCatalog\Model\Entities\CategoryFilter;
use EshopCatalog\Model\Entities\CategoryTmp;
use EshopCatalog\Model\Entities\Feature;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\AdminModule\Model\Categories;

/**
 * Class CategoryForm
 * @package EshopCatalog\AdminModule\Components\Categories
 */
class CategoryForm extends BaseControl
{
	/** @var int @persistent */
	public $categoryId;

	/** @var Category */
	public $category;

	/** @var CategoryTexts */
	public $categoryTexts;

	/** @var Categories */
	protected $categoryServices;

	/** @var SeoContainer */
	protected $seoContainerService;

	/** @var Category */
	protected $root;

	public function __construct(Categories $categories, SeoContainer $seoContainer)
	{
		$this->categoryServices    = $categories;
		$this->seoContainerService = $seoContainer;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);

		$root = $this->categoryServices->getEr()->findOneBy([
			'lvl' => 0,
		]);

		if (!$root) {
			$root              = new Category();
			$root->isPublished = 1;

			$texts = new CategoryTexts($root, $this->translator->getLocale());
			$texts->setName('ROOT');
			$root->setCategoryText($texts);

			$this->em->persist($root)->persist($texts)->flush();
		}
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$root       = $this->categoryServices->getEr()->findOneBy(['lvl' => 0, 'parent' => null]);
		$categories = [$root->getId() => ''];
		foreach ($this->em->getRepository(Category::class)->createQueryBuilder('c')
			         ->select('c.id, ct.name, c.lvl')
			         ->where('c.lvl > 0')
			         ->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->orderBy('c.root')->addOrderBy('c.lft')
			         ->getQuery()->getResult() as $c) {
			$name = ' ' . $c['name'];
			for ($i = 1; $i < $c['lvl']; $i++)
				$name = '---' . $name;
			$categories[$c['id']] = trim($name);
		}
		$filters = [];
		foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')->select('f.id, ft.name')
			         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
			         ->andWhere('f.useAsFilter = 1')
			         ->setParameters(['lang' => $this->translator->getLocale()])->orderBy('f.position')
			         ->getQuery()->getArrayResult() as $row) {
			$filters[$row['id']] = $row['name'];
		}

		$form->addText('name', 'eshopCatalog.categoryForm.name')->setRequired()->setMaxLength(255)->setIsMultilanguage();
		$form->addText('alias', 'eshopCatalog.categoryForm.alias')->setMaxLength(255)->setIsMultilanguage();
		$form->addText('nameH1', 'eshopCatalog.categoryForm.nameH1')->setMaxLength(255)->setIsMultilanguage();
		$form->addBool('isPublished', 'eshopCatalog.categoryForm.isPublished')->setDefaultValue(1);
		$form->addSelect('parent', $this->t('default.parent'), $categories)->setTranslator(null);
		$form->addFilesManager('image', 'default.image');
		$form->addEditor('desc', 'eshopCatalog.categoryForm.desc')->setHeight(300)->setIsMultilanguage();
		$form->addBool('filtersFromParent', 'eshopCatalog.categoryForm.filtersFromParent')->setDefaultValue(1);
		$form->addSortableCheckboxList('filters', $this->t('eshopCatalog.categoryForm.availableFilters'), $filters)->setTranslator(null);

		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->categoryId) {
				$this->category = $this->categoryServices->get($this->categoryId);
				$this->categoryTexts = $this->em->getRepository(CategoryTexts::class)->findOneBy(['id' => $this->categoryId, 'lang' => 'cs']);
			}

			$category = $this->category ?: new Category();
			$seoData  = $this->seoContainerService->getFormData($values->seo, true);

			if (isset($this->category))
				$category->setCategoryText($this->categoryTexts);
			else
				$category->addCategoryText('cs');

			$category->isPublished       = $values->isPublished;
			$category->image             = $values->image;
			$category->filtersFromParent = $values->filtersFromParent;

			// Texty
			$mTexts = [];
			foreach (['name', 'alias', 'description' => 'desc', 'nameH1'] as $pv => $v) {
				$pv = is_string($pv) ? $pv : $v;
				foreach ($values->$v as $lang => $val) {
					$mTexts[$lang][$pv] = $val;
				}
			}

			foreach ($mTexts as $lang => $vals) {
				$categoryText = $category->getCategoryText($lang) ?: new CategoryTexts($category, $lang);
				foreach ($vals as $k => $v)
					$categoryText->$k = $v;
				$categoryText->setSeo($seoData[$lang]);

				$this->em->persist($categoryText);
			}

			if ($values->parent == '') {
				$category->setParent(null);
			} else {
				$category->setParent($this->categoryServices->get($values->parent));
			}

			$this->em->persist($category);

			// FILTRY
			$formFilters     = $values->filters;
			$categoryFilters = $category->filters->toArray();

			// nové
			foreach (array_diff($formFilters, array_keys($categoryFilters)) as $k => $v) {
				$categoryFilter = new CategoryFilter($category, $this->em->getReference(Feature::class, $v), $k);
				$this->em->persist($categoryFilter);
				$category->filters->add($categoryFilter);
			}

			// odstranit
			foreach (array_diff(array_keys($categoryFilters), $formFilters) as $v) {
				$categoryFilter = $category->filters->get($v);
				if ($categoryFilter) {
					$category->filters->remove($v);
					$this->em->remove($categoryFilter);
				}
			}

			// seřadit
			foreach ($formFilters as $k => $v) {
				$categoryFilter = $category->filters->get($v);
				if ($categoryFilter) {
					$categoryFilter->position = $k;
					$this->em->persist($categoryFilter);
				}
			}

			$this->em->flush();
			$form->addCustomData('categoryId', $category->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.categoryForm.categorySaved', Flashes::FLASH_SUCCESS);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setCategory($id)
	{
		$this->categoryId    = $id;
		$this->category      = $this->em->getRepository(Category::class)->find($id);
		$this->categoryTexts = $this->em->getRepository(CategoryTexts::class)->findOneBy(['id' => $id, 'lang' => 'cs']);

		if ($this->category) {
			$c    = $this->category;
			$form = $this['form'];

			$defaults = ([
				'isPublished'       => $c->isPublished,
				'image'             => $c->image,
				'filtersFromParent' => $c->filtersFromParent,
			]);
			if (isset($c->parent) && array_key_exists($c->parent->getId(), $form['parent']->getItems()))
				$defaults['parent'] = $c->parent->getId();

			$form->setDefaults($defaults);

			// Texty
			$mTexts  = [];
			$seoData = [];
			foreach ($c->getCategoryTexts()->toArray() as $lang => $texts) {
				foreach (['name', 'alias', 'description' => 'desc', 'nameH1'] as $k => $v) {
					$k = is_string($k) ? $k : $v;

					$mTexts[$v][$lang] = $texts->$k;
				}

				foreach ($texts->getSeo() as $k => $v)
					$seoData[$k][$lang] = $v;
			}
			$form->setDefaults($mTexts);
			$this->seoContainerService->setDefaults($form['seo'], $seoData);

			// Filtry
			$defaults = [];
			foreach ($c->filters->toArray() as $k => $v) {
				if (array_key_exists($k, $form['filters']->getItems()))
					$defaults[] = $k;
			}
			$form['filters']->setDefaultValue($defaults);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}

