<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\SpecificPrices;

use Core\Model\UI\BaseControl;
use EshopCatalog\AdminModule\Model\SpecificPrices;
use Nette\Utils\Html;

/**
 * Class SpecificPricesGrid
 * @package EshopCatalog\AdminModule\Components\SpecificPrices
 */
class SpecificPricesGrid extends BaseControl
{
	/** @var Manufac */
	protected $specificPricesServices;

	public function __construct(SpecificPrices $specificPrices)
	{
		$this->specificPricesServices = $specificPrices;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->specificPricesServices->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		$qb = $this->specificPricesServices->getEr()->createQueryBuilder('sp')
			->orderBy('sp.position');

		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('specificPricesGrid:gridSortableRow!');

		// Columns
		$grid->addColumnLink('name', 'eshopCatalog.defaultGrid.name', 'SpecificPrices:edit');
		$grid->addColumnNumber('reduction', 'eshopCatalog.defaultGrid.reduction');
		$grid->addColumnStatus('isPercentage', 'eshopCatalog.defaultGrid.isPercentage')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.percentage')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.value')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];
		$grid->addColumnDateTime('fromDate', 'eshopCatalog.specificPricesGrid.from');
		$grid->addColumnDateTime('toDate', 'eshopCatalog.specificPricesGrid.to');
		$grid->addColumnStatus('isPublished', 'eshopCatalog.defaultGrid.isPublished')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterText('name', '');

		// Actions
		$grid->addAction('edit', '', 'SpecificPrices:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->specificPricesServices->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->specificPricesServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else
			$this->redirect('this');
	}
}
