<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Variants;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Variants;
use EshopCatalog\AdminModule\Model\VariantValues;
use EshopCatalog\Model\Entities\Variant;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;

/**
 * Class VariantValuesGrid
 * @package EshopCatalog\AdminModule\Components\Variants
 */
class VariantValuesGrid extends BaseControl
{
	/** @var Variants */
	protected $variantsService;

	/** @var VariantValues */
	protected $variantValuesService;

	/** @var Variant */
	protected $variant;

	public function __construct(Variants $variants, VariantValues $variantValues)
	{
		$this->variantsService      = $variants;
		$this->variantValuesService = $variantValues;
	}

	public function render()
	{
		if (!$this->variant)
			return '';
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->variantValuesService->remove($id))
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		else
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);

		if (!$this->variant)
			return $grid;

		$qb = $this->variantValuesService->getEr()->createQueryBuilder('vv')
			->join('vv.texts', 'vvt', 'WITH', 'vvt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->where('vv.variant = :id')->setParameter('id', $this->variant->getId())
			->orderBy('vv.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('variantValuesGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.value')->setRenderer(function($row) {
			return Html::el('a', ['href' => $this->getPresenter()->link('VariantsFeatures:editVariantValue', $row->getId())])
				->setText($row->getText($this->translator->getLocale())->name);
		});
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Actions
		$grid->addAction('edit', '', 'VariantsFeatures:editVariantValue')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->variantValuesService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->variantValuesService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else
			$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setVariant($id)
	{
		$this->variant = $this->variantsService->get($id);

		if (!$this->variant)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
