<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Kdyby\Events\EventManager;
use Kdyby\Events\EventArgsList;
use Nette\Utils\ArrayHash;
use Nette\Application\UI\Multiplier;

class CartAddForm extends BaseControl
{
	/** @var EventManager */
	public $eventManager;

	/** @var Products */
	protected $productsService;

	/** @var int @persistent */
	protected $productId;

	/** @var Product */
	protected $product;

	public function __construct(EventManager $eventManager, Products $products)
	{
		$this->eventManager    = $eventManager;
		$this->productsService = $products;
	}

	public function render()
	{
		$this->template->setFile($this->getTemplateFile());
		$this->template->product = $this->product;
		$this->template->render();
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);

		if ($this->productId && !$this->product) {
			$this->setProduct($this->productId);
		}
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		//$form->setAjax();

		$form->addHidden('productId');
		$form->addHidden('variantId')->setHtmlAttribute('class', ['variantIdInput']);

		$form->addText('quantity')->setDefaultValue(1)->addRule(BaseForm::MIN, 'Počet musí být alespoň %d', 1);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$presenter = $this->getPresenter();

		$addedItem            = new AddedCartItem();
		$addedItem->quantity  = $values->quantity;
		$addedItem->productId = $values->productId;
		$addedItem->optionId  = $values->variantId;

		$this->eventManager->dispatchEvent('EshopOrders\FrontModule\Model\Carts::onAddItem', new EventArgsList([$addedItem]));

		$presenter->redrawControl('cartPreview');
		$this->getPresenter()->redrawControl('cartPreviewMobile');
	}

	/**
	 * @param int|Product $product
	 */
	public function setProduct($product)
	{
		if ($product instanceof Product) {
			$this->product   = $product;
			$this->productId = $product->getId();
		} else {
			$this->productId = $product;
			$this->product   = $this->productsService->get($product);
			$this->productsService->loadVariants($this->product);
			$this->productsService->loadQuantity($this->product);
		}

		if ($this->productId) {
			$this['form']->setDefaults([
				'productId' => $this->productId,
				//TODO default variant ID
			]);
		}
	}
}
