<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\Model\Config;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Paginator;

/**
 * Class Navigation
 * @package EshopCatalog\FrontModule\Components
 */
class Navigation extends BaseControl
{
	/** @var DaoNavigationItem[] */
	protected $navs;

	/** @var string */
	protected $navigationType;

	/** @var Navigations */
	protected $navigationsService;

	/** @var Categories */
	protected $categoriesService;

	/** @var CacheService */
	protected $cacheService;

	/** @var bool */
	protected $cacheCleaned = false;

	/** @var Category */
	protected $currentCategory;

	public function __construct(Navigations $navigations, Categories $categories, CacheService $cacheService)
	{
		$this->navigationsService = $navigations;
		$this->categoriesService  = $categories;
		$this->cacheService       = $cacheService;
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
	}

	public function render($template = 'Default')
	{
		$this->template->navs                  = $this->getNavs();
		$this->template->categoryChilds        = $this->getCategoriesTree();
		$this->template->currentCategoryChilds = $this->getCurrentCategory() ? $this->getCategories($this->getCurrentCategory()->id) : [];
		$this->template->render($this->getTemplateFile($template));
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	/**
	 * @return DaoNavigationItem[]
	 */
	public function getNavs()
	{
		if (!$this->navs) {
			foreach ($this->navigationsService->getPublishedByGroupType($this->navigationType, true) as $nav) {
				$this->navs[$nav->id] = $nav;
			}
		}

		return $this->navs;
	}

	/**
	 * @param int $parentId
	 *
	 * @return Category[]
	 */
	public function getCategories($parentId)
	{
		return $this->categoriesService->getHierarchy($parentId);
	}

	public function getCategoriesTree()
	{
		$key  = 'eshopHeaderNavData_' . $this->translator->getLocale();
		$data = $this->cacheService->categoryCache->load($key, function(&$dep) {
			$dep = [Cache::TAGS => ['hierarchy', Navigations::CACHE_NAMESPACE], Cache::SLIDING => true, Cache::EXPIRE => '1 week'];

			$arr = [];
			foreach ($this->getNavs() as $nav) {
				if (isset($nav->componentParams['category']))
					$arr[$nav->componentParams['category']] = $this->getCategories($nav->componentParams['category']);
			}

			return $arr;
		});

		if (empty($data) && $this->cacheCleaned == false) {
			$this->cacheCleaned = true;
			$this->cacheService->categoryCache->remove($key);
			$this->getCategoryChilds();
		}

		return $data;
	}

	/**
	 * @param string $type
	 *
	 * @return $this
	 */
	public function setNavigationType($type)
	{
		$this->navigationType = $type;

		return $this;
	}

	public function getCurrentCategory()
	{
		if (!$this->currentCategory) {
			if ($this->getPresenter()->getParameter('id'))
				$this->currentCategory = $this->categoriesService->get($this->getPresenter()->getParameter('id'));
		}

		return $this->currentCategory;
	}
}
