<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Contributte\Application\LinkGenerator;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\FrontModule\Model\Dao;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;

/**
 * class Categories
 * @package EshopCatalog\Model
 *
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseFrontEntityService
{

	const CACHE_NAMESPACE = 'eshopCatalogCategories';

	protected $entityClass = Category::class;

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	public $cacheDep = [
		Cache::TAGS    => ['categories'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var array */
	protected $cPath = [];

	/** @var Dao\Category[] */
	public $cCategory = [];

	/** @var array */
	public $cLinks = [];

	/** @var bool */
	protected $cacheExist;

	/** @var LinkGenerator */
	public $linkGenerator;

	/**
	 * Categories constructor.
	 *
	 * @param CacheService $cacheService
	 * @param DefaultLang  $defaultLang
	 */
	public function __construct(CacheService $cacheService, DefaultLang $defaultLang)
	{
		$this->cacheService = $cacheService;
		$this->defaultLang  = $defaultLang;
	}

	protected function cacheExist()
	{
		if ($this->cacheExist === null)
			$this->cacheExist = count(glob(TMP_DIR . '/cache/_' . self::CACHE_NAMESPACE . '/*')) > 1;

		return $this->cacheExist;
	}

	public function getHierarchy($id = null)
	{
		if ($id) {
			$cat = $this->get($id);

			if (!$cat)
				return [];

			foreach ($cat->childrenId as $c) {
				$c = $this->get($c);

				if (!$c)
					continue;

				$c->setChildren($this->getHierarchy($c->id));
				$result[$c->id] = $c;
			}
		} else {
			$lvl1 = $this->getEr()->createQueryBuilder('c')
				->select('c.id')
				->where('c.lvl = 1')->andWhere('c.isPublished = 1')
				->orderBy('c.root')->addOrderBy('c.lft')
				->getQuery()->getScalarResult();

			foreach ($lvl1 as $c) {
				$category = $this->get($c['id']);

				$result[$category->id] = $category;
			}
		}

		return $result;
	}

	/**
	 * @param int $categoryId
	 *
	 * @return Dao\Category[]
	 */
	public function getPath($categoryId)
	{
		$key = 'path_' . $this->defaultLang->locale . '_' . $categoryId;

		return $this->cacheService->categoryCache->load($key, function(&$dep) use ($categoryId) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'category/' . $categoryId;

			if (!isset($this->cPath[$categoryId]) && $categoryId) {
				$path = $this->getEr()->getPathQueryBuilder($this->em->getReference($this->entityClass, $categoryId))
					->select('node.id, ct.alias')
					->join('node.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
					->setParameter('lang', $this->defaultLang->locale)->getQuery()->getResult(Query::HYDRATE_ARRAY);

				$arr = [];
				foreach ($path as $row) {
					$dep[Cache::TAGS][] = 'category/' . $row['id'];
					$arr[]              = $row;
				}

				$this->cPath[$categoryId] = $arr;
			}

			return $this->cPath[$categoryId];
		});
	}

	/**
	 * @param $id
	 *
	 * @return string|null
	 */
	public function getLink($id) { return $this->cLinks[$id] ?? null; }

	/**
	 * @param int $id
	 *
	 * @return Dao\Category
	 */
	public function get($id)
	{
		if (!isset($this->cCategory[$id])) {
			$locale = $this->defaultLang->locale;
			$key    = 'category_' . $locale . '_' . $id;

			if (!$this->cacheExist()) {
				$categories = $this->getEr()->createQueryBuilder('c', 'c.id')
					->addSelect('ct, child, childt, parent')
					->where('c.isPublished = 1')
					->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
					->leftJoin('c.children', 'child')
					->leftJoin('child.categoryTexts', 'childt')
					->leftJoin('c.parent', 'parent')
					->setParameter('lang', $locale)
					->getQuery()->getResult(Query::HYDRATE_ARRAY);

				foreach ($categories as $category) {
					$key                              = 'category_' . $locale . '_' . $category['id'];
					$this->cCategory[$category['id']] = $this->loadFromCache($key, $category);
				}
			} else {
				$this->cCategory[$id] = $this->cacheService->categoryCache->load($key);
				if ($this->cCategory[$id])
					$this->loadLink($key, $this->cCategory[$id]);
			}

			if (!$this->cCategory[$id]) {
				$category = $this->getEr()->createQueryBuilder('c', 'c.id')
					->addSelect('ct, child, childt, parent')
					->where('c.isPublished = 1')
					->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
					->leftJoin('c.children', 'child', 'WITH')
					->leftJoin('child.categoryTexts', 'childt')
					->leftJoin('c.parent', 'parent')
					->where('c.id = :id')->setParameter('id', $id)->setParameter('lang', $locale)
					->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

				$this->cCategory[$id] = $this->loadFromCache($key, $category);
			}
		}

		return $this->cCategory[$id];
	}

	/**
	 * @param string $key
	 * @param array  $category
	 *
	 * @return Dao\Category
	 */
	protected function loadFromCache($key, $category)
	{
		$category = $this->cacheService->categoryCache->load($key, function(&$dep) use ($category) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'category/' . $category['id'];

			if ($category['parent'])
				$dep[Cache::TAGS][] = 'category/' . $category['parent']['id'];
			foreach ($category['children'] as $child)
				$dep[Cache::TAGS][] = 'category/' . $child['id'];

			return $this->fillDao($this->normalizeHydrateArray($category));
		});

		if ($category)
			$this->loadLink($key, $category);

		return $category;
	}

	protected function loadLink(string $key, Dao\Category &$category)
	{
		$category->link = $this->cacheService->navigationCache->load($key . '_link', function(&$dep) use ($category) {
			$dep = [Cache::TAGS => [Navigations::CACHE_NAMESPACE], Cache::EXPIRE => '1 week', Cache::SLIDING => true];

			return $this->linkGenerator ? $this->linkGenerator->link('EshopCatalog:Front:Default:category', ['id' => $category->id]) : '';
		});
	}

	/**
	 * @param string $alias
	 *
	 * @return \Doctrine\ORM\QueryBuilder
	 */
	protected function findByAliasBase($alias)
	{
		return $this->getEr()->createQueryBuilder('c')
			->select('c.id')
			->where('c.isPublished = :published')->setParameter('published', 1)
			->join('c.categoryTexts', 'ct')
			->andWhere('ct.lang = :lang')->setParameter('lang', $this->defaultLang->locale)
			->andWhere('ct.alias = :alias')->setParameter('alias', $alias);
	}

	/**
	 * @param string $alias
	 *
	 * @return Dao\Category
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function findByAlias($alias)
	{
		$category = $this->findByAliasBase($alias)
			->getQuery()->getScalarResult();

		return $category ? $this->get($category['id']) : null;
	}

	public function findIdByAliases($alias)
	{
		$categories = $this->findByAliasBase($alias)
			->getQuery()->getResult(Query::HYDRATE_ARRAY);

		return $categories ? array_map(function($r) { return $r['id']; }, $categories) : null;
	}

	/**
	 * @param int $categoryId
	 *
	 * @return Navigation|null
	 */
	public function findNavigation($categoryId)
	{
		return $this->cacheService->categoryCache->load('categoryNavigation/' . $categoryId, function(&$dep) use ($categoryId) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'category/' . $categoryId;

			$result = $this->em->getRepository(Navigation::class)->createQueryBuilder('n')
				->where('n.componentType = :type')->setParameter('type', 'eshopCatalog.navigation.category')
				->getQuery()->getResult();

			foreach ($result as $row) {
				if ($row->componentParams['category'] == $categoryId) {
					$dep[Cache::TAGS][] = 'navigation/' . $row->getId();

					return $row;
				}
			}

			return null;
		});
	}

	/**
	 * @param Dao\Category    $category
	 * @param Navigation|null $activeNavigation
	 *
	 * @return Dao\Category[]
	 */
	public function getBreadcrumb($category, $activeNavigation = null)
	{
		if ($category instanceof Dao\Category == false)
			$category = $this->get($category);
		$path = $this->getPath($category->id);

		if (!$path)
			return [];

		if (is_null($activeNavigation)) {
			$i = 0;
			while (is_null($activeNavigation) && isset($path[$i])) {
				$activeNavigation = $this->findNavigation($path[$i]['id']);
				$i++;
			}
		}

		$arr = [];
		foreach (array_reverse($path) as $c) {
			if ($c['id'] == $activeNavigation->componentParams['category'])
				break;
			$arr[] = $this->get($c['id']);
		}

		return array_reverse($arr);
	}

	/**
	 * @param array|int $id
	 *
	 * @return array|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getForNavigation($id)
	{
		$qb = $this->getEr()->createQueryBuilder('c')->select('c.id, ct.name')
			->andWhere(is_array($id) ? 'c.id IN (:id)' : 'c.id = :id')->setParameter('id', $id)
			->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			->setParameter('lang', $this->defaultLang->locale)
			->getQuery();

		return is_array($id) ? $qb->getArrayResult() : $qb->getOneOrNullResult(Query::HYDRATE_ARRAY);
	}

	protected function fillDao($category)
	{
		/** @var Dao\Category $c */
		$c = (new Dao\Category())
			->setId($category['id'])
			->setName($category['categoryTexts']['name'])
			->setNameH1($category['categoryTexts']['nameH1'])
			->setAlias($category['categoryTexts']['alias'])
			->setShortDescription($category['categoryTexts']['shortDescription'])
			->setDescription($category['categoryTexts']['description'])
			->setImage($category['image'])
			->setLvl($category['lvl'])
			->setSeo($category['categoryTexts']['seo'])
			->setCreated($category['created'])
			->setModified($category['modified'] ?: $category['created'])
			->setFiltersFromParent($category['filtersFromParent']);

		if ($category['parent']) {
			$c->setParentId($category['parent']['id']);
		}

		if ($category['children'])
			foreach ($category['children'] as $child)
				if ($child['isPublished'])
					$c->addChildrenId($child['id']);

		return $c;
	}

	/**
	 * @param array $category
	 *
	 * @return array
	 */
	protected function normalizeHydrateArray($category)
	{
		$result                  = $category;
		$result['categoryTexts'] = $category['categoryTexts'][$this->defaultLang->locale];

		return $result;
	}
}
