<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Dao\FilterItem;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryFilter;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Variant;
use EshopCatalog\Model\Entities\VariantValue;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantCombination;
use Kdyby\Doctrine\EntityManager;
use Nette\Caching\Cache;
use Nette\Caching\IStorage;
use Nette\SmartObject;

/**
 * TODO misto hledání podle productsId používat třeba kategorie
 *
 * Class FilterService
 * @package EshopCatalog\FrontModule\Model
 */
class FilterService
{
	use SmartObject;

	const CACHE_NAMESPACE = 'eshopCatalogFeatureVariant';

	/** @var EntityManager */
	protected $em;

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var Variants */
	protected $variantsService;

	/** @var Features */
	protected $featuresService;

	/** @var Config */
	protected $config;

	/** @var array */
	public static $cacheTags = [];

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['filters'],
		Cache::EXPIRE  => '5 minutes',
		Cache::SLIDING => false,
	];

	/** @var array */
	protected $cFeatures, $cFeatureValues, $cVariants, $cVariantValues;

	/**
	 * FilterService constructor.
	 *
	 * @param EntityManager $em
	 * @param DefaultLang   $defaultLang
	 * @param CacheService  $cacheService
	 * @param Variants      $variants
	 * @param Features      $features
	 * @param Config        $config
	 */
	public function __construct(EntityManager $em, DefaultLang $defaultLang, CacheService $cacheService, Variants $variants, Features $features, Config $config)
	{
		$this->em              = $em;
		$this->defaultLang     = $defaultLang;
		$this->cacheService    = $cacheService;
		$this->variantsService = $variants;
		$this->featuresService = $features;
		$this->config          = $config;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return FilterGroup[]
	 */
	public function getFeatures($productIds)
	{
		$groups = [];
		$key    = 'features_' . md5(serialize($productIds));
		$data   = $this->cacheService->defaultCache->load($key, function(&$dep) use ($productIds) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'features';

			if (self::$cacheTags['features'])
				$dep[Cache::TAGS][] = self::$cacheTags['features'];

			return $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
				->select('IDENTITY(fp.feature) as featureId, IDENTITY(fp.featureValue) as valueId')
				->join('fp.feature', 'f', 'WITH', 'f.useAsFilter = 1')
				->join('fp.featureValue', 'fv', 'WITH', 'fv.isPublished = 1')
				->where('fp.product IN (:products)')->setParameter('products', $productIds)
				->orderBy('f.position')->addOrderBy('fv.position')
				->getQuery()->getScalarResult();
		});

		foreach ($data as $row) {
			if (!isset($groups[$row['featureId']])) {
				$tmp = $this->featuresService->getFeatureById($row['featureId']);

				if ($tmp && $tmp['name'])
					$groups[$row['featureId']] = (new FilterGroup())
						->setId($tmp['id'])
						->setName($tmp['name'])
						->setType('feature');
			}

			$g = $groups[$row['featureId']];

			if (!isset($g->items[$row['valueId']])) {
				$tmp = $this->featuresService->getFeatureValueById($row['valueId']);

				if ($tmp && $tmp['name'])
					$g->items[$row['valueId']] = (new FilterItem())
						->setId($tmp['id'])
						->setValue($tmp['name'])
						->setType('featureValue');
			}
		}

		return $groups;
	}

	/**
	 * @param int[] $productsIds
	 *
	 * @return FilterGroup[]
	 */
	public function getVariants($productsIds)
	{
		$groups = [];
		$key    = 'variants_' . md5(serialize($productsIds));
		$data   = $this->cacheService->defaultCache->load($key, function(&$dep) use ($productsIds) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'variants';

			if (self::$cacheTags['variants'])
				$dep[Cache::TAGS][] = self::$cacheTags['variants'];

			return $this->em->getRepository(ProductVariantCombination::class)->createQueryBuilder('pvc')
				->select('v.id as variantId, val.id as valueId')
				->join('pvc.productVariant', 'pv')
				->join('pvc.variant', 'v')
				->join('pvc.value', 'val')
				->where('pv.product IN (:products)')->setParameter('products', $productsIds)
				->andWhere('v.useAsFilter = 1')
				->orderBy('v.position')->addOrderBy('val.position')
				->getQuery()->getScalarResult();
		});

		foreach ($data as $row) {
			if (!isset($groups[$row['variantId']])) {
				$tmp = $this->variantsService->getVariantById($row['variantId']);

				if ($tmp && $tmp['name'])
					$groups[$row['variantId']] = (new FilterGroup())
						->setId($tmp['id'])
						->setName($tmp['name'])
						->setType('variant');
			}

			$g = $groups[$row['variantId']];

			if (!isset($g->items[$row['valueId']])) {
				$tmp = $this->variantsService->getVariantValueById($row['valueId']);

				if ($tmp && $tmp['name'])
					$g->items[$row['valueId']] = (new FilterItem())
						->setId($tmp['id'])
						->setValue($tmp['name'])
						->setType('variantValue');
			}
		}

		return $groups;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getMinMaxPrice($productIds)
	{
		$key = 'minMax_' . md5(serialize($productIds));

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($productIds) {
			$dep = $this->cacheDep;

			if (self::$cacheTags['minMax'])
				$dep[Cache::TAGS][] = self::$cacheTags['minMax'];
			$data = [];
			$tmp  = $this->em->getRepository(Product::class)->createQueryBuilder('p')
				->select('MIN(p.price), MAX(p.price)')
				->where('p.id IN (:ids)')->setParameter('ids', $productIds)
				->getQuery()->getScalarResult();
			if (isset($tmp[0]))
				$data = ['min' => $tmp[0][1], 'max' => $tmp[0][2]];

			return $data;
		});
	}

	public function getManufacturers($productIds)
	{
		$key = 'manu_' . md5(serialize($productIds));

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($productIds) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'manufacturers';

			if (self::$cacheTags['manu'])
				$dep[Cache::TAGS][] = self::$cacheTags['manu'];

			$toUppercase = $this->config->get('filterManufacturersUppercase', false);
			$data        = [];
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('m.id, m.name')
				         ->join('p.manufacturer', 'm')->where('p.id IN (:ids)')->andWhere('m.name != \'\'')
				         ->andWhere('m.isPublished = 1')
				         ->setParameter('ids', $productIds)->groupBy('m.id')->orderBy('m.position')
				         ->getQuery()->getScalarResult() as $row)
				$data[$row['id']] = (new FilterItem())
					->setId($row['id'])
					->setValue($toUppercase ? mb_strtoupper($row['name']) : $row['name']);

			return $data;
		});
	}

	/**
	 * @param int  $id
	 * @param bool $deep
	 *
	 * @return array
	 */
	public function getCategoryFilters(int $id, $deep = true)
	{
		$key  = 'categoryFilters_' . $id . ($deep ? '_deep' : '');
		$data = [];

		$features = $this->cacheService->categoryCache->load($key, function(&$dep) use ($id, $deep) {
			$dep                 = $this->cacheDep;
			$dep[Cache::EXPIRE]  = '1 month';
			$dep[Cache::SLIDING] = true;
			$dep[Cache::TAGS][]  = 'category/' . $id;

			$find = function(int $id) {
				return $this->em->getRepository(CategoryFilter::class)->createQueryBuilder('f')
					->select('IDENTITY(f.feature) as featureId')
					->andWhere('f.category = :category')->setParameters(['category' => $id])
					->orderBy('f.position')
					->getQuery()->getArrayResult();
			};

			$features = $find($id);

			if (!$features && $deep) {
				try {
					$stmt = $this->em->getConnection()->prepare("
						SELECT T2.id 
						FROM (
							SELECT @r AS _id,
								(SELECT @r := parent_id FROM eshop_catalog__category WHERE id = _id) AS parent_id, @l := @l + 1 AS lvl 
								FROM (SELECT @r := {$id}, @l := 0) vars, eshop_catalog__category h WHERE @r <> 0) T1 
						JOIN eshop_catalog__category T2 ON T1._id = T2.id ORDER BY T1.lvl DESC
						");
					$stmt->execute();
					$tmp = $stmt->fetchAll();
				} catch (\Exception $e) {
					$tmp = [];
				}
				array_shift($tmp);
				array_pop($tmp);
				$categories = [];
				foreach ($tmp as $row) {
					$dep[Cache::TAGS][] = 'category/' . $row['id'];
					$categories[]       = $row['id'];
				}

				while (!empty($categories) && empty($features)) {
					$id       = array_pop($categories);
					$features = $find((int) $id);

					if ($features)
						break;
				}
			}

			return array_map(function($row) { return $row['featureId']; }, $features);
		});

		if (!$features)
			return $data;

		foreach ($features as $featureId) {
			$tmp = $this->featuresService->getFeatureById($featureId);

			$data[$featureId] = (new FilterGroup())
				->setId($tmp['id'])
				->setName($tmp['name'])
				->setType('feature');

			foreach ($this->featuresService->getValuesByFeatureId((int) $featureId) as $tmp) {
				$data[$featureId]->items[$tmp['id']] = (new FilterItem())
					->setId($tmp['id'])
					->setValue($tmp['name'])
					->setType('featureValue');
			}
		}

		return $data;
	}
}
