<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Contributte\Application\LinkGenerator;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\AdminModule\Model\SpecificPrices;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductVariantCombination;
use EshopCatalog\Model\Entities\ProductVariantSupplier;
use Gallery\FrontModule\Model\Albums;
use Gallery\Model\Entities\Album;
use Kdyby\Doctrine\ResultSet;
use Nette\Caching\Cache;

/**
 * TODO možnost přesunutí do fasády
 *
 * Class Products
 * @package EshopCatalog\FrontModule\Model
 */
class Products extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'eshopCatalogProducts';

	protected $entityClass = Product::class;

	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var Albums @inject */
	public $albumsService;

	/** @var Manufacturers @inject */
	public $manufacturersService;

	/** @var ProductVariants */
	private $productVariantsSevice;

	/** @var FeatureProducts */
	private $featureProductsService;

	/** @var SpecificPrices */
	private $specificPricesService;

	/** @var CacheService */
	protected $cacheService;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var callable[] */
	protected $customQB = [];

	/** @var array */
	protected $cProductsIdInCategory = [], $cSearch, $cProductsIdBySearch = [], $cProductsIdAll = [];

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['products'],
		Cache::EXPIRE  => '15 minutes',
		Cache::SLIDING => false,
	];

	public function __construct(ProductVariants $productVariants, FeatureProducts $featurePorducts, CacheService $cacheService, SpecificPrices $specificPrices)
	{
		$this->productVariantsSevice  = $productVariants;
		$this->featureProductsService = $featurePorducts;
		$this->cacheService           = $cacheService;
		$this->specificPricesService  = $specificPrices;

		$this->afterConstruct();
	}

	protected function afterConstruct() { }

	/**
	 * @param int|array $id
	 *
	 * @return array|mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getForNavigation($id)
	{
		$qb = $this->getEr()->createQueryBuilder('p')->select('p.id, pt.name')
			->andWhere(is_array($id) ? 'p.id IN (:id)' : 'p.id = :id')->setParameter('id', $id)
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->setParameter('lang', $this->defaultLang->locale)
			->getQuery();

		return is_array($id) ? $qb->getArrayResult() : $qb->getOneOrNullResult(Query::HYDRATE_ARRAY);
	}

	/**
	 * @param int $categoryId
	 *
	 * @return int[]
	 */
	public function getProductsIdInCategory($categoryId, $start = 0, $limit = null, $sort = null, $filter = [])
	{
		$keyBase = md5(serialize([$categoryId, $sort, $filter]));

		if (isset($this->cProductsIdInCategory[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);

			return $this->cProductsIdInCategory[$keyBase];
		} else {
			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($categoryId, $sort, $filter) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'listing/' . $categoryId;
				$dep[Cache::EXPIRE] = '5 minutes';

				if (isset($filter['manu']))
					$dep[Cache::TAGS][] = 'manufacturers';
				if (isset($filter['features']))
					$dep[Cache::TAGS][] = 'features';

				$query = (new ProductQuery($this->defaultLang->locale))
					->onlyInStockOrSupplier()
					->inCategory($categoryId)
					->selectIds();

				if ($filter)
					$query->useFilter($this->prepareFilter($filter));
				$this->prepareSort($query, $sort);

				$qb = $query->getQueryBuilder($this->getEr());
				foreach ($this->customQB as $cqb)
					$cqb($qb);

				$ids = [];
				foreach ($qb->getQuery()->getScalarResult() as $v)
					$ids[] = $v['id'];

				return $ids;
			});

			if (is_null($sort)) {
				$this->cProductsIdInCategory[$keyBase] = array_unique(array_merge($this->getRecommendedIds($categoryId, 0, null, null, $filter), $ids));
			} else
				$this->cProductsIdInCategory[$keyBase] = $ids;
		}

		if ($limit)
			return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);

		return $this->cProductsIdInCategory[$keyBase];
	}

	/**
	 * @param int        $categoryId
	 * @param array|null $filter
	 *
	 * @return int
	 */
	public function getProductsInCategoryCount($categoryId, $filter = [])
	{
		$key = 'productsInCategoryCount_' . $categoryId . '_' . md5(serialize($filter));

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($categoryId, $filter) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'listing/' . $categoryId;

			$query = (new ProductQuery($this->defaultLang->locale))
				->onlyInStockOrSupplier()
				->inCategory($categoryId)
				->selectIds();

			if ($filter)
				$query->useFilter($this->prepareFilter($filter));

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			return count($qb->getQuery()->getArrayResult());
		});
	}

	/**
	 * @param int $categoryId
	 *
	 * @return Dao\Product[]|ResultSet
	 */
	public function getProductsInCategory($categoryId, $start = 0, $limit = null, $sort = null, $filter = [])
	{
		$products = [];
		foreach ($this->getProductsIdInCategory($categoryId, $start, $limit, $sort, $filter) as $id) {
			$product = $this->get($id);
			if ($product)
				$products[$id] = $product;
		}

		return $products;
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return int[]
	 */
	public function getProductsIdBySearch($q, $start = 0, $limit = null, $sort = null, $filter = [])
	{
		$keyBase = md5(serialize([$q, $sort, $filter]));

		if (isset($this->cProductsIdBySearch[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);

			return $this->cProductsIdBySearch[$keyBase];
		} else {
			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($q, $start, $limit, $sort, $filter) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'listing/search';
				$dep[Cache::EXPIRE] = '5 minutes';

				if (isset($filter['manu']))
					$dep[Cache::TAGS][] = 'manufacturers';
				if (isset($filter['features']))
					$dep[Cache::TAGS][] = 'features';

				$query = (new ProductQuery($this->defaultLang->locale))
					->search($q)
					->selectIds();

				if ($filter)
					$query->useFilter($this->prepareFilter($filter));
				$this->prepareSort($query, $sort);

				$qb = $query->getQueryBuilder($this->getEr());
				foreach ($this->customQB as $cqb)
					$cqb($qb);

				$ids = [];
				foreach ($qb->getQuery()->getScalarResult() as $v)
					$ids[] = $v['id'];

				return $ids;
			});

			if (is_null($sort)) {
				$this->cProductsIdBySearch[$keyBase] = array_unique(array_merge($this->getRecommendedIds($q, 0, null, null, $filter, 'search'), $ids));
			} else
				$this->cProductsIdBySearch[$keyBase] = $ids;
		}

		if ($limit)
			return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);

		return $this->cProductsIdBySearch[$keyBase];
	}

	/**
	 * @param int        $categoryId
	 * @param array|null $filter
	 *
	 * @return int
	 */
	public function getProductsBySearchCount($q, $filter = [])
	{
		$key = 'searchCount' . '_' . md5(serialize([$q, $filter]));

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($categoryId, $filter) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'listing/search';

			$query = (new ProductQuery($this->defaultLang->locale))
				->search($categoryId)
				->selectIds();

			if ($filter)
				$query->useFilter($this->prepareFilter($filter));

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			return count($qb->getQuery()->getArrayResult());
		});
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return Dao\Product[]
	 */
	public function getProductsBySearch($q, $start = 0, $limit = null, $sort = null, $filter = null)
	{
		$products = [];
		foreach ($this->getProductsIdBySearch($q, $start, $limit, $sort, $filter) as $id) {
			$product = $this->get($id);
			if ($product)
				$products[$id] = $product;
		}

		return $products;
	}

	/**
	 * @param string $q
	 * @param int    $start
	 * @param null   $limit
	 * @param null   $sort
	 * @param null   $filter
	 *
	 * @return int[]
	 */
	public function getProductsIdAll($start = 0, $limit = null, $sort = null, $filter = [])
	{
		$keyBase = md5(serialize([$sort, $filter]));

		if (isset($this->cProductsIdAll[$keyBase])) {
			if ($limit)
				return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);

			return $this->cProductsIdAll[$keyBase];
		} else {
			$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($sort, $filter) {
				$dep = $this->cacheDep;

				if (isset($filter['manu']))
					$dep[Cache::TAGS][] = 'manufacturers';
				if (isset($filter['features']))
					$dep[Cache::TAGS][] = 'features';

				$query = (new ProductQuery($this->defaultLang->locale))
					->selectIds();

				if ($filter)
					$query->useFilter($this->prepareFilter($filter));
				$this->prepareSort($query, $sort);

				$qb = $query->getQueryBuilder($this->getEr());
				foreach ($this->customQB as $cqb)
					$cqb($qb);

				$ids = [];
				foreach ($qb->getQuery()->getScalarResult() as $v)
					$ids[] = $v['id'];

				return $ids;
			});

			if (is_null($sort)) {
				$this->cProductsIdAll[$keyBase] = array_unique(array_merge($this->getRecommendedIds(null, 0, null, null, $filter), $ids));
			} else
				$this->cProductsIdAll[$keyBase] = $ids;;
		}

		if ($limit)
			return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);

		return $this->cProductsIdAll[$key];
	}

	/**
	 * @param int  $start
	 * @param null $limit
	 * @param null $sort
	 * @param null $filter
	 *
	 * @return Dao\Product[]
	 */
	public function getProductsAll($start = 0, $limit = null, $sort = null, $filter = [])
	{
		$products = [];
		foreach ($this->getProductsIdAll($start, $limit, $sort, $filter) as $id) {
			$product = $this->get($id);
			if ($product) {
				$this->loadQuantity($product);
				$products[$id] = $product;
			}
		}

		return $products;
	}

	/**
	 * @param int        $ids
	 * @param int        $start
	 * @param int|null   $limit
	 * @param mixed|null $sort
	 *
	 * @return Dao\Product[]
	 */
	public function getProductsByIds($ids, $start = 0, $limit = null, $sort = null, $filter = null)
	{
		$products = [];

		$ids = array_slice($ids, $start, $limit);

		foreach ($ids as $id) {
			$product = $this->get($id);
			if ($product)
				$products[$id] = $product;
		}

		return $products;
	}

	public function getRecommendedIds($q = null, $start = 0, $limit = null, $sort = null, $filter = [], $type = 'category')
	{
		$keyBase = md5('recommended_' . serialize([$q, $sort, $filter, $type]));
		$key     = "$keyBase-$start-$limit";

		return $this->cacheService->defaultCache->load($key, function(&$dep) use ($q, $start, $limit, $sort, $filter, $type) {
			$dep                = $this->cacheDep;
			$dep[Cache::EXPIRE] = '5 minutes';

			if (isset($filter['manu']))
				$dep[Cache::TAGS][] = 'manufacturers';
			if (isset($filter['features']))
				$dep[Cache::TAGS][] = 'features';

			$query = (new ProductQuery($this->defaultLang->locale))
				->withTag('tip')
				->selectIds();

			switch ($type) {
				case 'category':
					$dep[Cache::TAGS][] = 'listing/' . $q;
					$query->onlyInStockOrSupplier();
					if ($q)
						$query->inCategory($q);
					break;
				case 'search':
					$dep[Cache::TAGS][] = 'listing/search';
					$query->search($q);
					break;
			}

			if ($filter)
				$query->useFilter($this->prepareFilter($filter));
			$sort = 'p.price ASC';
			$this->prepareSort($query, $sort);

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			$ids = [];
			foreach ($qb->getQuery()->setMaxResults($limit)->setFirstResult($start)->getScalarResult() as $v)
				$ids[] = $v['id'];

			return $ids;
		});
	}

	/** Vrati Dao produkt podle ID.
	 * @return Dao\Product
	 */
	public function get($id)
	{
		$locale = $this->defaultLang->locale;
		$key    = 'product_' . $locale . '_' . $id;

		/** @var Dao\Product $product */
		$product = $this->cacheService->productCache->load($key, function(&$dep) use ($id, $locale) {
			$dep                 = $this->cacheDep;
			$dep[Cache::TAGS][]  = 'product/' . $id;
			$dep[Cache::EXPIRE]  = '1 month';
			$dep[Cache::SLIDING] = true;

			$query = (new ProductQuery($locale))
				->withCategories()
				->withTexts()
				->byId($id);

			$qb = $query->getQueryBuilder($this->getEr());
			foreach ($this->customQB as $cqb)
				$cqb($qb);

			$result = $qb->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

			return $result ? $this->fillDao($this->normalizeHydrateArray($result)) : null;
		});

		if ($product) {
			if ($product->galleryId)
				$product->setGallery($this->albumsService->get($product->galleryId));

			if ($product->manufacturerId)
				$product->setManufacturer($this->manufacturersService->get($product->manufacturerId));

			$this->loadPrice($product);

			return $product;
		}

		return null;
	}

	/**
	 * @param Dao\Product $product
	 */
	public function loadVariants(Dao\Product &$product)
	{
		$product->setVariants($this->productVariantsSevice->getProductVariantsForProduct($product->getId()));
	}

	/**
	 * @param int $idVariant
	 */
	public function getVariant(int $idVariant)
	{
		return $this->productVariantsSevice->get($idVariant);
	}

	/**
	 * @param Dao\Product $product
	 */
	public function loadFeatures(Dao\Product &$product)
	{
		$product->setFeatures($this->featureProductsService->getFeaturesForProduct($product->getId()));
	}

	/**
	 * @param Dao\Product $product
	 * @param null        $customerId
	 */
	public function loadPrice(Dao\Product &$product, $customerId = null, $cartCountProduct = 1)
	{
		// overeni slevy pro skupinu
		if (isset($customerId)) {

		}
		$tempPrice = $product->price;
		if ($this->specificPricesService->getPrice($product->price, array_merge([(int) $product->defaultCategory], $product->categories), $customerId, $cartCountProduct))
			$product->retailPrice = $tempPrice;

		bdump($product);
	}

	/**
	 * @param Dao\Product $product
	 * @param int         $limit
	 */
	public function loadAlternative(Dao\Product &$product, $limit)
	{
		$data = $this->getProductsIdInCategory($product->defaultCategory);
		shuffle($data);

		$products = [];
		foreach (array_slice($data, 0, $limit) as $id)
			$products[$id] = $this->get($id);
		$this->loadQuantity($products);

		$product->setAlternatives($products);
	}

	/**
	 * @param Dao\Product|Dao\Product[] $product
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadQuantity(&$product)
	{
		if (is_array($product))
			$id = array_map(function($row) { return $row->getId(); }, $product);
		else
			$id = $product->getId();

		$query = (new ProductQuery($this->defaultLang->locale))
			->byId($id)
			->selectQuantity();

		$data = $query->getQueryBuilder($this->getEr())->groupBy('p.id')->getQuery()->getArrayResult();

		if (is_array($product)) {
			foreach ($data as $id => $row)
				$product[$id]->setQuantity(intval($row['productQ']) + intval($row['suppliersQ']) + intval($row['variantsQ']) + intval($row['variantSuppliersQ']));
		} else if (isset($data[$id]))
			$product->setQuantity(intval($data[$id]['productQ']) + intval($data[$id]['suppliersQ']) + intval($data[$id]['variantsQ']) + intval($data[$id]['variantSuppliersQ']));
	}

	/**
	 * @param array $product
	 *
	 * @return Dao\Product
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	protected function fillDao($product): Dao\Product
	{
		$p = new Dao\Product($product['id']);
		$p->setQuantity($product['quantity']);
		$p->setPrice((double) $product['price']);
		$p->setRetailPrice((double) $product['retailPrice']);
		$p->setName($product['productTexts']['name']);
		$p->setShortDescription($product['productTexts']['shortDescription']);
		$p->setDescription($product['productTexts']['description']);
		$p->setSeo($product['productTexts']['seo']);
		$p->setCreated($product['created']);
		$p->setModified($product['modified']);
		$p->setDefaultCategory($product['defaultCategory']);
		$p->setCategories($product['categories']);
		$p->setLink($this->linkGenerator->link('EshopCatalog:Front:Default:product', [$product['id']]));
		$p->setEan($product['ean']);
		$p->setCode1($product['code1']);

		if ($product['gallery'])
			$p->setGalleryId($product['gallery']);

		if ($product['manufacturer'])
			$p->setManufacturerId($product['manufacturer']);

		return $p;
	}

	/**
	 * @param array $product
	 *
	 * @return array
	 */
	protected function normalizeHydrateArray($product)
	{
		$result                    = $product[0];
		$result['gallery']         = $product['gallery'];
		$result['manufacturer']    = $product['manufacturer'];
		$result['productTexts']    = $product[0]['productTexts'][$this->defaultLang->locale];
		$result['defaultCategory'] = $product['defaultCategory'];
		$result['categories']      = array_map(function($r) { return $r['id_category']; }, $product[0]['categoryProducts']);
		unset($result['categoryProducts']);

		return $result;
	}

	protected function prepareFilter($filter)
	{
		if (isset($filter['variants'])) {
			$vals = [];
			foreach ($filter['variants'] as $g => $v1) {
				$vals = array_merge($vals, $v1);
			}

			$tmp      = [];
			$variants = [];
			foreach ($this->em->getRepository(ProductVariantCombination::class)->createQueryBuilder('pvc')
				         ->addSelect('IDENTITY(pvc.variant) as variantId')
				         ->where('IDENTITY(pvc.value) IN (:variants)')->setParameter('variants', $vals)->getQuery()->getArrayResult() as $row) {
				$variants[$row['id_product_variant']][$row['variantId']][] = $row['id_variant_value'];
			}

			foreach ($variants as $variant => $groups) {
				foreach ($filter['variants'] as $selGroup => $selValues) {
					if (!isset($groups[$selGroup])) {
						unset($variants[$variant]);
						break;
					}

					$isIn = false;
					foreach ($selValues as $val) {
						if (in_array($val, $selValues)) {
							$isIn = true;
						}
					}

					if (!$isIn) {
						unset($variants[$variant]);
						break;
					}
				}
			}

			$filter['variants'] = $variants ? array_keys($variants) : [];
		}

		if (isset($filter['onlyStock']) && $filter['onlyStock'] === true) {
			$pvs = [];
			foreach ($this->em->getRepository(ProductVariantSupplier::class)->createQueryBuilder('pvs')
				         ->select('pvs.quantity, IDENTITY(pv.product) as product')
				         ->andWhere('pvs.quantity > 0')
				         ->join('pvs.productVariant', 'pv')->getQuery()->useResultCache(true, 60)->getArrayResult() as $row)
				$pvs[] = $row['product'];

			$filter['onlyStockPVS'] = $pvs;
		}

		if (isset($filter['features'])) {
			$count = count($filter['features']);
			$ids   = [];
			$qb    = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')->select('IDENTITY(fp.product) as product');
			$tmp   = [];

			foreach ($filter['features'] as $group => $vals) {
				$or  = [];
				$qb2 = clone $qb;
				foreach ($vals as $k => $v) {
					$or[] = "IDENTITY(fp.featureValue) = :fpfv_{$i}_{$k}";
					$qb2->setParameter("fpfv_{$i}_{$k}", $v);
				}
				$qb2->andWhere(implode(' OR ', $or));
				foreach ($qb2->getQuery()->useResultCache(true, 60)->getArrayResult() as $row)
					$tmp[$row['product']] = isset($tmp[$row['product']]) ? $tmp[$row['product']] + 1 : 1;
			}

			foreach ($tmp as $k => $v) {
				if ($v == $count)
					$ids[] = $k;
			}
			$tmp = null;

			$filter['features'] = empty($ids) ? 'empty' : $ids;
		}

		return $filter;
	}

	public function prepareSort(ProductQuery &$query, $sort)
	{
		if ($sort == 'recommended' || is_null($sort)) {
			$sort = 'p.price ASC';
		}

		if (is_string($sort)) {
			$sort = explode(' ', $sort, 2);
			$query->sortBy($sort[0], $sort[1] ?? null);
		} else if (is_array($sort))
			$query->sortBy($sort);
	}
}
