<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Lang\DefaultLang;
use Core\Model\Lang\Translator;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Entities\Variant;
use EshopCatalog\Model\Entities\VariantValue;
use Kdyby\Doctrine\EntityManager;
use Nette\Caching\Cache;
use Nette\SmartObject;

/**
 * Class Variants
 * @package EshopCatalog\FrontModule\Model
 */
class Variants
{
	use SmartObject;

	const CACHE_NAMESPACE = 'eshopCatalogVariants';

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var EntityManager */
	protected $em;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['variants'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var array */
	protected $cVariants, $cVariantValues;

	/**
	 * Variants constructor.
	 *
	 * @param EntityManager $em
	 * @param Translator    $translator
	 * @param CacheService  $cacheService
	 * @param DefaultLang   $defaultLang
	 */
	public function __construct(EntityManager $em, CacheService $cacheService, DefaultLang $defaultLang)
	{
		$this->em           = $em;
		$this->cacheService = $cacheService;
		$this->defaultLang  = $defaultLang;
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 */
	public function getVariantById($id)
	{
		if (!$this->cVariants[$id]) {
			$locale = $this->defaultLang->locale;
			$key    = 'variants_' . $locale;

			$this->cVariants = $this->cacheService->defaultCache->load($key, function(&$dep) use ($id, $locale) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'variants';

				$data = [];
				foreach ($this->em->getRepository(Variant::class)->createQueryBuilder('v')
					         ->select('v.id as id, vt.name as name')
					         ->andWhere('v.isPublished = 1')
					         ->join('v.texts', 'vt', 'WITH', 'vt.lang = :lang')
					         ->setParameter('lang', $locale)->getQuery()->getArrayResult() as $row) {
					$data[$row['id']] = $row;
				}

				return $data;
			});
		}

		return $this->cVariants[$id];
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 */
	public function getVariantValueById($id)
	{
		if (!$this->cVariantValues[$id]) {
			$locale = $this->defaultLang->locale;
			$key    = 'variantValues_' . $locale;

			$this->cVariantValues = $this->cacheService->defaultCache->load($key, function(&$dep) use ($id, $locale) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'variantValues';

				$data = [];
				foreach ($this->em->getRepository(VariantValue::class)->createQueryBuilder('vv')
					         ->select('vv.id as id, vvt.name as name')
					         ->andWhere('vv.isPublished = 1')
					         ->join('vv.texts', 'vvt', 'WITH', 'vvt.lang = :lang')
					         ->orderBy('vv.position')
					         ->setParameter('lang', $locale)->getQuery()->getArrayResult() as $row) {
					$data[$row['id']] = $row;
				}

				return $data;
			});
		}

		return $this->cVariantValues[$id];
	}
}
