<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Presenters;

use EshopCatalog\FrontModule\Components\IProductsFilterFactory;
use EshopCatalog\FrontModule\Components\IProductsListFactory;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Components\ICartAddFormFactory;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use Gallery\FrontModule\Model\Albums;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Paths;

class DefaultPresenter extends BasePresenter
{
	/** @var Category */
	protected $category;

	/** @var Product */
	protected $product;

	/** @var TemplatePage */
	protected $templatePage;

	/** @var Categories */
	protected $categoriesService;

	/** @var Products */
	protected $productsService;

	/** @var TemplatePages */
	protected $templatePagesService;

	/** @var Paths */
	protected $pathsService;

	/** @var Albums */
	protected $albumsService;

	/** @var Tags */
	protected $tagsService;

	/** @var Config */
	protected $config;

	/**
	 * TODO předělat na event
	 * @var callable
	 */
	public $onFilter = [];

	public function __construct(Config $config, Categories $categories, Products $products, TemplatePages $templatePages, Paths $paths, Albums $albums, Tags $tags)
	{
		$this->config               = $config;
		$this->categoriesService    = $categories;
		$this->productsService      = $products;
		$this->templatePagesService = $templatePages;
		$this->pathsService         = $paths;
		$this->albumsService        = $albums;
		$this->tagsService          = $tags;
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault($id)
	{
		$templatePage = $this->templatePagesService->get($this->getActiveNavigation()->componentParams['templateHome']);

		$this->template->pageClass          .= ' page-eshopcatalog-default';
		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->template));
	}

	public function actionProduct($id)
	{
		$this->product = $product = $this->productsService->get($id);

		if (!$product)
			$this->error('eshopCatalog.product.notFound');

		$templatePage = $this->templatePagesService->get($this->getActiveNavigation()->componentParams['templateProduct']);

		$this->productsService->loadAlternative($product, $this->config->get('productDetailAlternativeLimit', 4));
		$this->productsService->loadVariants($product);
		$this->productsService->loadFeatures($product);
		$this->productsService->loadQuantity($product);
		$this->tagsService->loadTagsToProduct($product);

		foreach ($product->alternative as &$p) {
			$this->tagsService->loadTagsToProduct($p);
		}

		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-product product-' . $id;
		$this->template->product            = $product;

		$path = $this->categoriesService->getPath($product->defaultCategory);
		if (isset($path[0]['id'])) {
			$baseNav                            = $this->categoriesService->findNavigation($path[0]['id']);
			$this->template->breadcrumb         = $this->navigationsService->getPath($baseNav);
			$this->template->categoryBreadcrumb = $product->defaultCategory ? $this->categoriesService->getBreadcrumb($product->defaultCategory, $baseNav) : [];
		}

		$this->setTitle($product->getSeo('title') ?: $product->name);
		$this->seoDataFromEntity = $product->getSeo();

		if ($product->getQuantity() <= 0)
			$this->seoDataFromEntity['robots'] = 'noindex';

		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->template));
	}

	public function actionCategory($id)
	{
		$category = $this->categoriesService->get($id);

		if (!$category)
			$this->error('eshopCatalog.category.notFound');

		$templatePage = $this->templatePagesService->get($this->getActiveNavigation()->componentParams['templatePage']);

		$this->category          = $category;
		$this->seoDataFromEntity = $category->getSeo();
		$this->templatePage      = $templatePage;

		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-category';
		$this->template->category           = $category;
		$this->template->categoryBreadcrumb = $this->categoriesService->getBreadcrumb($category, $this->getActiveNavigation());

		$this['list']->setCategory($this->category->getId())
			->setFilterComponent($this['productsFilter']);

		$paginator   = $this['list']->getPaginator();
		$title       = $this->category->getSeo('title') ?: $this->category->name;
		$titleSuffix = null;

		if ($paginator->getPage() > 1) {
			$titleSuffix = $this->translator->translate('eshopCatalogFront.pageXOfY', null, ['x' => $paginator->getPage(),
			                                                                                 'y' => $paginator->getPageCount()]);
			$title       .= ' ' . $titleSuffix;
		}

		$this->setTitle($title);
		$this->template->titleSuffix = $titleSuffix;
		$this->template->page        = $paginator->getPage();
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->template));

		if (!$this->isAjax() && empty($this['list']->getProducts())) {
			$this->seoDataFromEntity['robots'] = 'noindex, nofollow';
		}
	}

	public function actionSearch($q)
	{
		$templatePage       = $this->templatePagesService->get($this->getActiveNavigation()->componentParams['templateResult']);
		$this->templatePage = $templatePage;

		$this['list']->setProductsBySearch($q)
			->setFilterComponent($this['productsFilter']);

		$this->template->q                  = $q;
		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-search';

		$this->setTitle($this->translator->translate('front.search.searchQuery', null, ['q' => $q]));
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->template));

		if (!$this->isAjax() && empty($this['list']->getProducts())) {
			$this->activeNavigation->setSeo('robots', 'noindex, nofollow');
		}
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentCartAddForm(ICartAddFormFactory $factory)
	{
		$control = $factory->create();

		if ($this->product)
			$control->setProduct($this->product);

		return $control;
	}

	protected function createComponentList(IProductsListFactory $factory)
	{
		$control = $factory->create();

		return $control;
	}

	protected function createComponentProductsFilter(IProductsFilterFactory $factory)
	{
		$control = $factory->create();

		return $control;
	}

}
