<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gallery\Model\Entities\Album;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Core\Model\Entities\TSeo;
use Nette\Utils\DateTime;

/**
 * @ORM\Table(name="eshop_catalog__product", indexes={
 *     @ORM\Index(name="published", columns={"is_published", "id"}),
 *     @ORM\Index(name="published_cat", columns={"is_published", "id_category_default", "id"}),
 * })
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "ProductListener"})
 */
class Product
{
	use Identifier;
	use MagicAccessors;
	use TTranslateListener;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", length=1)
	 */
	public $isPublished;

	/**
	 * @var int
	 * @ORM\Column(name="in_stock", type="smallint", length=1)
	 */
	public $inStock;

	/**
	 * @var int
	 * @ORM\Column(name="quantity", type="integer")
	 */
	public $quantity;

	/**
	 * @var double
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $price;

	/**
	 * @var double
	 * @ORM\Column(name="retail_price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $retailPrice;

	/**
	 * @var Manufacturer
	 * @ORM\ManyToOne(targetEntity="Manufacturer", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_manufacturer", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $manufacturer;

	/**
	 * @var ProductExport
	 * @ORM\ManyToOne(targetEntity="ProductExport", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_product_export", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $productExport;

	/**
	 * @var ProductTexts[]
	 *
	 * @ORM\OneToMany(targetEntity="ProductTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 */
	protected $productTexts;

	/**
	 * @var CategoryProduct[]
	 * @ORM\OneToMany(targetEntity="CategoryProduct", mappedBy="product", indexBy="category", cascade={"all"})
	 */
	protected $categoryProducts;

	/**
	 * @ORM\ManyToOne(targetEntity="Category", cascade={"persist"})
     * @ORM\JoinColumn(name="id_category_default", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $idCategoryDefault;

	/**
	 * @var FeatureProduct[]
	 * @ORM\OneToMany(targetEntity="FeatureProduct", mappedBy="product", indexBy="product", cascade={"all"})
	 */
	protected $featureProducts;

	/**
	 * @var ProductVariant[]
	 * @ORM\OneToMany(targetEntity="ProductVariant", mappedBy="product", indexBy="product", cascade={"all"})
	 */
	protected $productVariants;

	/**
	 * @ORM\OneToMany(targetEntity="ProductTag", mappedBy="product", indexBy="product", cascade={"all"})
	 */
	protected $productTags;

	/**
	 * @var ProductSupplier
	 *
	 * @ORM\OneToMany(targetEntity="ProductSupplier", mappedBy="product", indexBy="id_supplier", cascade={"all"})
	 */
	protected $suppliers;

	/**
	 * @ORM\ManyToOne(targetEntity="VatRate")
	 * @ORM\JoinColumn(name="id_vat_rate", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $vatRate;

	/**
	 * @var string
	 * @ORM\Column(name="ean", type="string", length=20, nullable=true)
	 */
	public $ean;

	/**
	 * @var string
	 * @ORM\Column(name="code1", type="string", length=60, nullable=true)
	 */
	public $code1;

	/**
	 * @var string
	 * @ORM\Column(name="code2", type="string", length=60, nullable=true)
	 */
	public $code2;

	/**
	 * @var Album
	 *
	 * @ORM\OneToOne(targetEntity="Gallery\Model\Entities\Album", cascade={"persist"})
	 * @ORM\JoinColumn(name="gallery_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $gallery;

	/**
	 * @var DateTime
	 * @ORM\Column(name="created", type="datetime")
	 */
	private $created;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(name="modified", type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * @var array
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected $moreData = [];

	public function __construct()
	{
		$this->productTexts     = new ArrayCollection();
		$this->categoryProducts = new ArrayCollection();
		$this->featureProducts  = new ArrayCollection();
		$this->productVariants  = new ArrayCollection();
		$this->suppliers        = new ArrayCollection();
		$this->productTags      = new ArrayCollection();
		$this->quantity         = 0;
		$this->inStock          = 0;
		$this->moreData         = [];
		$this->created          = new DateTime('now');
	}

	public function addProductText($lang)
	{
		$this->productTexts->set($lang, new ProductTexts($this, $lang));
	}

	public function setProductText(ProductTexts $productTexts)
	{
		$this->productTexts->set($productTexts->getLang(), $productTexts);
	}

	public function getProductText($lang = null)
	{
		return $this->productTexts->get($lang ?: $this->locale) ?? null;
	}

	public function removeAllFeatureProduct()
	{
		$this->featureProducts->clear();
	}

	public function addFeatureProduct(FeatureProduct $featureProduct)
	{
		$this->featureProducts->add($featureProduct);
	}

	public function getFeatureProducts()
	{
		return $this->featureProducts->getValues();
	}

	public function removeAllProductVariants()
	{
		$this->productVariants->clear();
	}

	public function addProductVariant(ProductVariant $productVariant)
	{
		$this->productVariants->add($productVariant);
	}

	public function getProductVariants()
	{
		return $this->productVariants->getValues();
	}

	/**
	 * @param CategoryProduct $categoryProduct
	 *
	 * @return $this
	 */
	public function addCategoryProduct(CategoryProduct $categoryProduct)
	{
		if (!$this->categoryProducts->contains($categoryProduct))
			$this->categoryProducts->add($categoryProduct);

		return $this;
	}

	public function removeCategoryProduct(CategoryProduct $categoryProduct)
	{
		$this->categoryProducts->removeElement($categoryProduct);
	}

	public function getCategoryProducts()
	{
		return $this->categoryProducts;
	}

	public function removeAllCategoryProducts()
	{
		$this->categoryProducts->clear();
	}

	public function addProductTag(ProductTag $productTag)
	{
		$this->productTags->add($productTag);
	}

	public function getProductTags()
	{
		return $this->productTags;
	}

	public function removeAllProductTags()
	{
		$this->productTags->clear();
	}

	public function setManufacturer(Manufacturer $manufacturer)
	{
		$this->manufacturer = $manufacturer;
	}

	public function getManufacturer()
	{
		return $this->manufacturer;
	}

	public function setDefaultCategory(Category $category)
	{
		$this->idCategoryDefault = $category;
	}

	public function getDefaultCategory()
	{
		return $this->idCategoryDefault;
	}

	public function setVateRate(VatRate $vatRate)
	{
		$this->vatRate = $vatRate;
	}

	public function getVateRate()
	{
		return $this->vatRate;
	}

	/*************************************
	 * == ProductExport
	 */

	public function setProductExport(ProductExport $productExport)
	{
		$this->productExport = $productExport;
	}

	public function getProductExport()
	{
		return $this->productExport;
	}

	/*************************************
	 * == Supplier
	 */

	public function getSupplier($id)
	{
		return $this->suppliers->get($id);
	}

	public function getSuppliers()
	{
		return $this->suppliers;
	}

	public function setSupplier(ProductSupplier $ps)
	{
		if (!$this->suppliers->containsKey($ps->getSupplier()->getId())) {
			$this->suppliers->clear();
			$this->suppliers->add($ps);
		}
	}

	/*************************************
	 * == Dates
	 */

	public function getCreated() { return $this->created; }

	public function getModified() { return $this->modified; }

	public function validateCreated()
	{
		if ($this->created->format('y-m-d') <= 0)
			$this->created = new DateTime();
	}

	/*************************************
	 * == MoreData
	 */

	/**
	 * @param string $key
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue($key) { return $this->moreData[$key] ?? null; }

	/**
	 * @param string $key
	 * @param string $value
	 *
	 * @return $this
	 */
	public function setMoreDataValue($key, $value)
	{
		if (!is_array($this->moreData))
			$this->moreData = [];

		$this->moreData[$key] = $value;

		return $this;
	}

	/**
	 * @param array $data
	 *
	 * @return $this
	 */
	public function setMoreData($data)
	{
		$this->moreData = $data;

		return $this;
	}


}
