<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\FrontModule\Model\SeoContainer;
use EshopCatalog\AdminModule\Model\Helpers\ProductFormHelper;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductExport;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\AdminModule\Model\ProductTags;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantCombination;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\Supplier;
use EshopCatalog\Model\Entities\Tag;
use EshopCatalog\Model\Entities\Variant;
use EshopCatalog\Model\Entities\VariantValue;
use EshopCatalog\Model\Entities\VariantValueText;
use EshopCatalog\Model\Helpers\ExportEnums;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\Model\Entities\Album;
use Import\DI\ImportExtension;
use Nette\Application\UI\Form;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Http\Session;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureProducts;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\AdminModule\Model\Variants;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\AdminModule\Model\ProductVariants;
use EshopCatalog\AdminModule\Model\ProductVariantCombinations;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\FeatureProduct;

class ProductForm extends BaseControl
{
	/** @var Product */
	public $product;

	/** @var Products */
	protected $productServices;

	/** @var ProductVariants */
	protected $productVariantServices;

	/** @var ProductVariantCombinations */
	protected $productVariantCombinationServices;

	/** @var FeatureProducts */
	protected $featureProductServices;

	/** @var CategoryProducts */
	protected $categoryProductServices;

	/** @var Categories */
	protected $categoryServices;

	/** @var CategoryContainer */
	protected $categoryContainerService;

	/** @var Manufacturers */
	protected $manufacturerServices;

	/** @var Suppliers */
	protected $supplierServices;

	/** @var VatRates */
	protected $vatRateServices;

	/** @var Features */
	protected $featureServices;

	/** @var Variants */
	protected $variantsService;

	/** @var SeoContainer */
	protected $seoContainerService;

	/** @var IImagesZoneFactory */
	protected $imagesZoneFactory;

	/** @var Request */
	protected $httpRequest;

	/** @var Session */
	protected $session;

	/** @var TagsContainer */
	protected $tagsContainerService;

	/** @var ProductTags */
	protected $productTagsServices;

	/** @var ProductFormHelper */
	protected $helper;

	/** @var Session */
	private $sessionProduct;

	/** @var array */
	private $features;

	/** @var array */
	private $variants;

	public function __construct(Products $products, Features $features, Variants $variants, FeatureProducts $featureProducts, Categories $categories, CategoryProducts $categoryProducts, Manufacturers $manufacturers, Suppliers $suppliers, VatRates $vatRates, SeoContainer $seoContainer, IImagesZoneFactory $imagesZoneFactory, ProductVariants $productVariants, ProductVariantCombinations $productVariantCombinations, Request $request, Session $session, CategoryContainer $categoryContainer, TagsContainer $tagsContainer, ProductTags $productTags, ProductFormHelper $helper)
	{
		$this->productServices                   = $products;
		$this->featureServices                   = $features;
		$this->variantsService                   = $variants;
		$this->featureProductServices            = $featureProducts;
		$this->categoryContainerService          = $categoryContainer;
		$this->categoryServices                  = $categories;
		$this->categoryProductServices           = $categoryProducts;
		$this->manufacturerServices              = $manufacturers;
		$this->supplierServices                  = $suppliers;
		$this->seoContainerService               = $seoContainer;
		$this->vatRateServices                   = $vatRates;
		$this->imagesZoneFactory                 = $imagesZoneFactory;
		$this->httpRequest                       = $request;
		$this->productVariantServices            = $productVariants;
		$this->productVariantCombinationServices = $productVariantCombinations;
		$this->session                           = $session;
		$this->tagsContainerService              = $tagsContainer;
		$this->productTagsServices               = $productTags;
		$this->helper                            = $helper;
	}

	public function render()
	{
		$this->template->thisForm                    = $this['form'];
		$this->template->product                     = $this->product;
		$this->template->productFeatures             = $this->getSessionProduct()->productFeatures;
		$this->template->productVariants             = $this->getSessionProduct()->productVariants;
		$this->template->productVariantsCombinations = $this->getSessionProduct()->productVariantsCombinations;
		$this->template->groupCustomers              = $this->helper->getGroupCustomers();
		$this->template->enablePriceLevel            = $this->helper->config->get('enablePriceLevels', false);

		$this->eventDispatcher->dispatch(ProductForm::class . '::render',
			new ComponentTemplateEvent($this->template, $this));
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ========================  HANDLE
	 */

	public function handleLoadInputFeatureValue($featureType)
	{
		$presenter = $this->getPresenter();

		try {
			$this->loadComponentValueFeature($featureType);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('componentFeatureValue');
		$presenter->redrawControl('flashes');
	}

	public function handleLoadInputVariantValue($variantType)
	{
		$presenter = $this->getPresenter();

		try {
			$this->loadComponentValueVariant($variantType);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('componentVariantValue');
		$presenter->redrawControl('flashes');
	}

	public function handleAddFeature($feature, $featureValue)
	{
		$presenter = $this->getPresenter();
		if ($feature == '' || $featureValue == '')
			return;

		try {
			$this->addProductFeature($feature, $featureValue);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('componentProductFeatures');
		$presenter->redrawControl('flashes');
	}

	public function handleAddVariant($variant, $variantValue)
	{
		$presenter = $this->getPresenter();
		if ($variant == '' || $variantValue == '')
			return;

		try {
			$this->addProductVariant($variant, $variantValue);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('componentProductVariants');
		$presenter->redrawControl('flashes');
	}

	public function handleAddProductVariant($ean, $code1, $code2, $quantity, $priceImpact)
	{
		$presenter = $this->getPresenter();
		if (count($this->getSessionProduct()->productVariants) == 0)
			return;

		try {
			$this->addProductVariant2($ean, $code1, $code2, $quantity, $priceImpact);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('componentProductVariants');
		$this->redrawControl('componentProductVariantes');
		$presenter->redrawControl('flashes');
	}

	public function handleDeleteFeature($feature)
	{
		$presenter = $this->getPresenter();

		try {
			unset($this->getSessionProduct()->productFeatures[$feature]);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('componentProductFeatures');
		$presenter->redrawControl('flashes');
	}

	public function handleDeleteVariant($variant)
	{
		$presenter = $this->getPresenter();

		try {
			unset($this->getSessionProduct()->productVariants[$variant]);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('componentProductVariants');
		$presenter->redrawControl('flashes');
	}

	public function handleDeleteVariantCombination($variant)
	{
		$presenter = $this->getPresenter();

		try {
			unset($this->getSessionProduct()->productVariantsCombinations[$variant]);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('componentProductVariantes');
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ========================  COMPONENT
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addBool('isPublished', 'eshopCatalog.productForm.isPublished')->setDefaultValue(0);
		$form->addText('name', 'eshopCatalog.productForm.name')->setMaxLength(255)->setIsMultilanguage();
		if (Config::load('enableProductName2'))
			$form->addText('name2', 'eshopCatalog.productForm.name2')->setMaxLength(255)
				->setIsMultilanguage()->setNullable();
		$form->addText('ean', 'eshopCatalog.productForm.ean')->setMaxLength(13);
		$form->addText('code1', 'eshopCatalog.productForm.code1')->setMaxLength(60);
		$form->addText('code2', 'eshopCatalog.productForm.code2')->setMaxLength(60);
		$form->addInteger('quantity', 'eshopCatalog.productForm.quantity');
		if (Config::load('pseudoWarehouse'))
			$form->addBool('unlimitedQuantity', 'eshopCatalog.productForm.unlimitedQuantity')->setDefaultValue(0);
		if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
			$form->addBool('categoryGiftsAllowed', 'eshopCatalog.productForm.categoryGiftsAllowed')->setDefaultValue(1);
		$form->addText('price', 'eshopCatalog.productForm.price')->setRequired()->addRule(Form::FLOAT, 'Cena musí být číslo');
		$form->addText('retailPrice', 'eshopCatalog.productForm.retailPrice')->addRule(Form::FLOAT, 'Cena musí být číslo');
		$form->addEditor('shortDesc', 'eshopCatalog.productForm.shortDesc')->setToolbar('Text')->setDisableAutoP()->setHeight(100)
			->setMaxLength(250)->setIsMultilanguage();
		$form->addEditor('desc', 'eshopCatalog.productForm.desc')->setHeight(300)->setIsMultilanguage();

		if (class_exists(ImportExtension::class)) {
			$form->addBool('stopImportPrice', 'eshopCatalog.productForm.stopImportPrice')->setDefaultValue(1);
		}

		$form->addBool('discountDisabled', 'eshopCatalog.productForm.discountDisabled')->setDefaultValue(1);

		//Manufacturers - Vyrobci
		$manufacturers = ['' => 'eshopCatalog.productForm.choiceManufacturer'];
		foreach ($this->manufacturerServices->getAll() as $ms)
			$manufacturers[$ms->getId()] = $ms->name;
		$form->addSelect('manufacturer', 'eshopCatalog.productForm.manufacturer', $manufacturers);

		//Suppliers - Dodavatele
		// Aktuálně pouze zobrazení z importu

		//VatRates - DPH
		$vatRates = [];
		foreach ($this->vatRateServices->getAll() as $vrs)
			$vatRates[$vrs->getId()] = $vrs->name;
		$form->addSelect('vatRate', 'eshopCatalog.productForm.vatRate', $vatRates)->setPrompt('eshopCatalog.productForm.choiceVatRate')
			->setRequired();

		//Exporty
		$placeholders = [];
		if ($this->product && $this->product->getDefaultCategory() && $this->product->getDefaultCategory()->exports) {
			foreach ($this->product->getDefaultCategory()->exports->toArray() as $service => $vals) {
				$placeholders['p_' . $service]['categoryText'][$vals->lang] = $vals->categoryText;
			}

			foreach ($placeholders as $service => $vals) {
				if (!isset($form[$service]))
					continue;

				$form[$service]['categoryText']->setPlaceholder($vals['categoryText']);
			}
		}

		foreach (ExportEnums::$services as $k => $v) {
			$k         = 'p_' . $k;
			$container = new BaseContainer();

			$container->addBool('status', 'eshopCatalog.porovnavace.status', ExportEnums::$statuses)
				->setDefaultValue(2)
				->setIsMultilanguage();

			foreach ($v['fields'] as $f) {
				$container->addText($f, 'eshopCatalog.porovnavace.' . substr($k, 2) . '.' . $f)
					->setMaxLength(255)
					->setIsMultilanguage();
			}

			if (isset($container['categoryText']) && isset($placeholders[$k]['categoryText']))
				$container['categoryText']->setPlaceholder($placeholders[$k]['categoryText']);

			$form->addComponent($container, $k);
		}

		//PriceLevel
		if ($this->helper->config->get('enablePriceLevels', false)) {
			foreach ($this->helper->getGroupCustomers() as $groupCustomers) {
				$form->addText('pricelevel' . $groupCustomers->getId(), $groupCustomers->name)
					->addRule(Form::FLOAT, 'eshopCatalog.productForm.priceMustBeNumber');
			}
		}

		$form->addComponent($this->categoryContainerService->getContainer(), 'category');

		//Features - Vlastnosti
		$form->addSelect('typeFeature', 'eshopCatalog.productForm.typeFeature', $this->getFeatures())->setOmitted();

		//Variants - Varianty
		$form->addSelect('typeVariant', 'eshopCatalog.productForm.typeVariant', $this->getVariants())->setOmitted();

		$form->addText('eanAtt', 'eshopCatalog.productForm.ean')->setMaxLength(13);
		$form->addText('code1Att', 'eshopCatalog.productForm.code1')->setMaxLength(60);
		$form->addText('code2Att', 'eshopCatalog.productForm.code2')->setMaxLength(60);
		$form->addInteger('quantityAtt', 'eshopCatalog.productForm.quantity');
		$form->addText('unitPriceImpactAtt', 'eshopCatalog.productForm.unitPrice')
			->addRule(Form::FLOAT, 'eshopCatalog.productForm.priceMustBeNumber');

		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$form->addComponent($this->tagsContainerService->getContainer(), 'tags');

		$form->addHidden('preparedAlbumId');

		$this->eventDispatcher->dispatch(ProductForm::class . '::createForm',
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null));

		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		$nameFilled = false;
		foreach ($values->name as $v)
			if ($v)
				$nameFilled = true;

		if (!$nameFilled)
			$form->addError('eshopCatalog.productForm.nameMissing');

		if ($form->hasErrors())
			$this->redrawControl('formErrors');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$this->em->beginTransaction();
		try {
			$langValues = $form->convertMultilangValuesToArray();
			$product    = $this->product ?: new Product();

			$product->quantity         = $values->quantity;
			$product->price            = $values->price;
			$product->retailPrice      = $values->retailPrice ?: null;
			$product->isPublished      = $values->isPublished;
			$product->inStock          = $values->quantity > 0 ? 1 : 0;
			$product->ean              = $values->ean;
			$product->code1            = $values->code1;
			$product->code2            = $values->code2;
			$product->discountDisabled = $values->discountDisabled == 1 ? 0 : 1;

			if (Config::load('pseudoWarehouse'))
				$product->unlimitedQuantity = $values->unlimitedQuantity;
			if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
				$product->setCategoryGiftsAllowed($values->categoryGiftsAllowed);

			$this->em->persist($product);
			$this->em->flush();

			if ($values->preparedAlbumId)
				$product->gallery = $this->em->getRepository(Album::class)->getReference($values->preparedAlbumId);

			//vyrobce
			if ($values->manufacturer) {
				$manufacturer = $this->manufacturerServices->get($values->manufacturer);
				$product->setManufacturer($manufacturer);
			}

			if ($values->supplier) {
				$supplier        = $this->em->getReference(Supplier::class, $values->supplier);
				$productSupplier = new ProductSupplier($this->product, $supplier);
				//TODO dořešit více dodavatelů
				//$this->product->setSupplier($productSupplier);
			}

			//vatRate
			if ($values->vatRate) {
				$vatRate = $this->vatRateServices->get($values->vatRate);
				$product->setVateRate($vatRate);
			}

			// Texty
			/** @var ProductTexts[] $productTexts */
			$productTexts = $product->getProductTexts()->toArray();
			$seoData      = $this->seoContainerService->getFormData($values->seo, true);

			foreach ($langValues as $l => $v) {
				if ($v['name'] == '') {
					if (isset($productTexts[$l])) {
						$this->em->remove($productTexts[$l]);
						unset($productTexts[$l]);
					}
					continue;
				}

				if (!$productTexts[$l])
					$productTexts[$l] = new ProductTexts($product, $l);

				$productTexts[$l]->setName($v['name'])
					->setDescription($v['desc'])
					->setShortDescription($v['shortDesc'])
					->setSeo($seoData[$l] ?? []);

				if (Config::load('enableProductName2'))
					$productTexts[$l]->setName2($v['name2']);

				$this->em->persist($productTexts[$l]);
			}

			$product->setProductTexts($productTexts);

			//nejdrive vymazani
			if ($this->product) {
				$productVariants = $this->product->getProductVariants();
				foreach ($productVariants as $pa) {
					foreach ($pa->getProductCombinations() as $pv) {
						$this->productVariantCombinationServices->em->remove($pv);
					}
					$pa->clearCombinations();
					$this->productVariantServices->em->remove($pa);
				}
				$product->removeAllProductVariants();
			}

			$this->em->persist($product);
			$this->em->flush();

			//Vlastnosti
			$currentFeatureValuesId = [];
			$formFeatureValuesId    = [];
			foreach ($product->getFeatureProducts() as $fp)
				$currentFeatureValuesId[(int) $fp->featureValue->getId()] = (int) $fp->feature->getId();
			foreach ($this->getSessionProduct()->productFeatures as $k => $fp)
				$formFeatureValuesId[(int) $fp['idValue']] = $k;

			// Přidat vlastnost
			foreach (array_diff_key($formFeatureValuesId, $currentFeatureValuesId) as $k => $v) {
				$v  = $formFeatureValuesId[$k];
				$fp = new FeatureProduct($product,
					$this->em->getReference(Feature::class, $v), $this->em->getReference(FeatureValue::class, $k));
				$this->em->persist($fp);
			}

			// Odstranit vlasnost
			$removeFeatures = array_diff_key($currentFeatureValuesId, $formFeatureValuesId);
			if (!empty($removeFeatures)) {
				$this->em->createQueryBuilder()->delete(FeatureProduct::getClassName(), 'fp')
					->where('fp.product = :product')
					->andWhere('fp.featureValue IN (:values)')
					->setParameters([
						'product' => $product->getId(),
						'values'  => array_keys($removeFeatures),
					])->getQuery()->execute();
			}

			//PriceLevels
			if ($this->helper->config->get('enablePriceLevels', false)) {
				foreach ($this->helper->getGroupCustomers() as $gc) {
					if (isset($values['pricelevel' . $gc->getId()])) {
						if ($values['pricelevel' . $gc->getId()] != '') {
							$ppl        = $product->getPriceLevels()->get($gc->getId()) ?: new ProductPriceLevel($product, $gc);
							$ppl->price = $values['pricelevel' . $gc->getId()];
							$this->em->persist($ppl);
							if (!$product->getPriceLevels()->containsKey($gc->getId()))
								$product->getPriceLevels()->add($ppl);
						} else {
							$ppl = $product->getPriceLevels()->get($gc->getId());
							if ($ppl) {
								$product->getPriceLevels()->remove($gc->getId());
								$this->em->remove($ppl);
							}
						}
					}
				}
			}

			// Exporty
			$exports     = [];
			$exportsVals = [];
			foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('ppe')
				         ->where('ppe.id = :id')->setParameter('id', $product->getId())
				         ->getQuery()->getResult() as $row)
				$exports[$row->service][$row->lang] = $row;

			foreach (ExportEnums::$services as $serviceName => $v) {
				foreach ($langValues as $lang => $vals) {
					if (isset($vals['p_' . $serviceName]))
						$exportsVals[$lang][$serviceName] = $vals['p_' . $serviceName];
				}
			}

			foreach ($exportsVals as $lang => $services) {
				foreach ($services as $serviceName => $vals) {
					if (isset($exports[$serviceName][$lang]))
						$export = $exports[$serviceName][$lang];
					else
						$export = new ProductExport($product, $lang, $serviceName);

					$export->status       = (int) $vals['status'];
					$export->product      = $vals['product'] ?: null;
					$export->productName  = $vals['productName'] ?: null;
					$export->categoryText = $vals['categoryText'] ?: null;
					$export->bidCpc       = $vals['bidCpc'] ?: null;

					$this->em->persist($export);
				}
			}

			//Varianty
			foreach ($this->getSessionProduct()->productVariantsCombinations as $key => $productVariant) {
				$pa                  = new ProductVariant($product);
				$pa->ean             = $productVariant['ean'];
				$pa->code1           = $productVariant['code1'];
				$pa->code2           = $productVariant['code2'];
				$pa->quantity        = $productVariant['quantity'] ?: 0;
				$pa->unitPriceImpact = $productVariant['priceImpact'] ?: null;
				$pa->isDefault       = 0;
				foreach ($productVariant['variants'] as $k => $pv) {
					$a   = $this->em->getRepository(Variant::class)->find($k);
					$av  = $this->em->getRepository(VariantValue::class)->find($pv['idValue']);
					$pac = new ProductVariantCombination($pa, $av, $a);
					$pa->addProductCombination($pac);
				}
				$product->addProductVariant($pa);
			}

			//Categorie
			$this->categoryContainerService->saveData([$product->getId()], (array) $values->category);

			//Tags
			$tagsForRemove = [];
			foreach ($product->getProductTags()->toArray() as $tag)
				$tagsForRemove[$tag->tag->type] = $tag;

			foreach ($values->tags as $tagType => $vals) {
				if (!$vals->active)
					continue;

				if (!isset($tagsForRemove[$tagType])) {
					$productTag = new ProductTag($product, $this->em->getReference(Tag::class, $vals['id']));
					$product->addProductTag($productTag);
				} else
					$productTag = $tagsForRemove[$tagType];

				unset($tagsForRemove[$tagType]);
				$productTag->validFrom = $vals['dateFrom'];
				$productTag->validTo   = $vals['dateTo'];

				$this->em->persist($productTag);
			}

			foreach ($tagsForRemove as $tagType => $tag) {
				$product->removeProductTag($tag);
				$this->em->remove($tag);
			}

			if (isset($values->stopImportPrice)) {
				$values->stopImportPrice = $values->stopImportPrice == 1 ? 0 : 1;

				$product->setMoreDataValue('stopImportPrice', $values->stopImportPrice);
			}

			$this->em->persist($product);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $product;
			$this->eventDispatcher->dispatch(ProductForm::class . '::formSuccess', $event);

			$this->em->flush();
			$this->em->commit();

			$form->addCustomData('productId', $product->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.productForm.productSaved');
			$this->getPresenter()->redrawControl('flashes');
			$this->cleanSessionProduct();
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			$form->addError($e->getMessage());
			$this->getPresenter()->redrawControl('flashes');
			$this->redrawControl('formErrors');

			return false;
		}

		return true;
	}

	protected function createComponentImagesZone()
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->product && $this->product->gallery)
			$control->setAlbum($this->product->gallery->getId());

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */
			$dataRaw = $this->httpRequest->getPost('formData');
			$data    = [];

			if (!$dataRaw)
				return false;

			foreach ($dataRaw as $row) {
				$data[$row['name']] = $row['value'];
			}

			if (!$this->product || !$this->product->gallery) {
				$album = new Album(UPLOADS_PATH . '/products');
				$this->em->persist($album)->flush();
				$control->setAlbum($album->getId());
			}

			if ($this->product) {
				$this->product->gallery = $album;
				$this->em->persist($this->product)->flush();
			}
		};

		return $control;
	}

	/*******************************************************************************************************************
	 * ========================  GET / SET
	 */

	protected function getSessionProduct()
	{
		return $this->session->getSection('productForm_' . ($this->product ? $this->product->getId() : 'new'));
	}

	public function cleanSessionProduct()
	{
		$this->getSessionProduct()->remove();
	}

	protected function loadComponentValueFeature($featureType)
	{
		$form = $this['form'];

		if ($form->getComponent('valueFeature', false)) {
			$form->removeComponent($form['valueFeature']);
		}

		$form->addSelect('valueFeature', 'eshopCatalog.productForm.valueFeature', $this->getFeatureValues($featureType))->setOmitted();
	}

	protected function loadComponentValueVariant($variantType)
	{
		$form = $this['form'];

		if ($form->getComponent('valueVariant', false)) {
			$form->removeComponent($form['valueVariant']);
		}

		$form->addSelect('valueVariant', 'eshopCatalog.productForm.valueVariant', $this->getVariantValues($variantType))->setOmitted();
	}

	protected function addProductFeature($feature, $featureValue)
	{
		$productFeature['typName']                            = $this->features[$feature];
		$productFeature['idValue']                            = $featureValue;
		$productFeature['valueName']                          = $this->getNameFeatureValue($featureValue)->name;
		$this->getSessionProduct()->productFeatures[$feature] = $productFeature;
	}

	protected function addProductVariant($variant, $variantValue)
	{
		$productVariant['typName']                            = $this->getVariants()[$variant];
		$productVariant['idValue']                            = $variantValue;
		$productVariant['valueName']                          = $this->getNameVariantValue($variantValue)->name;
		$this->getSessionProduct()->productVariants[$variant] = $productVariant;
	}

	protected function addProductVariant2($ean, $code1, $code2, $quantity, $priceImpact)
	{
		$productVariant['ean']         = $ean;
		$productVariant['code1']       = $code1;
		$productVariant['code2']       = $code2;
		$productVariant['quantity']    = $quantity;
		$productVariant['priceImpact'] = $priceImpact;
		$productVariant['variants']    = $this->getSessionProduct()->productVariants;
		$arr                           = [];
		foreach ($this->getSessionProduct()->productVariants as $pa)
			$arr[] = $pa['idValue'];
		$key = implode('-', $arr);

		$this->getSessionProduct()->productVariantsCombinations[$key] = $productVariant;

		unset($this->getSessionProduct()->productVariants);
	}

	public function setProduct($id)
	{
		$this->product = $this->em->getRepository(Product::class)->createQueryBuilder('p')
			->addSelect('pt, m, cp, cd, fp, pTags, pTagsTag, s, vr, fpf, fpv, fpft, fpvt')
			->leftJoin('p.productTexts', 'pt')
			->leftJoin('p.manufacturer', 'm')
			->leftJoin('p.categoryProducts', 'cp')
			->leftJoin('p.idCategoryDefault', 'cd')
			->leftJoin('p.featureProducts', 'fp')
			->leftJoin('p.productTags', 'pTags')
			->leftJoin('pTags.tag', 'pTagsTag')
			->leftJoin('p.suppliers', 's')
			->leftJoin('p.vatRate', 'vr')
			->leftJoin('fp.feature', 'fpf')
			->leftJoin('fpf.featureTexts', ' fpft')
			->leftJoin('fp.featureValue', 'fpv')
			->leftJoin('fpv.featureValueTexts', 'fpvt')
			->where('p.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();

		if ($this->product) {
			/** @var Product $prod */
			$prod = $this->product;
			$form = $this['form'];
			$form->setDefaults([
				'isPublished'      => $prod->isPublished,
				'ean'              => $prod->ean,
				'quantity'         => $prod->quantity,
				'price'            => $prod->price,
				'retailPrice'      => $prod->retailPrice,
				'code1'            => $prod->code1,
				'code2'            => $prod->code2,
				'discountDisabled' => $prod->discountDisabled == 1 ? 0 : 1,
			]);

			if (Config::load('pseudoWarehouse'))
				$form['unlimitedQuantity']->setDefaultValue($prod->unlimitedQuantity);
			if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
				$form['categoryGiftsAllowed']->setDefaultValue($prod->isCategoryGiftsAllowed());

			if (class_exists(ImportExtension::class)) {
				$stopImportPrice = $prod->getMoreDataValue('stopImportPrice') == 1 ? 0 : 1;
				$form['stopImportPrice']->setValue($stopImportPrice);
			}

			// Manufacturer
			if ($prod->getManufacturer() && array_key_exists($prod->getManufacturer()->getId(), $form['manufacturer']->getItems()))
				$form['manufacturer']->setDefaultValue($prod->getManufacturer()->getId());

			//Suppliers
			$this->onAnchor[] = function() use($prod) {
				$this->template->suppliers = [];
				foreach ($prod->getSuppliers()->toArray() as $productSupplier) {
					/** @var ProductSupplier $productSupplier */
					$this->template->suppliers[$productSupplier->getSupplier()->getId()] = [
						'name'     => $productSupplier->getSupplier()->name,
						'quantity' => $productSupplier->quantity,
					];
				}
			};

			// Texty
			$mTexts  = [];
			$seoData = [];
			foreach ($prod->getProductTexts()->toArray() as $lang => $texts) {
				foreach (['name' => 'name', 'shortDesc' => 'shortDescription', 'desc' => 'description'] as $k => $v) {
					$mTexts[$k][$lang] = $texts->$v;
				}

				if (Config::load('enableProductName2'))
					$mTexts['name2'][$lang] = $texts->name2;

				foreach ($texts->getSeo() as $k => $v)
					$seoData[$k][$lang] = $v;
			}
			$form->setDefaults($mTexts);
			$this->seoContainerService->setDefaults($form['seo'], $seoData);

			if ($prod->vatRate && array_key_exists($prod->vatRate->getId(), $form['vatRate']->getItems()))
				$form['vatRate']->setDefaultValue($prod->vatRate->getId());

			//set hodnoty vlastnosti
			$featureProducts = $prod->getFeatureProducts();

			// Odstranění prázdných hodnot
			$removed = false;
			foreach ($featureProducts as $fp) {
				/** @var FeatureProduct $fp */
				if (empty($fp->getFeatureValue()->getFeatureValueTexts()->toArray()) || empty($fp->getFeature()->getFeatureTexts()->toArray())) {
					$this->em->remove($fp);
					$removed = true;
				}
			}

			if ($removed)
				$this->em->flush();

			if (!isset($this->getSessionProduct()->productFeatures))
				foreach ($featureProducts as $fp) {
					$this->addProductFeature($fp->getIdFeature(), $fp->getIdFeatureValue());
				}

			foreach ($this->getSessionProduct()->productFeatures as $k => $fp) {
				if ($fp['typName'] == null || $fp['valueName'] == null)
					unset($this->getSessionProduct()->productFeatures[$k]);
			}

			//set hodnoty variant
			$productVariants = $prod->getProductVariants();
			if (!isset($this->getSessionProduct()->productVariantsCombinations))
				foreach ($productVariants as $pa) {
					$arr = [];
					foreach ($pa->getProductCombinations() as $pv) {
						$this->addProductVariant($pv->getIdVariant(), $pv->getIdValue());
						$arr[] = $pv->getIdValue();
					}
					$this->addProductVariant2($pa->ean, $pa->code1, $pa->code2, $pa->quantity, $pa->unitPriceImpact);
				}


			//set kategorie
			$cat = [];
			foreach ($prod->getCategoryProducts()->toArray() as $cp)
				$cat['category'][] = $cp->getIdCategory();
			if ($prod->getDefaultCategory())
				$cat['defaultCategory'] = $prod->getDefaultCategory()->getId();

			$this->categoryContainerService->setDefaults($form['category'], $cat);

			//set tags
			$tags = [];
			foreach ($prod->getProductTags()->toArray() as $t) {
				$tags[$t->getTag()->type] = [
					'active'   => 1,
					'dateFrom' => $t->validFrom,
					'dateTo'   => $t->validTo,
					'id'       => $t->getTag()->getId(),
				];
			}
			$this->tagsContainerService->setDefaults($form['tags'], $tags);

			// priceLevel
			if ($this->helper->config->get('enablePriceLevels', false)) {
				foreach ($this->helper->getGroupCustomers() as $gc) {
					if (isset($form['pricelevel' . $gc->getId()]) && $prod->getPriceLevels()->containsKey($gc->getId())) {
						$form['pricelevel' . $gc->getId()]->setDefaultValue($prod->getPriceLevels()->get($gc->getId())->price);
					}
				}
			}

			// Exporty
			$exports = [];
			foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('ppe')
				         ->where('ppe.id = :id')->setParameter('id', $prod->getId())
				         ->getQuery()->getArrayResult() as $row) {
				foreach ($row as $k => $v) {
					if (in_array($k, ['service', 'lang', 'id']))
						continue;

					if (isset($form['p_' . $row['service']][$k]))
						$exports['p_' . $row['service']][$k][$row['lang']] = $v;
				}
			}
			$form->setDefaults($exports);

			$this->eventDispatcher->dispatch(ProductForm::class . '::setProduct', new SetFormDataEvent($form, $this->product));
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	/**
	 * @return array
	 */
	private function getFeatures()
	{
		if (!$this->features) {
			$features = [null => ''];
			foreach ($this->featureServices->getEr()->createQueryBuilder('f')->select('f.id, ft.name')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->where('f.isPublished = 1')
				         ->orderBy('f.position')
				         ->getQuery()->getResult() as $v)
				$features[$v['id']] = $v['name'];

			$this->features = $features;
		}

		return $this->features;
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 */
	private function getFeatureValues($id)
	{
		$featureValues = [null => ''];
		foreach ($this->featureServices->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')->select('fv.id, fvt.name')
			         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')->setParameter('lang', $this->translator->getLocale())
			         ->andWhere('fv.feature = :id')->setParameter('id', $id)
			         ->andWhere('fv.isPublished = 1')
			         ->orderBy('fv.position')->getQuery()->getResult() as $v)
			$featureValues[$v['id']] = $v['name'];

		return $featureValues;
	}

	/**
	 * @param int $id
	 *
	 * @return FeatureValueTexts|null
	 */
	private function getNameFeatureValue($id)
	{
		return $this->featureServices->em->getRepository(FeatureValueTexts::class)->findOneBy([
			'id'   => $id,
			'lang' => $this->translator->getLocale(),
		]);
	}

	/**
	 * @return array
	 */
	private function getVariants()
	{
		if (!$this->variants) {
			$variants = [null => ''];
			foreach ($this->variantsService->getEr()->createQueryBuilder('v')->select('v.id, vt.name')
				         ->join('v.texts', 'vt', 'WITH', 'vt.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->where('v.isPublished = 1')
				         ->orderBy('v.position')
				         ->getQuery()->getResult() as $v)
				$variants[$v['id']] = $v['name'];

			$this->variants = $variants;
		}

		return $this->variants;
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 */
	private function getVariantValues($id)
	{
		$variantValues = [null => ''];
		foreach ($this->variantsService->em->getRepository(VariantValue::class)->createQueryBuilder('vv')->select('vv.id, vvt.name')
			         ->join('vv.texts', 'vvt', 'WITH', 'vvt.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->andWhere('vv.variant = :id')->setParameter('id', $id)
			         ->andWhere('vv.isPublished = 1')
			         ->orderBy('vv.position')->getQuery()->getArrayResult() as $v)
			$variantValues[$v['id']] = $v['name'];

		return $variantValues;
	}

	/**
	 * @param int $id
	 *
	 * @return VariantValueText|null
	 */
	private function getNameVariantValue($id)
	{
		return $this->variantsService->em->getRepository(VariantValueText::class)->findOneBy([
			'variantValue' => $id,
			'lang'         => $this->translator->getLocale(),
		]);
	}
}
