<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Import;

use Core\Model\UI\Form\BaseContainer;
use Import\AdminModule\Model\ImportType\ImportType;
use Nette\Localization\ITranslator;
use Nette\Http\Request;

/**
 * Class TypeProduct
 * @package EshopCatalog\AdminModule\Model\Import
 */
class TypeProduct extends ImportType
{
	/** @var string */
	public $title = 'eshopCatalog.import.typeProduct';

	/** @var string */
	public $name = 'eshopCatalog.product';

	/** @var ITranslator */
	protected $translator;

	/** @var Request */
	protected $httpRequest;

	/** @var ImportHelper */
	protected $importHelper;

	public function __construct(ITranslator $translator, Request $request, ImportHelper $importHelper)
	{
		$this->translator   = $translator;
		$this->httpRequest  = $request;
		$this->importHelper = $importHelper;
	}

	/**
	 * @param $request
	 */
	public function setHttpRequest($request) { $this->httpRequest = $request; }

	public function getFormSyncOpts(array $importKeys, array $importValues, $defaults)
	{
		$container = new BaseContainer();

		$arr = [];
		foreach (['ean', 'code1', 'code2'] as $v)
			$arr[$v] = 'eshopCatalog.import.' . $v;
		$default = null;
		if (isset($defaults['syncBy']) && array_key_exists($defaults['syncBy'], $arr))
			$default = $defaults['syncBy'];
		$container->addSelect('syncBy', 'import.importForm.syncBy', $arr)->setDefaultValue($default)->setRequired();

		$container->addText('xPath', 'import.import.xPath')->setDefaultValue($defaults['xPath'] ?? null);
		$container->addSelect('onlyUpdate', 'import.import.onlyUpdate', [
			0 => 'default.no',
			1 => 'default.yes',
		])->setDefaultValue($defaults['onlyUpdate'] ?? 0);
		$container->addText('supplier', 'eshopCatalog.import.supplier')->setRequired()
			->setDefaultValue($defaults['supplier'] ?? null);

		$default = null;
		$arr     = $this->importHelper->categoriesList();
		if (isset($defaults['baseCategory']) && array_key_exists($defaults['baseCategory'], $arr))
			$default = $defaults['baseCategory'];

		$container->addSelect('baseCategory', 'import.import.baseCategory', $this->importHelper->categoriesList())
			->setDefaultValue($default);

		return $container;
	}

	public function getFormContainer(array $importKeys, array $importValues, $defaults)
	{
		$importKeys   = [null => null] + $importKeys;
		$importValues = [null => null] + $importValues;
		$container    = new BaseContainer();
		$postData     = $this->httpRequest->getPost('data');
		/** @var BaseContainer[] $containers */
		$containers = [];

		$fields = [
			'ean'              => ['type' => 'sel'],
			'code1'            => ['type' => 'sel'],
			'code2'            => ['type' => 'sel'],
			'supplierCode'     => ['type' => 'sel'],
			'lang'             => ['type' => 'sel'],
			'name'             => ['type' => 'sel'],
			'shortDescription' => ['type' => 'sel'],
			'description'      => ['type' => 'sel'],
			'quantity'         => ['type' => 'sel'],
			'inStock'          => ['type' => 'comparison'],
			'category'         => ['type' => 'sel', 'fixed' => false],
			'priceVat'         => ['type' => 'sel'],
			'retailPrice'      => ['type' => 'sel'],
			'vatRate'          => ['type' => 'sel'],
			'manufacturer'     => ['type' => 'sel'],
			'isPublished'      => ['type' => 'bol'],
			'params'           => ['type' => 'slist'],
			'images'           => ['type' => 'slist'],
			'imagesUrlPrefix'  => ['type' => 'sel'],
		];

		foreach ($fields as $k => $opts) {
			$containers[] = $c = $container->addContainer($k, 'eshopCatalog.import.' . $k);

			if ($opts['type'] == 'sel') {
				// Select
				$c->addSelect('key', 'import.importForm.select', $importKeys)->setDefaultValue($defaults[$k]['key'] ?? null);

				if (!isset($opts['fixed']) || $opts['fixed'])
					$c->addText('fixedValue', 'import.importForm.orFixedValue')
						->setDefaultValue($defaults[$c->getName()]['fixedValue'] ?? null);
			} else if ($opts['type'] == 'bol') {
				// Bolean
				$c->addSelect('key', 'import.importForm.select', $importKeys)
					->setAttribute('onchange', "importFormUpdateSelect('$k')")
					->setDefaultValue($defaults[$k]['key'] ?? null);

				$items = [];
				if (isset($importValues[$postData[$k]['key']]))
					$items = $importValues[$postData[$k]['key']] ?? [];
				else if (isset($defaults[$k]['key']))
					$items = $importValues[$defaults[$k]['key']] ?? [];

				if ($items)
					$items = [null => null] + $items;

				$c->addSelect('value', 'import.importForm.valueIs', $items)
					->setDefaultValue($defaults[$k]['value']);

				if (!isset($opts['fixed']) || $opts['fixed']) {
					$arr = [
						''   => $this->translator->translate('import.importForm.noUse'),
						'v1' => $this->translator->translate('default.yes'),
						'v0' => $this->translator->translate('default.no'),
					];

					$default = '';
					if (isset($defaults[$c->getName()]['fixedValue']) && array_key_exists('v' . $defaults[$c->getName()]['fixedValue'], $arr))
						$default = 'v' . $defaults[$c->getName()]['fixedValue'];

					$c->addRadioList('fixedValue', $this->translator->translate('import.importForm.orFixedValue'), $arr)->setDefaultValue($default);
				}
			} else if ($opts['type'] == 'slist') {
				// Sortable list
				$arr = $importKeys;
				array_shift($arr);
				$default = [];
				foreach ($defaults[$c->getName()]['key'] as $k)
					if (array_key_exists($k, $arr))
						$default[] = $k;
				$c->addSortableCheckboxList('key', 'import.importForm.select', $arr)->setDefaultValue($default);
			} else if ($opts['type'] == 'comparison') {
				$c->addSelect('key', 'import.importForm.select', $importKeys)
					->setAttribute('onchange', "importFormUpdateSelect('$k')")
					->setDefaultValue($defaults[$k]['key'] ?? null);

				$items = [];
				if (isset($importValues[$postData[$k]['key']]))
					$items = $importValues[$postData[$k]['key']] ?? [];
				else if (isset($defaults[$k]['key']))
					$items = $importValues[$defaults[$k]['key']] ?? [];

				if ($items)
					$items = [null => null] + $items;

				$comparison = [
					'<'  => '<',
					'<=' => '<=',
					'='  => '=',
					'>=' => '>=',
					'>'  => '>',
				];

				$c->addSelect('valueIs', 'import.importForm.valueIs', $comparison)
					->setDefaultValue($defaults[$k]['valueIs']);
				$c->addSelect('value', 'import.importForm.value', $items)
					->setDefaultValue($defaults[$k]['value']);
			}

			if ($k == 'category') {
				$c->addText('separator', 'import.importForm.categorySeparator')
					->setDefaultValue($defaults[$c->getName()]['separator'] ?? null);
				$c->addText('deepSeparator', 'import.importForm.categoryDeepSeparator')
					->setDefaultValue($defaults[$c->getName()]['deepSeparator'] ?? null);

				$default = null;
				$arr     = $this->importHelper->categoriesList();
				if (isset($defaults[$c->getName()]['fixedValue']) && array_key_exists($defaults[$c->getName()]['fixedValue'], $arr))
					$default = $defaults[$c->getName()]['fixedValue'];
				$c->addSelect('fixedValue', 'import.importForm.orFixedValue', ['' => ''] + $arr)
					->setDefaultValue($default);
			}
		}

		foreach ($containers as $c) {
			$arr     = [
				'all' => $this->translator->translate('import.importForm.applyOnAll'),
				'new' => $this->translator->translate('import.importForm.applyOnNew'),
			];
			$default = 'new';
			if (isset($defaults[$c->getName()]['applyOn']) && array_key_exists($defaults[$c->getName()]['applyOn'], $arr))
				$default = $defaults[$c->getName()]['applyOn'];
			$c->addRadioList('applyOn', 'import.importForm.applyOn', $arr)->setDefaultValue($default);
		}

		return $container;
	}

	public function processSubmitSyncOpts($values)
	{
		return $values;
	}

	public function processSubmitData($values)
	{
		$data = [];
		foreach ($values as $k => $v) {
			if (!$v->key && $v->fixedValue === '')
				continue;

			if ($v->key)
				unset($v->fixedValue);
			else if ($v->fixedValue !== '') {
				unset($v->key);
				unset($v->value);

				if (in_array($v->fixedValue, ['v0', 'v1']))
					$v->fixedValue = ltrim($v->fixedValue, 'v');
			}

			$data[$k] = $v;
		}

		return $data;
	}
}
