<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Export;

use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Provider\IXmlExportService;
use Nette\Utils\Strings;

class GoogleXmlService extends XmlService implements IXmlExportService
{
	/** @var string */
	protected $name = 'google';

	/** @var Sites */
	protected $sites;

	public function __construct(Sites $sites)
	{
		$this->sites = $sites;
	}

	public function startExport(): void
	{
		parent::startExport();

		$site     = $this->sites->getCurrentSite();
		$siteName = $site->getSiteName();

		$this->writeToFile('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<rss version="2.0" xmlns:g="http://base.google.com/ns/1.0">' . PHP_EOL .
			'<channel>' . PHP_EOL .
			'<title>' . $siteName . '</title>' . PHP_EOL .
			'<link>' . $this->baseUrl . '</link>' . PHP_EOL .
			'<description>Produkty obchodu ' . $siteName . '</description>' . PHP_EOL);
	}

	public function writeItem(Product $product): void
	{
		$export = $this->prepareExportData($product);

		if ($export['status'] == 0)
			return;

		$dom               = new \DOMDocument();
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('item');

		$child = $dom->createElement('g:id', (string) $product->getId());
		$shopitem->appendChild($child);

		$child     = $dom->createElement('title');
		$childText = $dom->createTextNode((string) $export['product']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('description');
		$childText = $dom->createCDATASection(Strings::normalize(strip_tags((string) $product->getDescription())));
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child = $dom->createElement('link', (string) $product->link);
		$shopitem->appendChild($child);

		$child = $dom->createElement('g:price', number_format($product->getPrice(), 2, ".", ""));
		$shopitem->appendChild($child);

		if ($product->quantity > 0) {
			$child = $dom->createElement('g:availability', 'in stock');
			$shopitem->appendChild($child);
		}

		if ($product->getGallery() && $product->getGallery()->cover) {
			$child = $dom->createElement('g:image_link', (string) ($this->baseUrl . $product->getGallery()->cover->file));
			$shopitem->appendChild($child);
		}

		if ($product->getEan()) {
			$child = $dom->createElement('g:gtin', (string) $product->getEan());
			$shopitem->appendChild($child);
		}

		if ($export['categoryText']) {
			$child = $dom->createElement('g:google_product_category', (string) $export['categoryText']);
			$shopitem->appendChild($child);
		}

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('g:brand', (string) $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		$dom->appendChild($shopitem);

		$this->writeToFile($dom->saveXML($dom->documentElement) . PHP_EOL);
	}

	public function endExport()
	{
		$this->writeToFile('</channel>' . PHP_EOL . '</rss>');

		parent::endExport();
	}
}
