<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Categories;

use Core\Model\Helpers\Arrays;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\CategoryTexts;
use Core\Model\UI\BaseControl;
use Core\Components\Flashes\Flashes;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use Doctrine\ORM\Query\Expr\Join;
use Nette\Application\UI\Multiplier;
use Nette\Utils\ArrayHash;

/**
 * TODO cache při odstranění
 *
 * Class CategoriesGrid
 * @package EshopCatalog\AdminModule\Components\Categories
 */
class CategoriesGrid extends BaseControl
{
	/** @var string */
	protected $siteIdent;

	/** @var Categories */
	protected $categoriesService;

	/** @var ICategoryFormFactory */
	protected $categoryFormFactory;

	/** @var array */
	protected $cData = null, $cDefaultCategoryProductsCount, $cOtherCategoryProductsCount;

	public function __construct(string $siteIdent, Categories $categories, ICategoryFormFactory $categoryFormFactory)
	{
		$this->siteIdent           = $siteIdent;
		$this->categoriesService   = $categories;
		$this->categoryFormFactory = $categoryFormFactory;
	}

	public function render(): void
	{
		$this->template->categories = $this->getData()['tree'];
		$this->template->siteIdent  = $this->siteIdent;
		$this->template->render($this->getTemplateFile());
	}

	protected function canUpdateGrid(): bool { return $this->categoriesService->getSessionSection()->categoriesGridUpdateEnabled ?? true; }

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		$presenter = $this->getPresenter();
		if ($this->categoriesService->removeCategory($id))
			$presenter->flashMessage('eshopCatalog.defaultGrid.removed', Flashes::FLASH_SUCCESS);
		else
			$presenter->flashMessage('eshopCatalog.defaultGrid.removeFailed', Flashes::FLASH_DANGER);

		if ($presenter->isAjax()) {
			$this->redrawControl('list');
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleEdit(int $id): void
	{
		$category = $this->categoriesService->get($id);

		if ($category) {
			$this['categoryForm']->categoryId = $id;
			$this->template->modalTitle       = $this->t('eshopCatalog.title.editCategory') . ' ' . $category->getCategoryText()->getName();
			$this->template->modal            = 'categoryForm';
			$this->redrawControl('modal');
		}
	}

	public function handleChangePosition(): void
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();

		$id   = (int) $request->getPost('id');
		$move = (int) $request->getPost('move');

		if ($id && $move) {
			$er  = $this->categoriesService->getEr();
			$cat = $this->categoriesService->get($id);

			if ($cat) {
				if ($move < 0)
					$er->moveUp($cat, abs($move));
				else if ($move > 0)
					$er->moveDown($cat, $move);
				$presenter->flashMessageSuccess('default.positionChanged');
			}
		} else
			$presenter->flashMessageDanger('default.positionChangeFailed');

		$this->em->flush();
		if ($id)
			$this->categoriesService->cleanCacheDeep($id);

		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	/**
	 * @return CategoryForm
	 */
	protected function createComponentCategoryForm(): CategoryForm
	{
		$control = $this->categoryFormFactory->create();
		if ($this->getParameter('id'))
			$control->setCategory((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			if ($this->canUpdateGrid())
				$this->redrawControl('list');
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			if ($this->canUpdateGrid())
				$this->redrawControl('list');
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentPublishForm(): Multiplier
	{
		return new Multiplier(function($id): BaseForm {
			$form = $this->createForm();
			//			$form->getElementPrototype()->novalidate(true);

			$form->addBool('publish', '')
				->setDefaultValue($this->getData()['data'][$id]['isPublished'] ?? 0)
				->setHtmlAttribute('onchange', 'window.naja.uiHandler.submitForm(this.form)');
			$form['publish']->disableJsWatcher = true;

			$form->onSuccess[] = function(BaseForm $form, ArrayHash $values) use ($id) {
				$presenter = $this->getPresenter();

				if ($this->categoriesService->setPublish((int) $id, $values->publish))
					$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
				else
					$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');

				if ($presenter->isAjax()) {
					$presenter->redrawControl('flashes');
				} else
					$presenter->redirect('this');
			};

			return $form;
		});
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function getData(): array
	{
		if ($this->cData !== null)
			return $this->cData;

		$rootId = $this->categoriesService->getEr()->createQueryBuilder('c')
				->select('c.id')
				->innerJoin(CategoryTexts::class, 'ct', Join::WITH, 'c.id = ct.id')
				->where('c.lvl = :lvl')
				->andWhere('ct.alias = :alias')
				->setParameters([
					'lvl'   => 0,
					'alias' => $this->siteIdent,
				])->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;

		if (!$rootId) {
			$newCategory     = new Category();
			$newCategoryText = new CategoryTexts($newCategory, $this->translator->getLocale());
			$newCategoryText->setName($this->siteIdent);

			$this->em->persist($newCategory)
				->persist($newCategoryText)
				->flush();

			$rootId = $newCategory->getId();
		}

		// Kategorie
		$qb   = $this->categoriesService->getEr()->createQueryBuilder('c', 'c.id')
			->select('c.id, c.isPublished, ct.name, IDENTITY(c.parent) as parent')
			->leftJoin('c.categoryTexts', 'ct')
			->where('ct.lang = :lang')
			->andWhere('c.root = :root')
			->setParameters([
				'lang' => $this->translator->getLocale(),
				'root' => $rootId,
			])
			->groupBy('c.id')
			->addOrderBy('c.lft');
		$data = $qb->getQuery()->getArrayResult();

		foreach ($data as $k => $row) {
			// Počet produktů podle výchozí kategorie
			$data[$k]['productsCount'] = count($this->getProductsCountByDefaultCategory((int) $k) + $this->getProductsCountByOtherCategories((int) $k));
		}

		$this->cData = [
			'data' => $data,
			'tree' => Arrays::buildTree($data, 'parent', 'id', $rootId),
		];

		return $this->cData;
	}

	protected function getProductsCountByDefaultCategory(int $categoryId)
	{
		if ($this->cDefaultCategoryProductsCount === null) {
			$this->cDefaultCategoryProductsCount = [];
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('p.id, IDENTITY(p.idCategoryDefault) as cat')
				         ->getQuery()->getScalarResult() as $row)
				$this->cDefaultCategoryProductsCount[(int) $row['cat']][] = $row['id'];

			foreach ($this->em->getRepository(ProductInSite::class)->createQueryBuilder('ps')
				         ->select('IDENTITY(ps.product) as id, IDENTITY(ps.category) as cat')
				         ->getQuery()->getScalarResult() as $row) {
				$this->cDefaultCategoryProductsCount[(int) $row['cat']][] = $row['id'];
			}
		}

		return $this->cDefaultCategoryProductsCount[$categoryId] ?? [];
	}

	protected function getProductsCountByOtherCategories(int $categoryId)
	{
		if ($this->cOtherCategoryProductsCount === null) {
			$this->cOtherCategoryProductsCount = [];
			foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
				         ->select('IDENTITY(cp.product) as id, IDENTITY(cp.category) as cat')
				         ->getQuery()->getScalarResult() as $row)
				$this->cOtherCategoryProductsCount[(int) $row['cat']][] = $row['id'];
		}

		return $this->cOtherCategoryProductsCount[$categoryId] ?? [];
	}
}
