<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\Entities\ExtraField;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Parameters;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\FrontModule\Model\SeoContainer;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\FormContainers\DocumentContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductSpeditionContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductSupplierContainer;
use EshopCatalog\AdminModule\Model\FormContainers\RelatedProductContainer;
use EshopCatalog\AdminModule\Model\Helpers\ProductFormHelper;
use EshopCatalog\AdminModule\Model\ProductDocuments;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\AdminModule\Model\ProductTags;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\Supplier;
use EshopCatalog\Model\Entities\SupplierTexts;
use EshopCatalog\Model\Entities\Tag;
use EshopStock\DI\EshopStockExtension;
use EshopStock\Model\Entities\SupplyProduct;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Import\DI\ImportExtension;
use Nette\Application\IPresenter;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureProducts;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\FeatureProduct;
use Nette\Utils\Html;
use Nette\Utils\Json;

class ProductForm extends BaseControl
{
	/** @var Product */
	public $product;

	protected Products $productServices;

	protected FeatureProducts $featureProductServices;

	protected CategoryProducts $categoryProductServices;

	protected Categories $categoryServices;

	protected CategoryContainer $categoryContainerService;

	protected Manufacturers $manufacturerServices;

	protected Suppliers $supplierServices;

	protected VatRates $vatRateServices;

	protected Features $featureServices;

	protected SeoContainer $seoContainerService;

	protected IImagesZoneFactory $imagesZoneFactory;

	protected Request $httpRequest;

	protected TagsContainer $tagsContainerService;

	protected ProductTags $productTagsServices;

	protected ProductFormHelper $helper;

	protected AvailabilityService $availabilityService;

	private ?array $cFeatures = null;

	private ?array $cFeatureValues = null;

	protected DocumentContainer $documentContainer;

	protected RelatedProductContainer $relatedProductContainer;

	protected ProductDocuments $productDocuments;

	protected ProductSupplierContainer $productSupplierContainer;

	protected ProductSpeditionContainer $speditionContainer;

	protected ?array $suppliersToAdd = null;

	public function __construct(Products $products, Features $features, FeatureProducts $featureProducts, Categories $categories,
	                            CategoryProducts $categoryProducts, Manufacturers $manufacturers, Suppliers $suppliers,
	                            VatRates $vatRates, SeoContainer $seoContainer, IImagesZoneFactory $imagesZoneFactory,
	                            Request $request, CategoryContainer $categoryContainer, TagsContainer $tagsContainer,
	                            ProductTags $productTags, ProductFormHelper $helper, DocumentContainer $documentContainer,
	                            ProductDocuments $productDocuments, RelatedProductContainer $relatedProductContainer,
	                            ProductSpeditionContainer $speditionContainer, AvailabilityService $availabilityService,
								ProductSupplierContainer $productSupplierContainer)
	{
		$this->productServices          = $products;
		$this->featureServices          = $features;
		$this->featureProductServices   = $featureProducts;
		$this->categoryContainerService = $categoryContainer;
		$this->categoryServices         = $categories;
		$this->categoryProductServices  = $categoryProducts;
		$this->manufacturerServices     = $manufacturers;
		$this->supplierServices         = $suppliers;
		$this->seoContainerService      = $seoContainer;
		$this->vatRateServices          = $vatRates;
		$this->imagesZoneFactory        = $imagesZoneFactory;
		$this->httpRequest              = $request;
		$this->tagsContainerService     = $tagsContainer;
		$this->productTagsServices      = $productTags;
		$this->helper                   = $helper;
		$this->documentContainer        = $documentContainer;
		$this->relatedProductContainer  = $relatedProductContainer;
		$this->productDocuments         = $productDocuments;
		$this->speditionContainer       = $speditionContainer;
		$this->availabilityService      = $availabilityService;
		$this->productSupplierContainer = $productSupplierContainer;
	}

	public function render(): void
	{
		$this->template->thisForm         = $this['form'];
		$this->template->product          = $this->product;
		$this->template->groupCustomers   = $this->helper->getGroupCustomers();
		$this->template->enablePriceLevel = $this->helper->config->get('enablePriceLevels', false);
		$this->template->sites            = $this->helper->sitesService->getAll();

		$features                      = $this->getFeatures();
		$this->template->features      = $features;
		$this->template->featureValues = $this->getFeatureValues();

		$defaults = [];
		if ($this->product) {
			foreach ($this->product->getFeatureProducts()->toArray() as $row)
				$defaults[] = [
					'feature' => $row->getFeature()->getId(),
					'value'   => $row->getFeatureValue()->getId(),
				];
		}
		$this->template->defaultFeatures = $defaults;

		$this->template->featuresJson = Json::encode([
			'features' => $features,
			'values'   => $this->getFeatureValues(),
		]);

		$this->template->suppliersToAdd = $this->suppliersToAdd;

		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), ProductForm::class . '::render');
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ========================  HANDLE
	 */

	public function handleShowSuppliersToAdd(): void
	{
		$this->suppliersToAdd = [];
		$availabilityPairs = [];
		$supplierPairs = [];

		$qb = $this->em->createQueryBuilder();
		$qb->select('a.id, at.name')
		   ->from(Availability::class, 'a')
		   ->leftJoin('a.texts', 'at', 'WITH', 'at.lang = :lang')
		   ->orderBy('a.position')
		   ->setParameters([
			   'lang' => $this->translator->getLocale(),
		   ]);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$availabilityPairs[$row['id']] = $row['name'];
		}

		$ids = array_map(fn(ProductSupplier $ps) => $ps->getSupplier()->getId(), $this->product->getSuppliers()->toArray());
		$qb = $this->em->getRepository(Supplier::class)->createQueryBuilder('s');
		$qb->select('s.id, s.name')
		   ->orderBy('s.name');

		if ($ids) {
			$qb->andWhere($qb->expr()->notIn('s.id', $ids));
		}

		$container = new BaseContainer;
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$this->suppliersToAdd[$row['id']] = $row['name'];
			$c = $this->productSupplierContainer->getContainer();
			$c->addCustomData('name', $row['name']);
			$container->addComponent($c, 'supp_' . $row['id']);
		}
		$this['form']->addComponent($container, 'suppliersContainer');

		$this->redrawControl('suppliers-to-add-snippet');
	}

	public function handleRemoveDocument($documentId)
	{
		foreach ($this->product->getDocuments() as $key => $doc) {
			if ($doc->getId() === (int) $documentId) {
				$this->productDocuments->remove((int) $documentId);
			}
		}
	}

	public function handleAddFeatureValue(string $id, string $val)
	{
		if (!$val)
			$this->getPresenter()->sendJson([]);

		$exist = $this->em->getRepository(FeatureValueTexts::class)->createQueryBuilder('fvt')
				->select('IDENTITY(fvt.id) as id, fvt.name as name')
				->innerJoin('fvt.id', 'fv', Join::WITH, 'fv.feature = :feature')
				->where('fvt.name = :v')
				->setParameters([
					'feature' => $id,
					'v'       => $val,
				])
				->getQuery()->getResult()[0] ?? null;

		if (!$exist) {
			$featureValue = new FeatureValue();
			$featureValue->setFeature($this->featureServices->getReference((int) $id));
			$featureValue->isPublished = 1;
			$this->em->persist($featureValue);

			foreach ($this->langsService->getLangs(false) as $lang) {
				$featureValueText = new FeatureValueTexts($featureValue, $lang->getTag());
				$featureValueText->setName($val);
				$this->em->persist($featureValueText);
			}

			$this->em->flush();
			$exist = [
				'id'      => $featureValue->getId(),
				'name'    => $val,
				'created' => true,
			];
		}

		$this->getPresenter()->sendJson([
			'data' => $exist,
		]);
	}

	public function handleCreateVariant(int $id): void
	{
		$variant = $this->productServices->createVariant($id);

		$this->getPresenter()->flashMessageSuccess('eshopCatalog.product.variantCreated');
		$this->getPresenter()->redirect('edit', $variant->getId());
	}

	public function handleRemoveFromVariants(): void
	{
		if ($this->productServices->removeFromVariants($this->product))
			$this->getPresenter()->flashMessageSuccess('default.saved');
		else
			$this->getPresenter()->flashMessageDanger('default.error');
		$this->redirect('this');
	}

	public function handleSetAsVariantFor(): void
	{
		if ($this->productServices->setAsVariantFor($this->product->getId(), (int) $this->getPresenter()->getHttpRequest()->getPost('productId'))) {
			$this->getPresenter()->flashMessageSuccess('default.saved');
			$this->redirect('this');
		} else {
			$this->getPresenter()->flashMessageDanger('eshopCatalog.productForm.setAsVariantForError');
			$this->getPresenter()->redrawControl('flashes');
		}
	}

	public function handleSetAsMainVariant(int $id): void
	{
		if ($this->productServices->setAsMainVariant($id)) {
			$this->getPresenter()->flashMessageSuccess('default.saved');
			$this->getPresenter()->redirect('edit', $id);
		} else {
			$this->getPresenter()->flashMessageDanger('default.errorl');
			$this->getPresenter()->redrawControl('flashes');
		}
	}

	public function handleDuplicateProduct(int $id): void
	{
		$newProductId = $this->productServices->duplicateProduct($id);
		if ($newProductId) {
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.product.duplicated');
			$this->getPresenter()->redirect('edit', $newProductId);
		} else {
			$this->getPresenter()->flashMessageDanger('eshopCatalog.product.duplicateFailed');
			$this->getPresenter()->redrawControl('flashes');
		}
	}

	/*******************************************************************************************************************
	 * ========================  COMPONENT
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$this->monitor(IPresenter::class, function() use ($form) {
			if ($this->product) {
				$form->addHeadLink(
					$this->link('createVariant!', $this->product->getId()),
					$this->t('eshopCatalog.product.createVariant'),
					'ajax',
				);
				$form->addHeadLink(
					$this->link('duplicateProduct!', $this->product->getId()),
					$this->t('eshopCatalog.product.duplicateProduct'),
					'ajax',
				);
			}
		});

		$form->addBool('isPublished', 'eshopCatalog.productForm.isPublished')->setDefaultValue(1);
		$form->addText('name', 'eshopCatalog.productForm.name')->setMaxLength(255)->setIsMultilanguage();
		if (Config::load('enableProductName2'))
			$form->addText('name2', 'eshopCatalog.productForm.name2')->setMaxLength(255)
				->setIsMultilanguage()->setNullable();
		$form->addText('ean', 'eshopCatalog.productForm.ean')->setMaxLength(13);
		$form->addText('code1', 'eshopCatalog.productForm.code1')->setMaxLength(60);
		$form->addText('code2', 'eshopCatalog.productForm.code2')->setMaxLength(60);

		if (Config::load('product.allowAssort'))
			$form->addBool('isAssort', 'eshopCatalog.productForm.isAssort')->setDefaultValue(0);

		if (Config::load('product.allowNote')) {
			$form->addBool('addNote', 'eshopCatalog.product.allowAddNote')->setDefaultValue(0);
			$form->addText('noteTitle', 'eshopCatalog.product.noteTitle')->setIsMultilanguage();
		}

		if (self::allowSaleDiscountCouponsAsProducts()) {
			$form->addBool('isDiscount', 'eshopCatalog.productForm.isDiscount')->setDefaultValue(0);
		}

		$form->addSelect('condition', 'eshopCatalog.productForm.condition', [
			'new'         => 'eshopCatalog.productConditions.new',
			'refurbished' => 'eshopCatalog.productConditions.refurbished',
			'used'        => 'eshopCatalog.productConditions.used',
		]);

		$form->addInteger('quantity', 'eshopCatalog.productForm.quantity')
			->setDescription('eshopCatalog.productForm.quantityDescription')
			->setDefaultValue(0);
		if (Config::load('pseudoWarehouse'))
			$form->addBool('unlimitedQuantity', 'eshopCatalog.productForm.unlimitedQuantity')->setDefaultValue(0);
		if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
			$form->addBool('categoryGiftsAllowed', 'eshopCatalog.productForm.categoryGiftsAllowed')->setDefaultValue(1);
		$form->addSelect('availability', $this->t('eshopCatalog.productForm.availability'), $this->helper->availabilityService->getOptionsForSelect())
			->setTranslator(null);
		$form->addSelect('availabilityAfterSoldOut', $this->t('eshopCatalog.productForm.availabilityAfterSoldOut'), [null => ''] + $this->helper->availabilityService->getOptionsForSelect())
			->setTranslator(null);

		$form->addText('price', 'eshopCatalog.productForm.price' . (Config::load('product.priceIsWithoutVat', false) ? 'WithoutVat' : ''))
			->setHtmlType('number')
			->setHtmlAttribute('step', .01)
			->setHtmlAttribute('min', 0)
			->setRequired();
		$form->addText('retailPrice', 'eshopCatalog.productForm.retailPrice' . (Config::load('product.priceIsWithoutVat', false) ? 'WithoutVat' : ''))
			->setHtmlType('number')
			->setHtmlAttribute('step', .01)
			->setHtmlAttribute('min', 0)
			->setNullable();
		if (Config::load('enablePurchasePrice'))
			$form->addText('purchasePrice', 'eshopCatalog.productForm.purchasePrice')
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0)
				->setNullable();
		$form->addEditor('shortDesc', 'eshopCatalog.productForm.shortDesc')->setToolbar('Text')->setDisableAutoP()->setHeight(100)
			->setMaxLength(250)->setIsMultilanguage();
		$form->addEditor('desc', 'eshopCatalog.productForm.desc')->setHeight(300)->setIsMultilanguage();

		if (class_exists(ImportExtension::class)) {
			$form->addBool('stopImportPrice', 'eshopCatalog.productForm.stopImportPrice')->setDefaultValue(1);
		}

		if (Config::load('product.allowStopImportQuantity')) {
			$form->addBool('stopImportQuantity', 'eshopCatalog.productForm.stopImportQuantity')->setDefaultValue(0);
		}

		if (Config::load('product.loadQuantityOnStock') && class_exists(EshopStockExtension::class)) {
			$form->addText('quantityOnStock', 'eshopCatalog.productForm.quantityOnStock')
				->setDefaultValue(0)
				->setHtmlAttribute('readonly', true)
				->setOmitted();
		}

		$form->addBool('discountDisabled', 'eshopCatalog.productForm.discountDisabled')->setDefaultValue(1);

		if (Config::load('product.allowDisablePickUpSpedition'))
			$form->addBool('disablePickUpSpedition', 'eshopCatalog.product.disablePickUpSpedition')->setDefaultValue(0);
		if (Config::load('product.allowMarkAsOversize'))
			$form->addBool('isOversize', 'eshopCatalog.product.isOversize')->setDefaultValue(0);
		if (Config::load('product.allowDisableStackInCart'))
			$form->addBool('disableStackInCart', 'eshopCatalog.product.disableStackInCart')->setDefaultValue(0);

		//Manufacturers - Vyrobci
		$manufacturers = ['' => ''];
		foreach ($this->manufacturerServices->getAll() as $ms)
			$manufacturers[$ms->getId()] = $ms->name;
		$form->addSelect('manufacturer', $this->t('eshopCatalog.productForm.manufacturer'), $manufacturers)
			->setTranslator(null);

		//Suppliers - Dodavatele
		// Aktuálně pouze zobrazení z importu

		//VatRates - DPH
		$vatRates = [];
		foreach ($this->vatRateServices->getAll() as $vrs)
			$vatRates[$vrs->getId()] = $vrs->name;
		$form->addSelect('vatRate', 'eshopCatalog.productForm.vatRate', $vatRates)->setPrompt('eshopCatalog.productForm.choiceVatRate')
			->setRequired();

		//PriceLevel
		if ($this->helper->config->get('enablePriceLevels', false)) {
			foreach ($this->helper->getGroupCustomers() as $groupCustomers) {
				$nameSuffix = '';
				if (Config::load('product.priceIsWithoutVat', false))
					$nameSuffix = ' ' . $this->t('eshopCatalog.withoutVat');
				$form->addText('pricelevel' . $groupCustomers->getId(), $groupCustomers->name . $nameSuffix)
					->setHtmlType('number')
					->setHtmlAttribute('step', .01)
					->setHtmlAttribute('min', 0)
					->setNullable();
			}
		}

		// Kategorie a Eshopy
		$form->addComponent($this->categoryContainerService->getContainers(), 'categories');

		//Features - Vlastnosti
		$form->addContainer('features');

		//Variants
		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$form->addComponent($this->tagsContainerService->getContainer(), 'tags');

		// Doprava
		if (Config::load('product.allowModifySpedition', false)) {
			$form->addComponent($this->speditionContainer->getContainer(), 'speditions');
		}

		$form->addHidden('preparedAlbumId');

		$this->eventDispatcher->dispatch(new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null), ProductForm::class . '::createForm');

		$form->addSaveCancelControl('saveControl');

		// Nastaveni pro varianty
		if ($this->product && $this->product->isVariant()) {
			if ($this->product->getIsVariantParent()) {
				$form->addText('variantName', 'eshopCatalog.product.variantName')
					->setMaxLength(255)
					->setNullable();
			} else {
				$variantUse    = $form->addContainer('variantUse');
				$parentProduct = $this->product->getVariantParent();

				foreach (get_object_vars($this->product->isVariant()) as $k => $v) {
					if (in_array($k, ['defaultImage', 'createdDefault']))
						continue;
					$variantUse->addBool($k, 'eshopCatalog.product.useForVariant')->setDefaultValue($v ? 1 : 0);
				}

				$form['manufacturer']->setDisabled()
					->setDefaultValue($parentProduct->getManufacturer() ? $parentProduct->getManufacturer()->getId() : null);
				$form['vatRate']->setDisabled()
					->setDefaultValue($parentProduct->getVateRate() ? $parentProduct->getVateRate()->getId() : null);
			}
		}
		// Dokumenty
		$form->addComponent($this->documentContainer->getContainer(), 'documents');

		// Souvisejici produkty
		$form->addComponent($this->relatedProductContainer->getContainer(), 'relatedProducts');

		$this->monitor(IPresenter::class, function(AbstractPresenter $presenter) use ($form) {
			$form['relatedProducts']['product']->getControlPrototype()->setAttribute('data-link', $presenter->link(':EshopCatalog:Cron:Products:findByTerm', [
				'excluded' => $this->product ? [$this->product->getId()] : [],
			]));

			$this->template->renderUseVariant = function() { return ''; };
			if ($this->product) {
				// vzkresleni zapnuti hodnot pro varianty
				$this->template->isVariant        = $this->product->isVariant() && !$this->product->getIsVariantParent();
				$this->template->renderUseVariant = function(string $c) {
					$form = $this['form'];
					$c    = 'use' . ucfirst($c);
					if (isset($form['variantUse'][$c]))
						return Html::el('div', [
							'class'             => 'variant-use-bool ' . ($form['variantUse'][$c]->getValue() ? 'active' : ''),
							'data-variant-bool' => $c,
						])->addHtml($form->getRenderer()->renderPair($form['variantUse'][$c]));

					return '';
				};
			}
		});

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		$nameFilled = false;
		foreach ($values->name as $v)
			if ($v)
				$nameFilled = true;

		if (!$nameFilled)
			$form->addError('eshopCatalog.productForm.nameMissing');

		if ($form->hasErrors())
			$this->redrawControl('formErrors');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$this->em->beginTransaction();
		try {
			$langValues = $form->convertMultilangValuesToArray();
			$product    = $this->product ?: new Product();
			/** @var ProductVariant|null $va */
			$va = $product->isVariant();

			$product->quantity         = $values->quantity;
			$product->price            = $values->price;
			$product->retailPrice      = $values->retailPrice ?: null;
			$product->purchasePrice    = $values->purchasePrice ?: null;
			$product->isPublished      = $values->isPublished;
			$product->inStock          = $values->quantity > 0 ? 1 : 0;
			$product->isDiscount       = (int) (self::allowSaleDiscountCouponsAsProducts() ? $values->isDiscount : 0);
			$product->ean              = $values->ean;
			$product->code1            = $values->code1;
			$product->code2            = $values->code2;
			$product->discountDisabled = $values->discountDisabled == 1 ? 0 : 1;
			$product->isAssort         = Config::load('product.allowAssort')
				? ($values->isAssort ? 1 : 0)
				: 0;
			$product->condition        = $values->condition;
			$product->setAvailability($this->helper->availabilityService->getReference($values->availability));
			$product->availabilityAfterSoldOut = $values->availabilityAfterSoldOut ? $this->helper->availabilityService->getReference($values->availabilityAfterSoldOut) : null;

			if (isset($values->disablePickUpSpedition))
				$product->disablePickUpSpedition = $values->disablePickUpSpedition ? 1 : 0;
			if (isset($values->isOversize))
				$product->isOversize = $values->isOversize ? 1 : 0;
			if (isset($values->disableStackInCart))
				$product->disableStackInCart = $values->disableStackInCart ? 1 : 0;

			if (Config::load('pseudoWarehouse')) {
				$product->unlimitedQuantity = $values->unlimitedQuantity;
				if ($product->unlimitedQuantity)
					$product->setAvailability($this->helper->availabilityService->getAllByIdent()['inStock']);
			}

			if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
				$product->setCategoryGiftsAllowed($values->categoryGiftsAllowed);

			$this->em->persist($product);
			$this->em->flush();

			if ($values->preparedAlbumId)
				$product->setGallery($this->em->getRepository(Album::class)->getReference($values->preparedAlbumId));

			//vyrobce
			if (!$product->isVariant() || $product->getIsVariantParent())
				$product->setManufacturer($values->manufacturer ? $this->manufacturerServices->get($values->manufacturer) : null);

			//vatRate
			if ($values->vatRate) {
				$vatRate = $this->vatRateServices->get($values->vatRate);
				$product->setVateRate($vatRate);
			}

			// Texty
			/** @var ProductTexts[] $productTexts */
			$productTexts = $product->getTexts()->toArray();
			$seoData      = $this->seoContainerService->getFormData($values->seo, true);

			// Extra fields
			$ef = $product->getExtraFieldsByKey();

			if (Config::load('product.allowNote')) {
				$addNote = $ef['addNote'] ?? null;

				if (!$addNote) {
					$addNote = new ExtraField('addNote', $values->addNote);
					$product->fillExtraField($addNote);
				}

				$addNote->value = $values->addNote;
				$this->em->persist($addNote);

				foreach ($langValues as $l => $v) {
					$noteTitle = $ef['noteTitle'][$l] ?? null;

					if (!$noteTitle) {
						$noteTitle       = new ExtraField('noteTitle', $v['noteTitle']);
						$noteTitle->lang = $l;
						$product->fillExtraField($noteTitle);
					}

					$noteTitle->value = $v['noteTitle'];
					$this->em->persist($noteTitle);
				}
			}

			foreach ($langValues as $l => $v) {
				if ($v['name'] == '') {
					if (isset($productTexts[$l])) {
						$this->em->remove($productTexts[$l]);
						unset($productTexts[$l]);
					}
					continue;
				}

				if (!$productTexts[$l])
					$productTexts[$l] = new ProductTexts($product, $l);

				$productTexts[$l]->setName($v['name'])
					->setAlias($v['name'])
					->setDescription($v['desc'])
					->setShortDescription($v['shortDesc'])
					->setSeo($seoData[$l] ?? []);

				if (Config::load('enableProductName2'))
					$productTexts[$l]->setName2($v['name2']);

				$this->em->persist($productTexts[$l]);
			}

			$product->setProductTexts($productTexts);

			$this->em->persist($product);
			$this->em->flush();

			//Vlastnosti
			$currentFeatures = [];
			$formFeatures    = [];
			foreach ($product->getFeatureProducts()->toArray() as $fp) {
				/** @var FeatureProduct $fp */
				$currentFeatures[(int) $fp->getFeatureValue()->getId()] = $fp;
			}

			foreach ($form->getHttpData()['features'] as $v)
				$formFeatures[$v] = $v;

			$newFeatures = array_diff_key($formFeatures, $currentFeatures);
			$oldFeatures = array_diff_key($currentFeatures, $formFeatures);

			foreach ($newFeatures as $value) {
				$fp = new FeatureProduct(
					$product,
					$this->em->getReference(FeatureValue::class, $value),
				);

				unset($formFeatures[$value]);
				$this->em->persist($fp);
			}

			foreach ($oldFeatures as $value => $fp) {
				$this->product->getFeatureProducts()->removeElement($fp);
				$this->em->remove($fp);
			}

			//PriceLevels
			if ($this->helper->config->get('enablePriceLevels', false) && (!$va || $va->usePriceLevels)) {
				foreach ($this->helper->getGroupCustomers() as $gc) {
					$remove = true;
					if (isset($values['pricelevel' . $gc->getId()])) {
						$v = $values['pricelevel' . $gc->getId()];
						if ($v != '' && $v !== null) {
							$remove     = false;
							$ppl        = $product->getPriceLevels()->get($gc->getId()) ?: new ProductPriceLevel($product, $gc);
							$ppl->price = $v;
							$this->em->persist($ppl);
							if (!$product->getPriceLevels()->containsKey($gc->getId()))
								$product->getPriceLevels()->add($ppl);
						}
					}

					if ($remove) {
						$ppl = $product->getPriceLevels()->get($gc->getId());
						if ($ppl) {
							$product->getPriceLevels()->remove($gc->getId());
							$this->em->remove($ppl);
						}
					}
				}
			}

			// Kategorie
			if (!$va || $va->isDefault)
				$this->categoryContainerService->saveDataMultipleSite([$product->getId()], (array) $values->categories);

			// Dodavatele
			if ($values->suppliers) {
				$suppliers = $product->getSuppliers();

				foreach ($values->suppliers as $k => $v) {
					$suppId = substr($k, 1);

					if (isset($suppliers[$suppId])) {
						$suppliers[$suppId]->isActive = (int) $v;
						$this->em->persist($suppliers[$suppId]);
					}
				}
			}

			//Tags
			$tagsForRemove = [];
			foreach ($product->getProductTags()->toArray() as $tag)
				$tagsForRemove[$tag->getTag()->type] = $tag;

			foreach ($values->tags as $tagType => $vals) {
				if (!$vals->active)
					continue;

				if (!isset($tagsForRemove[$tagType])) {
					$productTag = new ProductTag($product, $this->em->getReference(Tag::class, $vals['id']));
					$product->addProductTag($productTag);
				} else
					$productTag = $tagsForRemove[$tagType];

				unset($tagsForRemove[$tagType]);
				$productTag->validFrom = $vals['dateFrom'];
				$productTag->validTo   = $vals['dateTo'];

				$this->em->persist($productTag);
			}

			foreach ($tagsForRemove as $tagType => $tag) {
				$product->removeProductTag($tag);
				$this->em->remove($tag);
			}

			if (isset($values->stopImportPrice)) {
				$values->stopImportPrice = $values->stopImportPrice == 1 ? 0 : 1;
				$product->setMoreDataValue('stopImportPrice', $values->stopImportPrice);
			}

			if (isset($values->stopImportQuantity))
				$product->setMoreDataValue('stopImportQuantity', $values->stopImportQuantity);

			if (isset($values->stopImportQuantity))
				$product->setMoreDataValue('stopImportQuantity', $values->stopImportQuantity);

			// dokumenty
			$this->documentContainer->saveData($values['documents'], $product);

			// souvisejici produkty
			if (Config::load('allowRelatedProducts')) {
				$this->relatedProductContainer->saveData($form->getHttpData(), $product);
			}

			// Doprava
			if (Config::load('product.allowModifySpedition')) {
				$this->speditionContainer->saveData($values->speditions, $product);
			}

			// Pridani dodavatelů
			$suppliersContainer = $form->getHttpData()['suppliersContainer'];
			$suppliersToAdd = isset($suppliersContainer) && is_array($suppliersContainer) && count($suppliersContainer) > 0 ? $suppliersContainer : [];
			foreach ($suppliersToAdd as $k => $row) {
				if (isset($row['selected']) || $row['selected'] === 'on') {
					$supplier = $this->supplierServices->get((int) explode('_', $k)[1]);
					$availability = $this->availabilityService->getEr()->find((int) $row['availability']);
					if (!$supplier) {
						continue;
					}
					$this->productSupplierContainer->saveData($row, $supplier, $product, $availability);
				}
			}

			$this->em->persist($product);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $product;
			$this->eventDispatcher->dispatch($event, ProductForm::class . '::formSuccess');

			$this->availabilityService->updateAvailabilityByQuantity($product);
			$this->em->flush();
			$this->em->commit();

			if ($product->isVariant()) {
				if ($product->getIsVariantParent()) {
					$product->isVariant()->variantName = $values->variantName;
					$this->em->persist($product->isVariant());

					if ($product->getId()) {
						$this->productServices->updateVariantCategories($product, (int) $product->isVariant()->getVariantId());
					}
				} else {
					foreach ($values->variantUse as $k => $v) {
						if (in_array($k, ['product']) || $product->isVariant()->isDefault == 0 && $k == 'variantName')
							continue;

						$product->isVariant()->$k = $v;
					}

					$this->em->persist($product->isVariant());
				}

				$this->em->flush();
			}

			$form->addCustomData('productId', $product->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.productForm.productSaved');
			$this->getPresenter()->redrawControl('flashes');
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			$form->addError($e->getMessage());
			$this->getPresenter()->redrawControl('flashes');
			$this->redrawControl('formErrors');

			return false;
		}

		return true;
	}

	protected function createComponentImagesZone()
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->product && $this->product->isVariant()) {
			$control->onBeforeUpdate[] = function($id, $v) {
				$variant = $this->product->isVariant();

				$variant->defaultImage = (int) $id;
				$this->em->persist($variant);
			};

			$control->onPrepareDefaultImage[] = function(Image $image, &$arr) {
				$arr['variantImage'] = $image->getId() === $this->product->isVariant()->defaultImage ? 1 : 0;

				return $arr;
			};

			$control->addCustomFieldLatteBefore(__DIR__ . '/ProductFormImageZoneCustomFields.latte');
		}

		if ($this->product && $this->product->getGallery())
			$control->setAlbum($this->product->getGallery()->getId());

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */
			$dataRaw = $this->httpRequest->getPost('formData');
			$data    = [];

			if (!$dataRaw)
				return false;

			foreach ($dataRaw as $row) {
				$data[$row['name']] = $row['value'];
			}

			if (!$this->product || !$this->product->getGallery()) {
				$album = new Album(UPLOADS_PATH . '/products');
				$this->em->persist($album)->flush();
				$control->setAlbum($album->getId());
			}

			if ($this->product && isset($album)) {
				$this->product->setGallery($album);
				$this->em->persist($this->product)->flush();
			}
		};

		return $control;
	}

	/*******************************************************************************************************************
	 * ========================  GET / SET
	 */

	public function setProduct($id)
	{
		$this->product = $this->em->getRepository(Product::class)->createQueryBuilder('p')
			->addSelect('pt, m, cp, cd, fp, pTags, pTagsTag, s, vr, fpf, fpv, fpft, fpvt, av, avAf, ps, isV')
			->leftJoin('p.productTexts', 'pt')
			->leftJoin('p.manufacturer', 'm')
			->leftJoin('p.categoryProducts', 'cp')
			->leftJoin('p.idCategoryDefault', 'cd')
			->leftJoin('p.featureProducts', 'fp')
			->leftJoin('p.productTags', 'pTags')
			->leftJoin('pTags.tag', 'pTagsTag')
			->leftJoin('p.suppliers', 's')
			->leftJoin('p.vatRate', 'vr')
			->leftJoin('fp.featureValue', 'fpv')
			->leftJoin('fpv.featureValueTexts', 'fpvt')
			->leftJoin('fpv.feature', 'fpf')
			->leftJoin('fpf.featureTexts', ' fpft')
			->leftJoin('p.availability', 'av')
			->leftJoin('p.availabilityAfterSoldOut', 'avAf')
			->leftJoin('p.sites', 'ps')
			->leftJoin('p.isVariant', 'isV')
			->where('p.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();

		if ($this->product) {
			/** @var Product $prod */
			$prod       = $this->product;
			$parentProd = null;
			$variant    = null;

			if ($prod->isVariant()) {
				$variants = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
					->where('pv.variantId = :variantId')->setParameter('variantId', $prod->isVariant()->getVariantId())
					->getQuery()->getResult();

				$prod->variants = new ArrayCollection($variants);
				$this->product  = $prod;

				// kontrola hlavniho produktu
				if (!$prod->getVariantParent()) {
					$prod->isVariant()->isDefault = 1;
					$this->em->persist($prod->isVariant());
					$this->em->flush($prod->isVariant());
					$this->getPresenter()->redirect('this');
				}
			}

			if ($prod->isVariant() && !$prod->getIsVariantParent()) {
				$parentProd = $prod->getVariantParent();
				$variant    = $prod->isVariant();
			}

			$form = $this['form'];
			$d    = [
				'isPublished'      => $prod->isPublished,
				'ean'              => $prod->ean,
				'quantity'         => $prod->quantity,
				'price'            => $variant && !$variant->usePrice ? $parentProd->price : $prod->price,
				'retailPrice'      => $variant && !$variant->useRetailPrice ? $parentProd->retailPrice : $prod->retailPrice,
				'purchasePrice'    => $variant && !$variant->usePurchasePrice ? $parentProd->purchasePrice : $prod->purchasePrice,
				'code1'            => $prod->code1,
				'code2'            => $prod->code2,
				'discountDisabled' => ($variant && !$variant->useDiscountDisabled ? $parentProd->discountDisabled : $prod->discountDisabled) == 1 ? 0 : 1,
				'condition'        => $prod->condition,
			];

			if ($prod->isVariant() && $prod->getIsVariantParent()) {
				$d['variantName'] = $prod->isVariant()->variantName;
			}

			if (isset($this['form']['disablePickUpSpedition']))
				$d['disablePickUpSpedition'] = $prod->disablePickUpSpedition;
			if (isset($this['form']['isOversize']))
				$d['isOversize'] = $prod->isOversize;
			if (isset($this['form']['disableStackInCart']))
				$d['disableStackInCart'] = $prod->disableStackInCart;
			if (isset($this['form']['isDiscount']))
				$d['isDiscount'] = $prod->isDiscount;

			if ($prod->getAvailability() && array_key_exists($prod->getAvailability()->getId(), $form['availability']->getItems()))
				$form['availability']->setDefaultValue($prod->getAvailability()->getId());

			if ($prod->availabilityAfterSoldOut && array_key_exists($prod->availabilityAfterSoldOut->getId(), $form['availabilityAfterSoldOut']->getItems()))
				$form['availabilityAfterSoldOut']->setDefaultValue($prod->availabilityAfterSoldOut->getId());

			if (Config::load('product.allowAssort'))
				$d['isAssort'] = $prod->isAssort ? 1 : 0;

			if (Config::load('pseudoWarehouse'))
				$form['unlimitedQuantity']->setDefaultValue($variant && !$variant->useUnlimitedQuantity ? $parentProd->unlimitedQuantity : $prod->unlimitedQuantity);
			if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
				$form['categoryGiftsAllowed']->setDefaultValue($variant && !$variant->useCategoryGiftsAllowed ? $parentProd->isCategoryGiftsAllowed() : $prod->isCategoryGiftsAllowed());

			if (class_exists(ImportExtension::class)) {
				$stopImportPrice = ($variant && !$variant->useImportPrice ? $parentProd : $prod)->getMoreDataValue('stopImportPrice') == 1 ? 0 : 1;
				$form['stopImportPrice']->setValue($stopImportPrice);
			}

			if (Config::load('product.allowStopImportQuantity')) {
				$stopImportQuantity = ($variant ? $parentProd : $prod)->getMoreDataValue('stopImportQuantity');
				$form['stopImportQuantity']->setValue($stopImportQuantity);
			}

			if (Config::load('product.loadQuantityOnStock') && class_exists(EshopStockExtension::class)) {
				$qb = $this->em->getRepository(SupplyProduct::class)->createQueryBuilder('sp')
					->select('COUNT(sp)')
					->join('sp.product', 'p')
					->where('p.id = :id AND sp.order IS NULL AND sp.writtenOffDate IS NULL')
					->setParameter('id', (int) $id);
				$form['quantityOnStock']->setValue($qb->getQuery()->getSingleResult());
			}

			// Manufacturer
			if ($prod->getManufacturer() && array_key_exists($prod->getManufacturer()->getId(), $form['manufacturer']->getItems()))
				$form['manufacturer']->setDefaultValue($prod->getManufacturer()->getId());

			//Suppliers
			if ($prod->getSuppliers()->count()) {
				$formSupp = $form->addContainer('suppliers');

				foreach ($prod->getSuppliers()->toArray() as $sup) {
					$formSupp->addBool('s' . $sup->getSupplier()->getId(), '')
						->setDefaultValue($sup->isActive);
				}
			}

			$this->onAnchor[] = function() use ($prod) {
				$this->template->suppliers = [];
				foreach ($prod->getSuppliers()->toArray() as $productSupplier) {
					/** @var ProductSupplier $productSupplier */
					$this->template->suppliers[$productSupplier->getSupplier()->getId()] = [
						'name'       => $productSupplier->getSupplier()->name,
						'code'       => $productSupplier->code,
						'quantity'   => $productSupplier->quantity,
						'supplierId' => $productSupplier->getSupplier()->getId(),
					];
				}
			};

			// Extra Fields
			$ef = $prod->getExtraFieldsValues();
			if (Config::load('product.allowNote')) {
				$d['addNote']   = $ef['addNote'] ?? false;
				$d['noteTitle'] = $ef['noteTitle'] ?? null;
			}

			$form->setDefaults($d);

			// Texty
			$mTexts  = [];
			$seoData = [];

			$prodTexts   = $prod->getProductTexts()->toArray();
			$parentTexts = [];
			if ($variant) {
				$parentTexts = $this->em->getRepository(ProductTexts::class)->createQueryBuilder('pt', 'pt.lang')
					->where('pt.id = :id')->setParameter('id', $parentProd->getId())
					->getQuery()->getResult();
			}

			foreach ($prodTexts as $lang => $texts) {
				foreach (['name' => 'name', 'shortDesc' => 'shortDescription', 'desc' => 'description'] as $k => $v) {
					$mTexts[$k][$lang] = $variant && !$variant->{'use' . ucfirst($v)} ? $parentTexts[$lang]->$v : $texts->$v;
				}

				if (Config::load('enableProductName2'))
					$mTexts['name2'][$lang] = $variant && !$variant->useName2 ? $parentTexts[$lang]->name2 : $texts->name2;

				foreach ($texts->getSeo() as $k => $v)
					$seoData[$k][$lang] = $v;
			}
			$form->setDefaults($mTexts);
			$this->seoContainerService->setDefaults($form['seo'], $seoData);

			if ($prod->getVateRate() && array_key_exists($prod->getVateRate()->getId(), $form['vatRate']->getItems()))
				$form['vatRate']->setDefaultValue($prod->getVateRate()->getId());

			// Vlastnosti
			//			$fd = [
			//				'features'     => [],
			//				'featuresList' => [],
			//			];
			//
			//			foreach ($prod->getFeatureProducts()->toArray() as $fp) {
			//				/** @var FeatureProduct $fp */
			//				$featureId                        = $fp->getFeature()->getId();
			//				$featureValueId                   = $fp->getFeatureValue()->getId();
			//				$fd['features']['f' . $featureId] = $featureValueId;
			//				$fd['featuresList'][]             = $featureId;
			//			}
			//			$form->setDefaults($fd);

			//set hodnoty variant

			//set kategorie
			$allCategoriesId = array_map(function($c) { return $c->getCategory()->getId(); }, $prod->getCategoryProducts()->toArray());
			$this->categoryContainerService->setDefaultsMultipleSite($form['categories'], $prod->sites->toArray(), $allCategoriesId);

			//set tags
			$tags = [];
			foreach ($prod->getProductTags()->toArray() as $t) {
				$tags[$t->getTag()->type] = [
					'active'   => 1,
					'dateFrom' => $t->validFrom,
					'dateTo'   => $t->validTo,
					'id'       => $t->getTag()->getId(),
				];
			}
			$this->tagsContainerService->setDefaults($form['tags'], $tags);

			// priceLevel
			$prodPriceLevels = $prod->getPriceLevels()->toArray();
			if ($variant && !$variant->usePriceLevels) {
				$prodPriceLevels = [];
				foreach ($this->em->getRepository(ProductPriceLevel::class)->createQueryBuilder('pl',)
					         ->where('pl.productId = :id')->setParameter('id', $parentProd->getId())
					         ->getQuery()->getResult() as $row)
					$prodPriceLevels[$row->getGroupId()->getId()] = $row;
			}
			if ($this->helper->config->get('enablePriceLevels', false)) {
				foreach ($this->helper->getGroupCustomers() as $gc) {
					if (isset($form['pricelevel' . $gc->getId()]) && isset($prodPriceLevels[$gc->getId()])) {
						$form['pricelevel' . $gc->getId()]->setDefaultValue($prodPriceLevels[$gc->getId()]->price);
					}
				}
			}

			$this->monitor(IPresenter::class, function() use ($prod, $form) {
				$documentsData = [];
				foreach ($this->productDocuments->getByProduct($prod->getId()) as $document) {
					$documentsData[] = [
						'id'         => $document->getId(),
						'fileName'   => $document->name,
						'filePath'   => $document->file,
						'lang'       => $document->lang,
						'removeLink' => $this->link('removeDocument', $document->getId()),
					];
				}
				$this->documentContainer->setDefaults($form['documents'], $documentsData);

				if (Config::load('allowRelatedProducts')) {
					$this->relatedProductContainer->setDefaults($form['relatedProducts'], $prod->getRelatedProducts()->toArray());
				}

				if (Config::load('product.allowModifySpedition', false))
					$this->speditionContainer->setDefaults($form['speditions'], $prod->getSpeditions()->toArray());
			});

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->product), ProductForm::class . '::setProduct');
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	protected function getFeatures(): array
	{
		if ($this->cFeatures === null) {

			$this->cFeatures = [];
			foreach ($this->featureServices->getEr()->createQueryBuilder('f')
				         ->select('f.id, ft.name')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->where('f.isPublished = 1')
				         ->orderBy('f.position')
				         ->getQuery()->getResult() as $v)
				$this->cFeatures[$v['id']] = trim($v['name']);
		}

		return $this->cFeatures;
	}

	protected function getFeatureValues(?int $id = null): array
	{
		if ($this->cFeatureValues === null) {
			$this->cFeatureValues = [];

			foreach ($this->featureServices->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
				         ->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->andWhere('fv.isPublished = 1')
				         ->orderBy('fv.position')
				         ->getQuery()->getResult() as $v)
				$this->cFeatureValues[$v['featureId']][] = [
					'id'   => $v['id'],
					'name' => trim($v['name']),
				];
		}

		return $id ? $this->cFeatureValues[$id] ?? [] : $this->cFeatureValues;
	}

	protected static function allowSaleDiscountCouponsAsProducts(): bool
	{
		return Parameters::load('eshopSales.autoGenerateDiscountCoupon.enable', false) && class_exists('EshopSales\DI\EshopSalesExtension');
	}
}
