<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Seller;

use Core\AdminModule\Model\Sites;
use Core\Model\Countries;
use Core\Model\Helpers\FormHelper;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Event\SellerEvent;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\AdminModule\Model\Event\SellerFormEvent;
use EshopCatalog\AdminModule\Model\Event\SetSellerFormDataEvent;
use EshopCatalog\Model\Entities\Seller;
use EshopCatalog\Model\Entities\SellerBankAccount;
use EshopCatalog\Model\Entities\SellerInSite;
use Nette\Application\IPresenter;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class SellerForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	/** @var Seller */
	protected ?Seller $seller = null;

	/** @var Sellers */
	protected Sellers $sellersService;

	/** @var Sites */
	protected Sites $sitesService;

	/** @var Countries */
	protected Countries $countries;

	public function __construct(Sellers $sellers, Sites $sites, Countries $countries)
	{
		$this->sellersService = $sellers;
		$this->sitesService   = $sites;
		$this->countries      = $countries;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$sites       = $this->sitesService->getOptionsForSelect();
		$sitesInUse  = $this->sellersService->getSitesInUse();
		$sellerSites = [];

		if ($this->seller) {
			foreach ($this->seller->getSites() as $v)
				$sellerSites[$v->getSite()->getIdent()] = $v;
		}

		foreach ($sitesInUse as $k => $v) {
			if (isset($sellerSites[$k])) {
				unset($sitesInUse[$k]);
			} else {
				$sites[$k] .= ' (' . $v['sellerName'] . ')';
			}
		}

		$form->addText('title', 'eshopCatalog.seller.title')
			->setMaxLength(255)
			->setRequired();
		$form->addText('name', 'eshopCatalog.seller.name')
			->setMaxLength(255)
			->setRequired();
		$form->addText('ic', 'eshopCatalog.seller.ic')
			->setMaxLength(12)
			->setNullable();
		$form->addText('dic', 'eshopCatalog.seller.dic')
			->setMaxLength(14)
			->setNullable();
		$form->addText('email', 'eshopCatalog.seller.email')
			->setMaxLength(255)
			->setRequired();
		$form->addText('phone', 'eshopCatalog.seller.phone')
			->setMaxLength(255);
		$form->addBool('sendOrder', 'eshopCatalog.seller.sendOrder')
			->setDefaultValue(1);
		$form->addBool('sendStatus', 'eshopCatalog.seller.sendStatus')
			->setDefaultValue(1);
		$form->addSelect('sendEmailType', 'eshopCatalog.seller.sendEmailType', [
			Seller::EMAIL_TYPE_SELLER => 'eshopCatalog.seller.sendEmailTypeList.seller',
			Seller::EMAIL_TYPE_SITE   => 'eshopCatalog.seller.sendEmailTypeList.site',
		]);

		$form->addCheckboxList('sites', 'eshopCatalog.seller.eshops', $sites)
			->setDisabled(array_keys($sitesInUse));

		foreach (['', '2'] as $k) {
			$container = $form->addContainer('address' . $k);
			$container->addText('address', 'eshopCatalog.seller.street')
				->setMaxLength(255)
				->setNullable();
			$container->addText('city', 'eshopCatalog.seller.city')
				->setMaxLength(255)
				->setNullable();
			$container->addText('postal', 'eshopCatalog.seller.postal')
				->setMaxLength(255)
				->setNullable();
			$container->addSelect('country', 'eshopCatalog.seller.country', $this->countries->getAllNameColumn());
		}

		$container = $form->addContainer('bankAccount');
		$container->addText('bankName', 'eshopCatalog.sellerBankAccount.bankName')
			->setMaxLength(255)
			->setNullable();
		$container->addText('numberPart1', 'eshopCatalog.sellerBankAccount.numberPart1')
			->setMaxLength(255)
			->setNullable();
		$container->addText('numberPart2', 'eshopCatalog.sellerBankAccount.numberPart2')
			->setMaxLength(255)
			->setNullable();
		$container->addText('swift', 'eshopCatalog.sellerBankAccount.swift')
			->setMaxLength(255)
			->setNullable();
		$container->addText('iban', 'eshopCatalog.sellerBankAccount.iban')
			->setMaxLength(255)
			->setNullable();
		$container->addText('note', 'eshopCatalog.sellerBankAccount.note')
			->setMaxLength(255)
			->setNullable();

		$form->addContainer('extendedContainers');

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		$this->monitor(IPresenter::class, function() use ($form) {
			$sellerFormEvent           = new SellerFormEvent($form);
			$sellerFormEvent->template = $this->template;
			$this->eventDispatcher->dispatch($sellerFormEvent, 'eshopCatalog.createSellerForm');
		});

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$seller      = $this->id ? $this->sellersService->get($this->id) : new Seller($values->title, $values->name);
			$sites       = $seller->getSites()->toArray();
			$bankAccount = $seller->getBankAccounts()->count() ? $seller->getBankAccounts()->first() : new SellerBankAccount($seller);

			FormHelper::fillEntityByValues($seller, $values);
			FormHelper::fillEntityByValues($bankAccount, $values->bankAccount);

			foreach (array_diff(array_keys($sites), $values->sites) as $v) {
				$this->em->remove($sites[$v]);
			}

			foreach (array_diff($values->sites, array_keys($sites)) as $v) {
				$inSite = new SellerInSite($seller, $this->sitesService->getReference($v));
				$this->em->persist($inSite);
			}

			foreach (['', 2] as $k) {
				foreach (['address', 'city', 'postal', 'country'] as $v) {
					$seller->{$v . $k} = $values->{'address' . $k}->$v;
				}
			}
			$seller->setIc($values->ic ?: null);
			$seller->setDic($values->dic ?: null);

			$this->eventDispatcher->dispatch(new SellerEvent($seller, $values), 'eshopCatalog.sellerBeforeSave');

			$this->em->persist($seller)->persist($bankAccount)->flush();
			$form->addCustomData('sellerId', $seller->getId());
			$this->getPresenter()->flashMessageSuccess('default.saved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setSeller(int $id): void
	{
		$this->id     = $id;
		$this->seller = $seller = $this->sellersService->get($id);

		if (!$seller)
			throw new InvalidArgumentException();

		FormHelper::fillContainerByEntity($this['form'], $seller);
		if ($seller->getBankAccounts()->count())
			FormHelper::fillContainerByEntity($this['form']['bankAccount'], $seller->getBankAccounts()->first());

		$form = $this['form'];
		$d    = [
			'sites' => $seller->getSitesIdent(),
		];

		$this->eventDispatcher->dispatch(new SetSellerFormDataEvent($form, $seller), 'eshopCatalog.setSellerFormData');

		foreach (['', 2] as $k) {
			foreach (['address', 'city', 'postal'] as $v) {
				$d['address' . $k][$v] = $seller->{$v . $k};
			}

			if ($seller->{'country' . $k}) {
				$country = $this->countries->get($seller->{'country' . $k});
				if ($country)
					$d['address' . $k]['country'] = $country->getId();
			}
		}

		$form->setDefaults($d);
	}

	public function getSeller(): ?Seller { return $this->seller; }
}
