<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductSpedition;
use EshopOrders\AdminModule\Model\Speditions;
use Nette\Utils\ArrayHash;

class ProductSpeditionContainer
{
	protected EntityManagerDecorator $em;

	protected Speditions $speditions;

	protected Products $products;

	public function __construct(EntityManagerDecorator $em, Speditions $speditions, Products $products)
	{
		$this->em         = $em;
		$this->speditions = $speditions;
		$this->products   = $products;
	}

	public function getContainer(bool $isMultiple = false)
	{
		$container = new BaseContainer();

		if (!Config::load('product.allowModifySpedition', false))
			return $container;

		if ($isMultiple) {
			$container->addSelect('disableApply', 'eshopCatalog.productSpeditionContainer.disableApply', [
				'override'      => 'eshopCatalog.productSpeditionContainer.override',
				'applyDisabled' => 'eshopCatalog.productSpeditionContainer.applyDisabled',
			]);
		}

		$containerForSpeditions = new BaseContainer();

		foreach ($this->speditions->getForSelectOption() as $id => $name) {
			$container2 = new BaseContainer();
			$container2->addCustomData('name', $name);
			$container2->addBool('disabled', $name);
			$container2->addHidden('id', (string) $id);

			$containerForSpeditions->addComponent($container2, (string) $id);
		}
		$container->addComponent($containerForSpeditions, 'speditions');

		$container->addCustomData('template', __DIR__ . '/ProductSpeditionContainer.latte');

		return $container;
	}

	/**
	 * @param ArrayHash     $values
	 * @param Product|array $products
	 */
	public function saveData(ArrayHash $values, $products): void
	{
		if ($products instanceof Product)
			$products = [$products];

		$speditions   = (array) $values->speditions;
		$disableApply = $values->disableApply ?? 'override';

		$canRemoveDisable = in_array($disableApply, ['override']);
		$canAddDisable    = in_array($disableApply, ['override', 'applyDisabled']);

		foreach ($products as $product) {
			if ($product instanceof Product === false)
				$product = $this->products->getReference((int) $product);

			$currentSpeditions = $product->getSpeditions()->toArray();

			if ($canRemoveDisable) {
				foreach (array_diff_key($currentSpeditions, $speditions) as $entity)
					$this->em->remove($entity);
			}

			foreach ($speditions as $id => $data) {
				/** @var ProductSpedition $entity */
				$entity = $product->getSpeditions()->get($id);

				if ($data->disabled === 0) {
					if ($entity && $canRemoveDisable)
						$this->em->remove($entity);

					continue;
				}

				if ($canAddDisable) {
					if ($entity) {
						$entity->speditionDisabled = $data->disabled;
					} else {
						$entity                    = new ProductSpedition($product, $this->speditions->getReference($id));
						$entity->speditionDisabled = $data->disabled;
					}
				}

				$this->em->persist($entity);
			}
		}
	}

	public function setDefaults(BaseContainer &$container, array $values)
	{
		$d = [
			'speditions' => [],
		];

		foreach ($values as $id => $entity) {
			/** @var ProductSpedition $entity */
			$d['speditions'][$id] = [
				'disabled' => $entity->speditionDisabled,
			];
		}

		$container->setDefaults($d);
	}
}
