<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Countries;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\DaoHelper;
use EshopCatalog\Model\Entities\Seller;
use EshopCatalog\FrontModule\Model\Dao;
use Doctrine\ORM\Query\Expr\Join;

class Sellers extends BaseFrontEntityService
{
	protected $entityClass = Seller::class;

	/** @var array */
	protected $cSellersForSite = [];

	/** @var Countries */
	protected Countries $countries;

	public function __construct(Countries $countries)
	{
		$this->countries = $countries;
	}

	public function getSellerForSite(string $siteIdent): ?Dao\Seller
	{
		if (!isset($this->cSellersForSite[$siteIdent])) {
			$this->cSellersForSite[$siteIdent] = false;

			$seller = $this->getEr()->createQueryBuilder('s')
					->addSelect('bank')
					->innerJoin('s.sites', 'sites', Join::WITH, 'sites.site = :site')
					->leftJoin('s.bankAccounts', 'bank')
					->setParameter('site', $siteIdent)
					->getQuery()->setMaxResults(1)->getArrayResult()[0] ?? null;

			if ($seller) {
				$seller['bankAccount'] = $seller['bankAccounts'][0] ?? null;
				unset($seller['bankAccounts']);
				$this->cSellersForSite[$siteIdent] = $this->fillDao($seller);
			}
		}

		return $this->cSellersForSite[$siteIdent] ?: null;
	}

	protected function fillDao(array $data): Dao\Seller
	{
		$dao         = new Dao\Seller((int) $data['id']);
		$bankAccount = new Dao\BankAccount();
		$countries   = $this->countries->getDao();

		if ($data['bankAccount']) {
			$bankAccount = DaoHelper::fillDaoFromArray($data['bankAccount'], $bankAccount);
			unset($data['bankAccount']);
		}

		/** @var Dao\Seller $dao */
		$dao = DaoHelper::fillDaoFromArray($data, $dao, ['country', 'country2']);
		if ($data['address'])
			$dao->street = $data['address'];
		if ($data['address2'])
			$dao->street2 = $data['address2'];
		$dao->sendOrderToSeller   = (bool) $data['sendOrder'];
		$dao->sendStatusToSeller  = (bool) $data['sendStatus'];
		$dao->sendSellerEmailType = $data['sendEmailType'];
		$dao->setBankAccount($bankAccount);

		$dao->country  = $countries[$data['country']] ?? null;
		$dao->country2 = $countries[$data['country2']];

		return $dao;
	}
}
