<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\DaoHelper;
use Core\Model\Sites;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\Tag;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Entities\Spedition;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;

/**
 * Class Tags
 * @package EshopCatalog\FrontModule\Model
 */
class Tags extends BaseFrontEntityService
{
	/** @var Cache */
	protected $productsCache;

	const CACHE_NAMESPACE = 'eshopCatalogTags';

	protected $entityClass = Tag::class;

	protected Sites $sites;

	/** @var array */
	protected $cTags;

	protected ?float $cFreeFrom = null;

	protected ?array $freeFromArr = null;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS   => [self::CACHE_NAMESPACE, Products::CACHE_NAMESPACE],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(Sites $sites)
	{
		$this->sites = $sites;
	}

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function getProductsCache()
	{
		if ($this->productsCache === null)
			$this->productsCache = new Cache($this->cacheStorage, Products::CACHE_NAMESPACE);

		return $this->productsCache;
	}

	/**
	 * @param Dao\Product|Dao\Product[] $product
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadTagsToProduct(&$product)
	{
		$products = is_array($product) ? $product : [$product];

		$tags = $this->getIdsForProducts();

		$tagForPrice  = Config::load('tagWhenPriceIsLowerThenRetail');
		$freeDelivery = $this->get('freeDelivery');

		foreach ($products as &$prod) {
			$freeDeliveryMinimum = $this->getFreeDeliveryMinimumPrice($prod->disabledSpeditions ?: []);

			if (isset($tags[$prod->getId()])) {
				foreach ($tags[$prod->getId()] as $tagId)
					$prod->tags[$tagId] = $this->get($tagId);
			}

			if ($tagForPrice && $prod->getPrice() < $prod->getRetailPrice())
				$prod->tags[$tagForPrice] = $this->get($tagForPrice);

			if ($freeDeliveryMinimum !== null && $freeDelivery && $prod->getPrice() >= $freeDeliveryMinimum) {
				$freeDelivery->isAuto       = true;
				$prod->tags['freeDelivery'] = $freeDelivery;
			}
		}
	}

	public function getIdsForProducts(): array
	{
		$result = [];
		$now    = (new DateTime())->format('Y-m-d H:i:00');

		foreach ($this->getEr()->createQueryBuilder('t')
			         ->select('t.id, t.type, pt.validFrom, pt.validTo, IDENTITY(pt.product) as product')
			         ->join(ProductTag::class, 'pt', 'WITH', 'pt.tag = t')
			         ->andWhere('pt.validFrom IS NULL OR pt.validFrom <= :now')
			         ->andWhere('pt.validTo IS NULL OR pt.validTo >= :now')
			         ->setParameter('now', $now)
			         ->getQuery()->getArrayResult() as $row) {
			$result[$row['product']][] = $row['type'];
		}

		return $result;
	}

	/**
	 * @param int|string $id
	 *
	 * @return Dao\Tag|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		return $this->getAll()[$id] ?? null;
	}

	/**
	 * @return Dao\Tag[]
	 */
	public function getAll(): array
	{
		if ($this->cTags === null) {
			$this->cTags = [];

			foreach ($this->getEr()->createQueryBuilder('t')
				         ->select('t.id, t.type, t.image, tt.name, t.color, t.bgColor')
				         ->join('t.texts', 'tt', 'WITH', 'tt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->getQuery()->getArrayResult() as $row) {
				$tag                       = $this->fillDao($row);
				$this->cTags[$row['type']] = $tag;
				$this->cTags[$row['id']]   = $tag;
			}
		}

		return $this->cTags ?? [];
	}

	/**
	 * @param array $tag
	 *
	 * @return Dao\Tag
	 */
	protected function fillDao($tag)
	{
		/** @var Dao\Tag $dao */
		$dao            = DaoHelper::fillDaoFromArray($tag, new Dao\Tag());
		$dao->textColor = $tag['color'];
		$dao->isAuto    = false;

		return $dao;
	}

	protected function loadFreeFromArr(): array
	{
		if ($this->freeFromArr === null) {
			$this->freeFromArr = [];

			foreach ($this->em->getRepository(Spedition::class)->createQueryBuilder('s')
				         ->select('s.id, s.freeFrom')
				         ->innerJoin(PaymentSpedition::class, 'ps', Query\Expr\Join::WITH, 'ps.spedition = s.id')
				         ->innerJoin('ps.sites', 'sites', Query\Expr\Join::WITH, 'sites.ident = :ident')
				         ->where('s.freeFrom > 0')->andWhere('s.freeFrom IS NOT NULL')
				         ->setParameter('ident', $this->sites->getCurrentSite()->getIdent())
				         ->groupBy('s.id')
				         ->getQuery()->getArrayResult() as $row)
				$this->freeFromArr[$row['id']] = (float) $row['freeFrom'];
		}

		return $this->freeFromArr;
	}

	protected function getFreeDeliveryMinimumPrice(array $skipSpeditions = []): ?float
	{
		if (!empty($skipSpeditions)) {
			$min = 99999999;
			$skipSpeditions = array_flip($skipSpeditions);
			foreach ($this->loadFreeFromArr() as $id => $val) {
				if (isset($skipSpeditions[$id]))
					continue;

				if ($val < $min)
					$min = $val;
			}

			return $min;
		}

		if ($this->cFreeFrom === null) {
			if ($this->loadFreeFromArr())
				$this->cFreeFrom = min($this->loadFreeFromArr());
		}

		return $this->cFreeFrom;
	}
}
