<?php declare(strict_types = 1);

namespace EshopCatalog\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\AdminModule\Model\Products as AdminProducts;
use EshopCatalog\Model\Config;
use Nette\Application\LinkGenerator;
use Nette\Localization\ITranslator;
use Nette\Utils\Validators;

/**
 * Class ProductsController
 * @Path("/product")
 */
class ProductsController extends BaseController
{
	protected Products $productsDaoService;

	protected ProductsFacade $productsFacade;

	protected AdminProducts $products;

	protected ITranslator $translator;

	protected LinkGenerator $linkGenerator;

	public function __construct(Products $productsDaoService, ProductsFacade $productsFacade, AdminProducts $products, ITranslator $translator, LinkGenerator $linkGenerator)
	{
		$this->productsDaoService = $productsDaoService;
		$this->productsFacade     = $productsFacade;
		$this->products           = $products;
		$this->translator         = $translator;
		$this->linkGenerator      = $linkGenerator;

		ProductsFacade::setMode(ProductsFacade::MODE_CHECKOUT);
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return array
	 * @Path("/")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="start", type="int", description="start", required=false, in="query"),
	 *      @RequestParameter(name="limit", type="int", description="limit", required=false, in="query"),
	 *      @RequestParameter(name="filter", type="string", description="filter", required=false, in="query")
	 * })
	 */
	public function index(ApiRequest $request, ApiResponse $response)
	{
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', 0);
		$sort   = $request->getQueryParam('sort', 'ASC');

		$productIds = $this->productsDaoService->getProductsIdAll($offset, $limit, $sort);

		return $productIds ? array_values($this->productsFacade->getProducts($productIds)) : [];
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return Product[]
	 * @Path("/get/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int")
	 * })
	 */
	public function product(ApiRequest $request, ApiResponse $response)
	{
		$id      = $request->getParameter('id');
		$product = $this->productsFacade->getProduct($id);

		if (!$product) {
			$this->sendError($response, "Product with this id doesn´t exist");
		}

		return [$product];
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return array
	 * @Path("/search/{query}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="query", type="string", required=true),
	 *      @RequestParameter(name="start", type="int", description="start", required=false, in="query"),
	 *      @RequestParameter(name="limit", type="int", description="limit", required=false, in="query"),
	 *      @RequestParameter(name="filter", type="string", description="filter", required=false, in="query"),
	 *      @RequestParameter(name="sort", type="string", description="sort", required=false, in="query")
	 * })
	 */
	public function search(ApiRequest $request, ApiResponse $response)
	{
		$query  = urldecode($request->getParameter('query'));
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', null);
		$filter = $request->getQueryParam('filter', null);
		$sort   = $request->getQueryParam('sort', null);

		// TODO do modelu
		$qb = $this->productsDaoService->getEr()->createQueryBuilder('p', 'p.id')
			->select('p.id, pt.name, p.price, p.code1, p.code2, p.ean, p.isPublished, vat.rate as vatRate, i.path, i.filename, cover.path as coverPath, cover.filename as coverFilename')
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->join('p.vatRate', 'vat')
			->leftJoin('p.gallery', 'g')
			->leftJoin('g.images', 'cover', Query\Expr\Join::WITH, 'cover.isCover = 1')
			->leftJoin('g.images', 'i', Query\Expr\Join::WITH, 'i.position = 0')
			->orderBy('pt.name')
			->setParameter('lang', $this->translator->getLocale());

		if ($query) {
			$qb->orWhere('pt.name LIKE :term')
				->orWhere('p.code1 LIKE :term')
				->orWhere('p.code2 LIKE :term')
				->orWhere('p.ean LIKE :term')
				->orWhere('p.id LIKE :term')
				->setParameter('term', '%' . $query . '%');
		}

		$qb->andWhere('p.isDeleted = 0');
		$result              = $qb->getQuery()->getArrayResult();
		$canCacheProductsUrl = Config::load('canCacheProductsUrl');
		$useOldUrls          = Config::load('useOldUrls');
		$locale              = $this->translator->getLocale();
		foreach ($result as &$row) {
			$link = '#';
			if (!$canCacheProductsUrl && $this->linkGenerator) {
				$linkParams = [
					'id'     => $row['id'],
					'locale' => $locale,
				];

				//				if ($useOldUrls && $product->variantOf && $product->variantOf != $product->getId()) {
				//					$linkParams['variant'] = $product->getId();
				//				}

				//				$link = $this->linkGenerator->link('EshopCatalog:Front:Default:product', $linkParams); TODO @Roman trva to vygenerovat hrozne dlouho, proc?
			}
			$row['link'] = $link;

			if (!Validators::isNone($row['coverPath']) && !Validators::isNone($row['coverFilename'])) {
				$row['path']     = $row['coverPath'];
				$row['filename'] = $row['coverFilename'];
			}
			unset($row['coverPath'], $row['coverFilename']);
		}

		return $result;
	}
}
