<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use Nette\Utils\ArrayHash;

class CartAddForm extends BaseControl
{
	const VIEWTYPE_DEFAULT = 'default';
	const VIEWTYPE_DETAIL  = 'detail';

	protected Products $productsService;

	/** @var int @persistent */
	public $productId;

	/** @var Product */
	protected $product;

	protected Carts $cartsService;

	protected Currencies $currencies;

	public string $viewType = self::VIEWTYPE_DEFAULT;

	public function __construct(Products $products, Carts $cartsService, Currencies $currencies)
	{
		$this->productsService = $products;
		$this->cartsService    = $cartsService;
		$this->currencies      = $currencies;
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);

		if ($this->productId && !$this->product) {
			$this->setProduct($this->productId);
		}
	}

	public function render(): void
	{
		if (Config::load('allowAddToCartOnlyForLoggedUsers') && !$this->getPresenter()->getUser()->isLoggedIn())
			return;

		$cart = $this->cartsService->getCurrentCart();

		$this->template->viewType = $this->viewType;
		$this->template->product  = $this->product;
		$this->template->cartItem = $cart->getCartItemByProductId($this->product->getId());

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form          = $this->createForm();
		$minimumAmount = $this->product->minimumAmount;

		$form->addHidden('productId');
		$form->addHidden('updateQuantity', false);
		$form->addHidden('otherData');
		$form->addText('quantity')->setDefaultValue($minimumAmount)
			->setHtmlAttribute('data-add-to-cart-quantity-input')
			->setHtmlAttribute('data-minimum-amount', $minimumAmount)
			->addRule(
				BaseForm::MIN,
				'eshopCatalogFront.cart.minimumQuantity',
				$minimumAmount > 1 ? $minimumAmount : 0
			);

		if (Config::load('pseudoWarehouse') && $this->product->unlimitedQuantity === 0 && $this->product->getQuantity() > 0) {
			$form['quantity']->addRule(BaseForm::MAX, 'eshopCatalogFront.cart.maximumQuantity', $this->product->getQuantity());
			$form['quantity']->setHtmlAttribute('data-max', $this->product->getQuantity());
		}

		if ((Config::load('product.allowNote') && $this->product->getExtraField('addNote'))
			|| $this->product->isAssort) {
			$label = $this->product->getExtraField('noteTitle') ?: $this->translator->translate('eshopCatalogFront.product.addNote');
			$form->addCheckbox('addNote', $label);
			$form->addTextArea('note', $label);
			$form->addHidden('noteTitle', $label);
		}

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->getPresenter();

		$canAddToCart = $this->productsService->getEr()->createQueryBuilder('p')
				->select('av.canAddToCart')
				->where('p.id = :id')
				->andWhere('p.isDeleted = 0')
				->setParameter('id', $values->productId)
				->innerJoin('p.availability', 'av')
				->getQuery()->setMaxResults(1)->getArrayResult()[0]['canAddToCart'] ?? null;

		if ($canAddToCart) {
			$canAddToCart = (int) $canAddToCart;
		}

		if (!$canAddToCart) {
			$presenter->flashMessageInfo('eshopOrderFront.cart.itemCannotBeAddedToCart');
			$presenter->redrawControl('flashes');

			return false;
		}

		$quantity = (int) $values->quantity;
		$cart     = $this->cartsService->getCurrentCart();
		$cartItem = $cart->getCartItemByProductId((int) $values->productId);
		if ($quantity >= 1) {
			if ($cartItem && $values->updateQuantity) {
				$this->eventDispatcher->dispatch(new UpdatedCartItemEvent(
					(int) $cartItem->getId(),
					(int) $values->quantity,
					(int) $cartItem->quantity,
					(array) $values,
					(int) $cartItem->productId,
				), 'eshopOrders.cartUpdateItem');
				$presenter->payload->cartEvent = 'cartUpdateItem';
			} else {
				$addedItem            = new AddedCartItem();
				$addedItem->quantity  = (int) $values->quantity ?: 1;
				$addedItem->productId = (int) $values->productId;
				$addedItem->moreData  = (array) $values;

				$this->eventDispatcher->dispatch(new AddedCartItemEvent($addedItem), 'eshopOrders.cartAddItem');
				$presenter->payload->cartEvent = 'cartAddItem';
			}
		} else if ($cartItem) {
			$item = new RemovedCartItemEvent($cartItem->getId());
			$this->eventDispatcher->dispatch($item, 'eshopOrders.cartRemoveItem');
			$presenter->payload->cartEvent = 'cartRemoveItem';
		}

		$cartItem = $this->cartsService->getCurrentCart()->getCartItemByProductId((int) $values->productId);
		if ($cartItem) {
			$response = [
				'productId'            => (int) $values->productId,
				'quantity'             => $cartItem->getQuantity(),
				'price'                => $cartItem->getPrice(),
				'priceWithoutVat'      => $cartItem->getPriceWithoutVat(),
				'totalPrice'           => $cartItem->getTotalPrice(),
				'totalPriceWithoutVat' => $cartItem->getTotalPriceWithoutVat(),
				'name'                 => $cartItem->title,
				'image'                => $cartItem->getImage()
					? $this->imagePipe->request($cartItem->getImage()->getFilePath(), Config::load('frontProductDetail.addedToCartPopupImageSize', '80x80'), 'exact')
					: '',
			];
		}
		$presenter->payload->cartItem = $response ?? [];

		$curr                         = $this->currencies->getCurrent();
		$presenter->payload->currency = [
			'symbol'  => $curr->symbol,
			'decimal' => $curr->decimals,
		];

		return true;
	}

	/**
	 * @param int|Product $product
	 */
	public function setProduct($product): self
	{
		if ($product instanceof Product) {
			$this->product   = $product;
			$this->productId = $product->getId();
		} else {
			$this->productId = $product;
		}

		if ($this->productId) {
			$this['form']->setDefaults([
				'productId' => $this->productId,
			]);
		}

		return $this;
	}
}
