<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Dao;

use Core\Model\Dao\Country;
use Core\Model\Entities\TSeo;
use Core\Model\Helpers\Strings;
use EshopCatalog\Model\Config;
use EshopCatalog\FrontModule\Model\Dao\Manufacturer;
use EshopCatalog\Model\Entities\Feature;
use EshopGifts\FrontModule\Model\Dao\Gift;
use Gallery\FrontModule\Model\Dao\Album;
use Gallery\FrontModule\Model\Dao\Image;
use Nette\Utils\DateTime;

class Product
{
	use TSeo;

	/** @var int */
	public $id;

	/** @var int */
	public $quantity;

	public int $quantityExternal = 0;

	public int $minimumAmount = 1;

	public ?string $currency = null;

	/**
	 * Základní cena bez slev
	 *
	 * @var float
	 */
	public $basePrice;

	/** @var float */
	public $price;

	/** @var float */
	public $retailPrice;

	/** @var float */
	public $basePriceInBaseCurrency;

	/** @var float */
	public $priceInBaseCurrency;

	/** @var float */
	public $retailPriceInBaseCurrency;

	/** @var float */
	public $recyclingFee;

	/** @var int */
	public $manufacturerId;

	/** @var Manufacturer */
	private $manufacturer;

	/** @var string */
	public $name;

	/** @var string */
	public $name2;

	/**
	 * Základní název pro skupinu variant
	 */
	public ?string $variantName = null;

	/**
	 * Základní obrázek pro skupinu variant
	 */
	public ?Image $variantImage = null;

	/** @var string */
	public $shortDescription;

	/** @var string */
	public $description;

	/** @var DateTime */
	public $created;

	/** @var DateTime */
	public $modified;

	/** @var int */
	public $galleryId;

	/** @var Album */
	public $gallery;

	/** @var FeatureProduct[] */
	public array $features = [];

	public array $featureValuesIds = [];

	/** @var RelatedGroup[] */
	protected array $related = [];

	/** @var self[] */
	public $alternative;

	/** @var int */
	public $defaultCategoryId;

	/** @var Category */
	public $defaultCategory;

	/** @var int[] */
	public $categories;

	/** @var Tag[] */
	public $tags;

	/** @var string */
	public $link = '#';

	/** @var int */
	public $vatRate;

	/** @var string */
	protected $ean;

	/** @var string */
	public $code1;

	/** @var string */
	public $code2;

	/** @var Availability */
	protected $availability;

	public ?Availability $externalAvailability = null;

	public ?string $preorderText = null;

	/** @var int */
	public $unlimitedQuantity = 0;

	/** @var array */
	protected $extraFields = [];

	/** @var bool */
	public $discountDisabled = false;

	/** @var bool */
	public $categoryGiftsAllowed = true;

	/** @var Gift[] */
	public $gifts = [];

	public array $giftsIds = [];

	public bool $isActive = true;

	public bool $canAddToCart = true;

	public bool $isAssort = false;

	public bool $isDiscount = false;

	/** @var array|self[] */
	public array $variants = [];

	public ?int $variantId = null;

	public ?int $variantOf = null;

	public array $variantProductIds = [];

	public array $helperData = [];

	/** @var FeatureProduct[] */
	protected array $cVariantDifferences = [];

	public string $condition = 'new';

	/** @var Document[] */
	protected array $documents = [];

	/** @var Video[] */
	protected array $videos = [];

	/** @var bool */
	public bool $disablePickUpSpedition = false;

	/** @var bool */
	public bool $isOversize = false;

	public array $inSites = [];

	public array $disabledSpeditions = [];

	/** @var int|null Package width */
	public ?int $width = null;

	/** @var int|null Package height */
	public ?int $height = null;

	/** @var int|null Package depth */
	public ?int $depth = null;

	/** @var int|null Package weight */
	public ?int $weight = null;

	public ?int     $hsCustomsCode   = null;
	public ?Country $countryOfOrigin = null;

	public function __construct($id)
	{
		$this->id = $id;
	}

	/**
	 * @return int
	 */
	public function getId()
	{
		return $this->id;
	}

	public function isInSite(string $siteIdent): bool
	{
		return in_array($siteIdent, $this->inSites);
	}

	/*******
	 * == Quantity
	 */

	public function getQuantityExternal(): int { return $this->quantityExternal; }

	public function getQuantity(): int
	{
		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1)
			return 500;

		return (int) $this->quantity;
	}

	public function setQuantityExternal(int $quantity): self
	{
		$this->quantityExternal = $quantity;

		return $this;
	}

	public function setQuantity(int $quantity): self
	{
		$this->quantity = $quantity;

		return $this;
	}

	/**
	 * Vrátí text pro překlad s počtem ks skladem
	 *
	 * @param bool $useExternal
	 *
	 * @return string
	 */
	public function getQuantityText(bool $useExternal = false): string
	{
		$q     = $useExternal ? $this->quantityExternal : $this->getQuantity();
		$limit = Config::load('quantityTextLimit', 15);

		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1)
			$text = 'inStock';
		else if ($q <= 0)
			$text = 'soldOut';
		else if ($q <= 4 && $q <= $limit)
			$text = 'lastInStock1';
		else if ($q <= 15 && $q <= $limit)
			$text = 'lastInStock2';
		else
			$text = 'inStock';

		return 'eshopCatalogFront.product.' . $text;
	}

	/**
	 * Vrátí stav poštu ks skladem
	 *
	 * @param bool $useExternal
	 *
	 * @return string
	 */
	public function getQuantityColor(bool $useExternal = false): string
	{
		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1)
			return 'green';

		$q     = $useExternal ? $this->quantityExternal : $this->getQuantity();
		$limit = Config::load('quantityTextLimit', 15);
		if ($q <= 0)
			return 'red';

		if ($q <= $limit)
			return 'yellow';

		return 'green';
	}

	/*******
	 * == Price
	 */

	public function getPrice(): float { return (float) $this->price; }

	public function setPrice(float $price): self
	{
		$this->price = $price;

		return $this;
	}

	public function getBasePrice(): float { return (float) $this->basePrice; }

	public function setBasePrice(float $price): self
	{
		$this->basePrice = $price;

		return $this;
	}

	public function getPriceWithoutVat(): float { return ($this->getPrice() / (1 + ($this->vatRate / 100))); }

	public function getRetailPrice(): float
	{
		return (float) $this->retailPrice;
	}

	/**
	 * @param float $retailPrice
	 */
	public function setRetailPrice(float $retailPrice)
	{
		$this->retailPrice = $retailPrice;
	}

	public function getRetailPriceWithoutVat(): float { return ($this->getRetailPrice() / (1 + ($this->vatRate / 100))); }

	public function getRecyclingFee(): float
	{
		return (float) $this->recyclingFee;
	}

	/**
	 * @param float $retailPrice
	 */
	public function setRecyclingFee(float $recyclingFee)
	{
		$this->recyclingFee = $recyclingFee;
	}

	/**
	 * @return Manufacturer
	 */
	public function getManufacturer()
	{
		return $this->manufacturer;
	}

	/**
	 * @param mixed $manufacturer
	 */
	public function setManufacturer($manufacturer)
	{
		$this->manufacturer = $manufacturer;
	}

	public function getName(): string
	{
		$str = $this->name;

		if (Config::load('addManufacturerBeforeProductName', false) && $this->getManufacturer())
			$str = trim($this->getManufacturer()->name . ' ' . $str);

		$variantDiff = $this->getVariantDifferences();
		if (Config::load('addOneVariantDiffToProductName', false) && count($variantDiff) === 1) {
			$value = array_values($variantDiff)[0]->value;

			if (!Strings::endsWith($str, $value))
				$str .= ' - ' . $value;
		}

		return $str;
	}

	public function getVariantImage(): ?Image
	{
		return $this->variantImage;
	}

	public function getVariantName(): ?string
	{
		$str = $this->variantName;

		if (!$str) {
			foreach ($this->variants as $var) {
				if ($var->variantName) {
					$str = $var->variantName;
					break;
				}
			}

			if (!$str)
				return null;
		}

		if (Config::load('addManufacturerBeforeProductName') && $this->getManufacturer())
			$str = trim($this->getManufacturer()->name . ' ' . $str);

		return $str;
	}

	/**
	 * @param string $name
	 */
	public function setName($name)
	{
		$this->name = $name;
	}

	public function setName2($name)
	{
		$this->name2 = $name;
	}

	public function getName2(): ?string { return Config::load('enableProductName2') ? $this->name2 : $this->name; }

	/**
	 * @return string
	 */
	public function getDescription()
	{
		return $this->description;
	}

	/**
	 * @param string $description
	 */
	public function setDescription($description)
	{
		$this->description = $description;
	}

	/**
	 * @return Album|null
	 */
	public function getGallery()
	{
		if (!$this->gallery && $this->variantOf && $this->variants)
			$this->gallery = isset($this->variants[$this->variantOf]) ? $this->variants[$this->variantOf]->gallery : null;

		return $this->gallery;
	}

	/**
	 * @param mixed $gallery
	 */
	public function setGallery($gallery)
	{
		$this->gallery = $gallery;
	}

	public function addFeature(FeatureProduct $feature)
	{
		$this->features[$feature->idFeature] = $feature;
	}

	/**
	 * @param FeatureProduct[] $features
	 */
	public function setFeatures($features)
	{
		$this->features += $features;
	}

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeatures() { return $this->features; }

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeaturesForTags(): array
	{
		$arr = [];
		foreach ($this->getFeatures() as $feature) {
			if ($feature->showAsTag)
				$arr[$feature->idFeatureValue] = $feature;
		}

		return $arr;
	}

	protected ?array $cFeaturesForTab = null;

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeaturesForTab(): array
	{
		if ($this->cFeaturesForTab === null) {
			$this->cFeaturesForTab = [];

			foreach ($this->getFeatures() as $feature) {
				if ($feature->productTabTitle && ($feature->longDescription || $feature->moreLink))
					$this->cFeaturesForTab[$feature->idFeatureValue] = $feature;
			}
		}

		return $this->cFeaturesForTab;
	}

	/**
	 * @param int $id
	 *
	 * @return FeatureProduct[]
	 */
	public function getFeaturesById(int $id)
	{
		$list = [];

		foreach ($this->getFeatures() as $feature) {
			if ($feature->idFeature === $id)
				$list[$feature->idFeatureValue] = $feature;
		}

		return $list;
	}

	/**
	 * @param int $id
	 *
	 * @return FeatureProduct|null
	 */
	public function getFeatureById($id)
	{
		foreach ($this->getFeatures() as $feature) {
			if ($feature->idFeature === $id)
				return $feature;
		}

		return null;
	}

	public function getGroupedFeatures(): array
	{
		/** @var FeatureProduct[] $result */
		$result = [];

		foreach ($this->getFeatures() as $feature) {
			if ($feature->type === Feature::TYPE_RANGE) {
				$feature->value = str_replace(',', '.', (string) $feature->rawValue);
			}

			if (!isset($result[$feature->idFeature])) {
				if ($feature->type === Feature::TYPE_RANGE && $feature->unit) {
					$feature->name .= ' (' . $feature->unit . ')';
				}

				$result[$feature->idFeature] = $feature;
			} else {
				$result[$feature->idFeature]->value = trim($result[$feature->idFeature]->value) . ', ' . trim($feature->value);
			}
		}

		return $result;
	}

	/**
	 * @param string $type
	 *
	 * @return Tag|null
	 */
	public function getTag($type) { return $this->tags[$type] ?? null; }

	/**
	 * @param self[] $array
	 *
	 * @return $this
	 */
	public function setAlternatives($array)
	{
		$this->alternative = $array;

		return $this;
	}

	/** @return int */
	public function getVatRate()
	{
		return $this->vatRate;
	}

	/**
	 * @param int $vatRate
	 *
	 * @return Product
	 */
	public function setVatRate($vatRate): Product
	{
		$this->vatRate = $vatRate;

		return $this;
	}

	/** @return string */
	public function getEan()
	{
		return $this->ean;
	}

	/**
	 * @param string $ean
	 *
	 * @return Product
	 */
	public function setEan($ean): Product
	{
		$this->ean = $ean;

		return $this;
	}

	/**
	 * @return string
	 */
	public function getCode1()
	{
		return $this->code1;
	}

	/**
	 * @param $code1
	 *
	 * @return Product
	 */
	public function setCode1($code1): Product
	{
		$this->code1 = $code1;

		return $this;
	}

	/**
	 * @return string
	 */
	public function getCode2()
	{
		return $this->code2;
	}

	/**
	 * @param $code2
	 *
	 * @return Product
	 */
	public function setCode2($code2): Product
	{
		$this->code2 = $code2;

		return $this;
	}

	public function addExtraField($key, $value): self
	{
		$this->extraFields[$key] = $value;

		return $this;
	}

	public function setExtraFields(array $array): self
	{
		$this->extraFields = $array;

		return $this;
	}

	public function getExtraFields(): array { return $this->extraFields; }

	/**
	 * @param string $key
	 * @param mixed  $default
	 *
	 * @return mixed|null
	 */
	public function getExtraField(string $key, $default = null) { return $this->extraFields[$key] ?? $default; }

	/** @return Gift[] */
	public function getGifts(): array { return $this->gifts; }

	public function getAvailability(): ?Availability { return $this->availability; }

	public function setAvailability(?Availability $availability = null): self
	{
		$this->availability = $availability;

		return $this;
	}

	/**
	 * ====== Variants
	 */

	public function getSortedVariantsByFeature(?int $variantId = null, bool $addSelf = false)
	{
		if (!$this->features)
			return $this->variants;

		if ($variantId === null)
			$variantId = array_values($this->features)[0]->idFeature;

		$tmp = [];
		$not = [];

		$variants = $this->variants;
		if ($addSelf)
			$variants[$this->getId()] = $this;

		foreach ($variants as $variant) {
			$feature = $variant->getFeatureById($variantId);

			if ($feature) {
				$tmp[$feature->valuePosition][] = $variant;
			} else {
				$not[] = $variant;
			}
		}
		ksort($tmp);

		$tmp2 = [];
		foreach ($tmp as $k => $v) {
			foreach ($v as $k2 => $v2) {
				$tmp2[$v2->getId()] = $v2;
			}
		}

		return array_merge($tmp2, $not);
	}

	public function hasVariants(): bool { return $this->variants ? true : false; }

	public function getVariantDifferences(): array
	{
		if (empty($this->variants))
			return [];

		if (empty($this->cVariantDifferences)) {
			$used     = [];
			$features = [];
			foreach ($this->features as $v) {
				if (!$v->useForVariantDiff)
					continue;

				$features[$v->getKey()] = $v;
			}

			foreach ($this->variants as $v) {
				foreach ($v->features as $f) {
					if (!$f->useForVariantDiff)
						continue;

					$used[$f->getKey()]++;
				}
			}

			foreach ($used as $k => $v)
				if ($v === count($this->variants))
					unset($features[$k]);

			if (count($features) === 1) {
				$featureId = array_values($features)[0]->idFeature;
				uksort($this->variants, function($a, $b) use ($featureId) {
					$a = isset($this->variants[$a]) ? $this->variants[$a]->getFeatureById($featureId) : null;
					$b = isset($this->variants[$b]) ? $this->variants[$b]->getFeatureById($featureId) : null;

					if (!$a) {
						$a                = new FeatureProduct();
						$a->valuePosition = 999;
					}
					if (!$b) {
						$b                = new FeatureProduct();
						$b->valuePosition = 999;
					}

					return $a->valuePosition <=> $b->valuePosition;
				});
			}

			$this->cVariantDifferences = $features;
		}

		return $this->cVariantDifferences;
	}

	public function getVariantsDiffSteps(): array
	{
		$arr              = [
			'diffs' => [],
			'data'  => [],
		];
		$requiredFeatures = [];

		foreach ($this->getVariantDifferences() as $feature) {
			$row = [
				'feature'  => $feature,
				'variants' => [],
			];

			foreach ($this->getSortedVariantsByFeature($feature->idFeature, true) as $variant) {
				$hasCount = 0;

				foreach ($requiredFeatures as $requiredFeatureValue => $requiredFeature) {
					foreach ($variant->getFeatures() as $variantFeature) {
						if ($variantFeature->idFeatureValue === $requiredFeatureValue) {
							$hasCount++;
							break;
						}
					}
				}

				if ($hasCount === count($requiredFeatures)) {
					/** @var FeatureProduct $variantFeature */
					$variantFeature = array_values($variant->getFeaturesById($feature->idFeature))[0];
					$key            = $variantFeature->idFeatureValue;

					if (!isset($row['variants'][$key])) {
						$row['variants'][$key]['variant']  = $variant;
						$row['variants'][$key]['variants'] = [];
						$row['variants'][$key]['feature']  = $variantFeature;
					}

					$row['variants'][$key]['variants'][] = $variant;
				}
			}

			$requiredFeatures[$feature->idFeatureValue] = $feature;
			$arr['data'][$feature->idFeatureValue]      = $row;
			$arr['diffs'][$feature->idFeatureValue]     = $feature;
		}

		if (count($requiredFeatures) > 1) {
			foreach ($arr['data'] as &$row) {
				$checkRequireFeatures = $requiredFeatures;
				unset($checkRequireFeatures[$row['feature']->idFeatureValue]);

				foreach ($row['variants'] as &$vals) {
					foreach ($vals['variants'] as $variant) {
						if (count(array_intersect_key($checkRequireFeatures, array_flip($variant->featureValuesIds))) === count($checkRequireFeatures)) {
							$vals['variant'] = $variant;
							break;
						}
					}
				}
			}
		}

		return $arr;
	}

	public function getFromPrice(): float
	{
		$min = $this->getPrice();

		foreach ($this->variants as $v)
			if ($v->getPrice() < $min)
				$min = $v->getPrice();

		return $min;
	}

	public function getFromPriceWithoutVat(): float
	{
		$min = $this->getPriceWithoutVat();

		foreach ($this->variants as $v)
			if ($v->getPriceWithoutVat() < $min)
				$min = $v->getPriceWithoutVat();

		return $min;
	}

	/**
	 * @return self[]
	 */
	public function getRelated(): array
	{
		return is_array($this->related) ? $this->related : [];
	}

	/**
	 * @param Product[] $products
	 */
	public function setRelated(array $products): void
	{
		$this->related = $products;
	}

	/**
	 * @param Document $document
	 */
	public function addDocument(Document $document): void { $this->documents[] = $document; }

	/**
	 * @return array|Document[]
	 */
	public function getDocuments(): array { return $this->documents; }

	/**
	 * @param Video $video
	 */
	public function addVideo(Video $video): void { $this->videos[] = $video; }

	/**
	 * @return array|Video[]
	 */
	public function getVideos(): array { return $this->videos; }
}
