<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Categories;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Model\Products;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Config;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\Model\Event\RouteInEvent;
use EshopCatalog\Model\Event\RouteInFindAliasEvent;
use Navigations\Model\Event\RouteOutEvent;
use Navigations\Model\Event\RouteUpdateCacheDep;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Http\Request;
use Nette\Http\Url;
use Nette\Utils\ArrayHash;
use Pages\AdminModule\Model\TemplatePages;

class Category extends BaseNavigation implements INavigationItem
{
	protected $title = 'category';

	protected $presenter = 'EshopCatalog:Front:Default';
	protected $action    = 'category';

	const CACHE_CATEGORY = 'eshopCatalogCategory';

	protected Categories    $categoriesService;
	protected Products      $productsService;
	protected TemplatePages $templatePagesService;
	protected Request       $request;
	protected Home          $home;
	protected Container     $container;

	/** @var null|int */
	protected $formRootId = null;

	public function __construct(Categories $categories, Products $products, TemplatePages $templatePages, Request $request,
	                            Home       $home, Container $container)
	{
		$this->categoriesService    = $categories;
		$this->productsService      = $products;
		$this->templatePagesService = $templatePages;
		$this->request              = $request;
		$this->home                 = $home;
		$this->container            = $container;
	}

	public function navigationFormCustomHandle(NavigationForm $component, array $data = []): void
	{
		$this->formRootId = $this->getRootCategory($data['site']);
		$component->loadComponent($data['componentId']);
		$component->redrawControl('component');
	}

	protected function getRootCategory(string $site): ?int
	{
		$result = $this->categoriesService->getEr()->createQueryBuilder('c')->select('c.id')
			->innerJoin('c.categoryTexts', 'ct', Join::WITH, 'ct.alias = :alias')
			->setParameter('alias', $site)
			->getQuery()->setMaxResults(1)->getScalarResult();

		return isset($result[0]['id']) ? (int) $result[0]['id'] : null;
	}

	/**
	 * @param BaseForm $baseForm
	 *
	 * @return BaseContainer
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer();
		$rootId    = $this->formRootId;

		if (!$rootId)
			$rootId = $this->getRootCategory($baseForm->getCustomData('loadComponent')['site'] ?: $baseForm->getValues()['site']);

		$categories = [];
		if ($rootId) {
			foreach ($this->categoriesService->getEr()->createQueryBuilder('c')->select('c.id, c.lvl, ct.name')
				         ->where('c.isPublished = 1')
				         ->andWhere('c.root = :root')
				         ->andWhere('c.lvl > 0')
				         ->join('c.categoryTexts', 'ct')
				         ->setParameters([
					         'root' => $rootId,
				         ])
				         ->groupBy('c.id')
				         ->addOrderBy('c.lft')
				         ->getQuery()->getArrayResult() as $category) {
				$title = ' ' . $category['name'];
				for ($i = 1; $i < $category['lvl']; $i++)
					$title = '---' . $title;
				$categories[$category['id']] = $title;
			}
		}

		$site  = $this->navFormSite ?: $baseForm->getCustomData('loadComponent')['site'];
		$pages = $this->templatePagesService->getOptionsForSelectGrouped();
		$pages = $pages[$site ?? $baseForm['site']->getValue()] ?? $pages;


		$container->addSelect('category', $this->t('eshopCatalog.navigationBuilder.category'), $categories)
			->setTranslator(null);
		$container->addSelect('templatePage', $this->t('eshopCatalog.navigationBuilder.template'), $pages)
			->setTranslator(null);

		if ($baseForm->getParent()->lookup(null, false)) {
			$componentId = $baseForm->getParent()->getParameter('componentId');
			if ($componentId)
				$container['category']->setHtmlAttribute('data-custom-link', $baseForm->getParent()->link('componentCustom!', [$componentId]));
		} else {
			$baseForm->onAnchor[] = function() use ($container, $baseForm) {
				$container['category']->setHtmlAttribute('data-custom-link', $baseForm->getParent()->link('componentCustom!', [$baseForm->getValues()->componentType]));
			};
		}

		return $container;
	}

	public function getTemplateFile(): ?string
	{
		return __DIR__ . '/Category.latte';
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{

	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		// Přidat klíče pouze pokud se nejedná o filtr, jinak se to zasekne a nepůjdou upravit kategorie
		if (!isset($urlParams['productsFilter-filter']) && !isset($urlParams['productsFilter-sort'])) {
			$cache[Cache::TAGS][] = 'category/' . $urlParams['id'];
			$cache[Cache::TAGS][] = 'categories';
		}

		$this->eventDispatcher->dispatch(new RouteUpdateCacheDep($cache, $urlParams), BaseNavigation::class . '::routeUpdateCacheDep');

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		return $this->presenter == $presenter && $this->action == $action;
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 *
	 * @return array|mixed
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$parts    = explode('/', $urlParams['path']);
		$category = null;

		$return = [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => null,
		];

		if (count($parts) > 1) {
			$useBaseCategoryInUrl = (bool) Config::load('product.useBaseCategoryInUrl');
			$cfUrls               = Config::load('allowCategoryFullUrlField', false)
				? $this->categoriesService->getCategoryFullUrl(null, $urlParams['locale'] ?? $this->translator->getLocale())
				: [];
			$urlPath              = trim($urlParams['path'], '/');

			foreach (array_reverse($parts) as $alias) {
				foreach ($this->categoriesService->findByAlias($alias) as $c) {
					if (!$c->link) {
						continue;
					}

					$url = new Url($c->link);

					$catUrlPath = trim(urldecode($url->getPath()), '/');

					if ($catUrlPath === $urlPath || $catUrlPath === $urlParams['locale'] . '/' . $urlPath) {
						$category = $c->getId();
						break 2;
					} else if ($useBaseCategoryInUrl) {
						$tmp = $parts;
						array_pop($tmp);

						if ($catUrlPath === implode('/', $tmp)) {
							$productNavs = $this->container->getService('navigations.navigations')->getPublishedByComponent('eshopCatalog.navigation.home');

							if ($productNavs) {
								$productNav = array_values($productNavs)[0];
								$isProduct  = $this->home->routerIn($productNav, ['path' => $urlParams['path']]);

								if ($isProduct) {
									$isProduct['activeNavigation'] = $productNav;

									return $isProduct;
								}
							}
						}
					}
				}

				$this->eventDispatcher->dispatch(new RouteInFindAliasEvent($alias, $urlParams, $return), BaseNavigation::class . '::routeInFindAlias');
				if (isset($return['id'])) {
					if (!isset($return['activeNavigation']))
						$return['activeNavigation'] = $navigation;

					$category = $return['id'];
					break;
				}
			}
		}

		if ($category)
			$return['id'] = $category;
		else
			$return['id'] = $navigation->componentParams['category'];

		$this->eventDispatcher->dispatch(new RouteInEvent($navigation, $urlParams, $return), BaseNavigation::class . '::routeIn');

		return $return;
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		$cfUrl       = '';
		$cfUrls      = [];
		$cfUrlParams = [];

		$categoryId = $urlParams['id'] ?? $navigation->componentParams['category'];
		$path       = $this->categoriesService->getQueryPath((int) $categoryId);
		$tmp        = [];

		if (Config::load('allowCategoryFullUrlField', false)) {
			$cfUrls = $this->categoriesService->getCategoryFullUrl(null, $navigation->getLang());
			$cfUrl  = $cfUrls[$categoryId] ?? '';
		}

		if (!$cfUrl) {
			if ($path && !$path[0])
				array_shift($path);

			if (Strings::contains((string) $urlParams['id'], '|')) {
				$tmp[] = ltrim((string) $this->request->getUrl()->getPath(), '/');
			} else if (count($path) > 1) {
				$parentFound = false;
				foreach (array_reverse($path) as $k => $c) {
					if ($c['id'] == $navigation->componentParams['category']) {
						$fullUrl      = $cfUrls[$c['id']] ?? null;
						$navLinkParts = explode('/', (string) $navigation->link);
						$tmp[]        = $fullUrl ?: ($navigation->alias ?: end($navLinkParts));
						$parentFound  = true;
						break;
					} else {
						$tmp[] = $c['alias'];
					}
				}

				if (!$parentFound)
					return null;

				$tmp = array_reverse($tmp);
			} else {
				if (isset($urlParams['id']) && $navigation->componentParams['category'] != $urlParams['id'])
					return null;

				$tmp[] = $navigation->alias;
			}

			$cfUrl = implode('/', $tmp);

			if ($cfUrl) {
				$cfUrl = '/' . ltrim($cfUrl, '-');
			}
		}

		$this->eventDispatcher->dispatch(new RouteOutEvent($navigation, $urlParams, $cfUrl), BaseNavigation::class . '::routeOut');

		return $cfUrl ? ('/' . ltrim($cfUrl, '/')) : null;
	}

	/**
	 * @param string $lang
	 * @param null   $navigation
	 * @param bool   $siteMapData
	 *
	 * @return array
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		$category = $this->categoriesService->getEr()->createQueryBuilder('c')->select('c.lft, c.gt')
			->where('c.id = :id')
			->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			->setParameters([
				'lang' => $lang,
				'id'   => $navigation->componentParams['category'],
			])
			->getQuery()->getOneOrNullResult(Query::HYDRATE_SCALAR);

		foreach ($this->categoriesService->getEr()->createQueryBuilder('c')->select('c.id, ct.name, ct.seo, c.modified')
			         ->andWhere('c.isPublished = 1')->andWhere('c.lft >= :lft')->andWhere('c.lft <= :gt')
			         ->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			         ->setParameters([
				         'lang' => $lang,
				         'lft'  => $category['lft'],
				         'gt'   => $category['gt'],
			         ])
			         ->getQuery()->getScalarResult() as $row) {
			$id = $row['id'];

			$data = [
				'action'    => $this->action,
				'id'        => $id,
				'presenter' => $this->presenter,
				'locale'    => $lang,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($row['modified'])
					$siteMapUrl->setModified($row['modified']);

				$seo = $row['seo'] ? unserialize($row['seo']) : null;
				if ($seo) {
					if (!isset($seo['addToSiteMap']))
						$seo['addToSiteMap'] = '1';
					if ($seo['addToSiteMap'] === '0' || is_string($seo['robots']) && strpos('noindex', $seo['robots']) !== false)
						continue;

					$siteMapUrl->setSeo($seo, null, true);
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}
			$arr[] = $data;
		}

		return $arr;
	}
}
