<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Helpers\Traits\TPosition;
use Core\Model\Helpers\Traits\TPublish;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Caching\Cache;

/**
 * Class Manufacturers
 * @package EshopCatalog\AdminModule\Model
 *
 * @method Manufacturer|null|object = getReference($id)
 * @method Manufacturer[]|null getAll()
 * @method Manufacturer|null get($id)
 */
class Manufacturers extends BaseEntityService
{
	use TPublish;
	use TPosition;

	protected $entityClass = Manufacturer::class;

	/** @var CacheService */
	protected $cacheService;

	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	/**
	 * @throws \Doctrine\DBAL\DBALException
	 */
	public function mergeAll()
	{
		$fks = [];
		foreach ($this->em->getConnection()->fetchAllAssociative("
				SELECT *
					FROM information_schema.KEY_COLUMN_USAGE
					WHERE
  						REFERENCED_TABLE_NAME = 'eshop_catalog__manufacturer'
  						AND REFERENCED_COLUMN_NAME = 'id'
  						AND CONSTRAINT_SCHEMA = '{$this->em->getConnection()->getDatabase()}'") as $tmp) {
			if ($tmp['TABLE_NAME'] == 'eshop_catalog__manufacturer_texts')
				continue;

			$fks[] = $tmp;
		}

		foreach ($this->getEr()->createQueryBuilder('m')->select('m.name, count(m.id)')->groupBy('m.name')->having('count(m.id) > 1')
			         ->getQuery()->getArrayResult() as $duplicate) {
			$ids = [];
			foreach ($this->getEr()->createQueryBuilder('m')->select('m.id')->where('m.name = :name')->setParameter('name', $duplicate['name'])
				         ->getQuery()->getArrayResult() as $tmp)
				$ids[] = $tmp['id'];

			$first = array_shift($ids);

			foreach ($fks as $fk) {
				$this->em->getConnection()
					->executeStatement("UPDATE {$fk['TABLE_NAME']} SET {$fk['COLUMN_NAME']} = {$first} WHERE {$fk['COLUMN_NAME']} IN (" . implode(',', $ids) . ")");
			}

			$this->em->createQuery('DELETE FROM ' . $this->entityClass . ' m WHERE m.id IN (:id)')->setParameter('id', $ids)->execute();
		}

		$this->cacheService->defaultCache->clean([Cache::TAGS => ['manufacturers']]);
	}

	public function getOptionsForSelect(): array
	{
		$result = [];
		foreach ($this->getEr()->createQueryBuilder('m')->select('m.id, m.name')
			         ->orderBy('m.position')
			         ->getQuery()->getScalarResult() as $row)
			$result[$row['id']] = $row['name'];

		return $result;
	}

	public function sortAlphabetically(): bool
	{
		$data = [];
		foreach ($this->getEr()->createQueryBuilder('m')->select('m.id, m.name')
			         ->getQuery()->getArrayResult() as $k => $row) {
			$data[] = [
				'id'   => $row['id'],
				'name' => $row['name'],
			];
		}
		usort($data, fn($a, $b) => Strings::sortCzech($a['name'], $b['name']));

		$conn      = $this->em->getConnection();
		$tableName = $this->em->getClassMetadata($this->entityClass)->getTableName();
		foreach ($data as $k => $v) {
			$conn->update($tableName, [
				'position' => $k,
			], [
				'id' => $v['id'],
			]);
		}

		return true;
	}

	public function getOrCreate(string $name): Manufacturer
	{
		$lowerName = Strings::lower(trim($name));
		foreach ($this->getOptionsForSelect() as $k => $v) {
			if (Strings::lower($v) === $lowerName) {
				return $this->getReference($k);
			}
		}

		$manufacturer       = new Manufacturer();
		$manufacturer->name = trim($name);
		$this->em->persist($manufacturer);

		return $manufacturer;
	}
}
