<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Helpers\FeaturesHelper;
use Navigations\Model\Navigations;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;

class FeatureProducts extends BaseFrontEntityService
{
	protected $entityClass = FeatureProduct::class;

	protected Features $features;

	public ?LinkGenerator $linkGenerator = null;

	public ?Navigations $navigations = null;

	protected array $cProductFeatures = [];

	protected array $cFeatureValuesIds = [];

	protected array $cacheDep = [
		Cache::TAGS   => ['featureProduct'],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(Features $features)
	{
		$this->features = $features;
	}

	public function getFeatureValuesIdsForProducts(array $ids): array
	{
		$whereIds = [];
		$result   = [];

		foreach ($ids as $id) {
			if (isset($this->cFeatureValuesIds[$id])) {
				$result[$id] = $this->cFeatureValuesIds[$id];
			} else {
				$whereIds[]                   = $id;
				$this->cFeatureValuesIds[$id] = [];
				$result[$id]                  = [];
			}
		}

		if (!empty($whereIds)) {
			foreach (array_chunk($whereIds, 900) as $chunk) {
				foreach ($this->getEr()->createQueryBuilder('fp')
					         ->select('IDENTITY(fp.product) as product, IDENTITY(fp.featureValue) as value')
					         ->andWhere('fp.product IN (' . implode(',', $chunk) . ')')
					         ->getQuery()->getScalarResult() as $row) {
					$this->cFeatureValuesIds[$row['product']][] = $row['value'];
					$result[$row['product']][]                  = $row['value'];
				}
			}
		}

		return $result;
	}

	/**
	 * @param Dao\Product[] $products
	 *
	 * @return Dao\FeatureProduct[][]
	 */
	public function getFeaturesForProduct(array $products): array
	{
		/** @var Dao\Product[] $whereIds */
		$whereIds = [];
		$result   = [];

		foreach ($products as $product) {
			$id = $product->getId();

			if (isset($this->cProductFeatures[$id])) {
				$result[$id] = $this->cProductFeatures[$id];
			} else {
				$whereIds[] = $product;
			}
		}

		if (!empty($whereIds)) {
			$data = [];
			foreach ($whereIds as $product) {
				foreach ($product->featureValuesIds as $valueId) {
					$value = $this->features->getFeatureValueById($valueId);
					if (!$value)
						continue;
					$feature = $this->features->getFeatureById($value['featureId']);
					if (!$feature)
						continue;

					$data[$product->getId()][$valueId] = $this->fillDao($product->getId(), $feature, $value);
				}
			}

			foreach ($data as $productId => $features) {
				uasort($features, fn($a, $b) => $a->position <=> $b->position);
				$result[$productId]                 = $features;
				$this->cProductFeatures[$productId] = $features;
			}
		}

		return $result;
	}

	protected function fillDao(int $productId, array $feature, array $value): Dao\FeatureProduct
	{
		$fp                    = new Dao\FeatureProduct();
		$fp->idProduct         = $productId;
		$fp->idFeature         = $feature['id'];
		$fp->position          = $feature['position'];
		$fp->name              = $feature['name'];
		$fp->idFeatureValue    = $value['id'];
		$fp->valuePosition     = $value['position'];
		$fp->value             = $value['name'] . ($feature['unit'] ? ' ' . $feature['unit'] : '');
		$fp->rawValue          = (string) $value['name'];
		$fp->unit              = $feature['unit'];
		$fp->useForVariantDiff = (int) $feature['useForVariantDiff'];
		$fp->valueType         = $feature['valueType'] ?: Feature::VALUE_TYPE_TEXT;

		if (isset($value['extraFields'])) {
			$fp->setExtraFields($value['extraFields']);
		}

		if (Config::load('features.allowImages')) {
			$fp->image = $value['image'];
		}
		if (Config::load('features.allowDescription')) {
			$fp->productTabTitle  = $feature['productTabTitle'];
			$fp->showAsTag        = (bool) $value['showAsTag'];
			$fp->tagText          = $value['tagText'];
			$fp->shortDescription = $value['shortDescription'];
			$fp->longDescription  = $value['longDescription'];
			$fp->tagTextColor     = $value['tagTextColor'];
			$fp->tagBgColor       = $value['tagBgColor'];

			if ($value['moreLink']) {
				$v = explode('|', $value['moreLink']);
				if ($v[0] === 'text')
					$fp->moreLink = $v[1];
				else if ($v[0] === 'navigation') {
					if ($nav = $this->navigations->getNavigation((int) $v[1]))
						$fp->moreLink = $nav->link;
				} else if ($v[0] === 'article')
					$fp->moreLink = $this->linkGenerator->link('Blog:Front:Articles:detail', ['id' => $v[1]]);
			}
		}

		if ($fp->valueType === Feature::VALUE_TYPE_COLOR && !empty($value['value1'])) {
			$fp->extraFields += $this->features->featuresHelper->parseValueColors((string) $value['value1']);
		}

		return $fp;
	}
}
