<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\Criteria;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Tag;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use Nette\Utils\DateTime;

class ProductQuery
{
	/** @var array|\Closure[] */
	private $filter = [];

	/** @var array|\Closure[] */
	private $select = [];

	/** @var string */
	protected $lang;

	public ?string $siteIdent = null;

	public static bool $ignoreProductPublish = false;

	public function __construct($lang)
	{
		$this->lang = $lang;
	}

	public function skipId($id)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($id) {
			$qb->andWhere('p.id != :notId')->setParameter('notId', $id);
		};

		return $this;
	}

	public function withTexts($addSelect = true)
	{
		$this->filter[] = function(QueryBuilder $qb) {
			$qb->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
				->setParameter('lang', $this->lang);
		};
		if ($addSelect)
			$this->select[] = function(QueryBuilder $qb) {
				$qb->addSelect('pt');
			};

		return $this;
	}

	public function withCategories()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->addSelect('cp')
				->leftJoin('p.categoryProducts', 'cp');
		};

		return $this;
	}

	public function withVatRate()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->addSelect('vr.rate as vatRate')
				->leftJoin('p.vatRate', 'vr');
		};

		return $this;
	}

	public function withVariantParent(): self
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->leftJoin('p.isVariant', 'variants')
				->addSelect('MAX(variants.variantId) as variantId')
				->addSelect('variants.defaultImage as variantImage')
				->addSelect('variants.variantName as variantName');
		};

		return $this;
	}

	public function withTags()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->leftJoin('p.productTags', 'ptags');
		};

		return $this;
	}

	public function withTag($tag)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($tag) {
			$now = new DateTime();
			$now->setTime((int) $now->format("H"), (int) $now->format("i"), 0);
			$tagKey = (string) $tag;
			$qb->innerJoin('p.productTags', 'pTags', 'WITH', 'pTags.tag = :pTagId' . $tagKey . ' AND (pTags.validFrom <= :pTagFrom' . $tagKey . ' OR pTags.validFrom IS NULL) AND (pTags.validTo >= :pTagTo' . $tagKey . ' OR pTags.validTo IS NULL)')
				->setParameter('pTagFrom' . $tagKey, $now)
				->setParameter('pTagTo' . $tagKey, $now)
				->setParameter('pTagId' . $tagKey, $tag)
				->groupBy('p.id');
		};

		return $this;
	}

	public function withoutTag($tag)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($tag) {
			$qb->leftJoin('p.productTags', 'pTags')
				->leftJoin('pTags.tag', 'tags')
				->andWhere('pTags IS NULL OR tags.type != :withoutTag')->setParameter('withoutTag', $tag)
				->groupBy('p.id');
		};

		return $this;
	}

	public function loadSites(): self
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->addSelect('GROUP_CONCAT(IDENTITY(sites.site)) as inSites')
				->leftJoin('p.sites', 'sites', Join::WITH, 'sites.isActive = 1');
		};

		return $this;
	}

	public function inSite(string $siteIdent): self
	{
		$this->filter[] = function(QueryBuilder $qb) use ($siteIdent) {
			if (in_array('sites', $qb->getAllAliases())) {
				$qb->andWhere('sites.isActive = 1')
					->andWhere('sites.site = :siteIdent')
					->setParameter('siteIdent', $siteIdent);
			} else {
				$qb->innerJoin('p.sites', 'sites', 'WITH', 'sites.site = :site AND sites.isActive = 1')
					->setParameter('site', $siteIdent);
			}
		};

		return $this;
	}

	public function disableListing(?int $disableListing = 0): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($disableListing) {
			$qb->andWhere('p.disableListing = :disableListing')
				->setParameter('disableListing', $disableListing);
		};

		return $this;
	}

	public function inCategory($id)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($id) {
			// Pokud chceme hledat pro eshop, je potreba nejdriv zavolat inSite()
			if (!in_array('sites', $qb->getAllAliases()))
				$qb->leftJoin('p.sites', 'sites', Join::WITH, 'sites.isActive = 1');

			if (is_array($id) && count($id) === 1)
				$id = $id[0];

			if (is_array($id))
				$qb->andWhere('cp.category IN (:categoryId) OR sites.category IN (:categoryId)');
			else
				$qb->andWhere('cp.category = :categoryId OR sites.category = :categoryId');
			$qb->setParameter('categoryId', $id)
				->leftJoin('p.categoryProducts', 'cp');
		};

		return $this;
	}

	public function inSiteCategory($id)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($id) {
			// Pokud chceme hledat pro eshop, je potreba nejdriv zavolat inSite()
			if (!in_array('sites', $qb->getAllAliases()))
				$qb->leftJoin('p.sites', 'sites', Join::WITH, 'sites.isActive = 1');

			if (is_array($id) && count($id) === 1)
				$id = $id[0];

			if (is_array($id))
				$qb->andWhere('sites.category IN (:categoryId)');
			else
				$qb->andWhere('sites.category = :categoryId');
			$qb->setParameter('categoryId', $id);

			$qb->innerJoin('sites.category', 'sitesCat', Join::WITH, 'sitesCat.isPublished = 1');
		};

		return $this;
	}

	public function inOtherCategories($id)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($id) {
			// Pokud chceme hledat pro eshop, je potreba nejdriv zavolat inSite()

			if (is_array($id) && count($id) === 1)
				$id = $id[0];

			$qb->innerJoin('p.categoryProducts', 'cp', Join::WITH, is_array($id) ? 'cp.category IN (:categoryId)' : 'cp.category = :categoryId')
				->innerJoin('cp.category', 'cpCat', Join::WITH, 'cpCat.isPublished = 1')
				->setParameter('categoryId', $id);
		};

		return $this;
	}

	public function search($query)
	{
		$this->withTexts(false);

		$this->filter[] = function(QueryBuilder $qb) use ($query) {
			$qb->leftJoin('p.manufacturer', 'manu');

			$searchIn   = Config::load('searchIn');
			$searchType = Config::load('searchType');

			if ($searchIn['explodeQuery'] !== null)
				$query = explode($searchIn['explodeQuery'], $query);
			else
				$query = [$query];

			if ($searchType === 'orAnd') {
				$criteria = [
					'name'             => [],
					'code1'            => [],
					'ean'              => [],
					'code2'            => [],
					'name2'            => [],
					'shortDescription' => [],
					'description'      => [],
				];

				foreach ($query as $k => $word) {
					$word = trim((string) $word, ',. ');

					$criteria['name'][] = "CONCAT(
					CASE WHEN manu IS NOT NULL THEN manu.name ELSE 0 END,
					' ', pt.name)
					LIKE :search_" . $k;

					$criteria['code1'][] = 'p.code1 LIKE :search_' . $k;
					$criteria['ean'][]   = 'p.ean LIKE :search_' . $k;

					if ($searchIn['code2'])
						$criteria['code2'][] = 'p.code2 LIKE :search_' . $k;
					if ($searchIn['name2'])
						$criteria['name2'][] = 'pt.name2 LIKE :search_' . $k;
					if ($searchIn['shortDescription'])
						$criteria['shortDescription'][] = 'pt.shortDescription LIKE :search_' . $k;
					if ($searchIn['description'])
						$criteria['description'][] = 'pt.description LIKE :search_' . $k;

					$qb->setParameter('search_' . $k, Strings::lower("%$word%"));
				}

				$where = [];
				foreach ($criteria as $v) {
					if (empty($v)) {
						continue;
					}

					$where[] = '(' . implode(' AND ', $v) . ')';
				}

				if (!empty($where)) {
					$qb->andWhere(implode(' OR ', $where));
				}
			} else {
				foreach ($query as $k => $word) {
					$word = trim((string) $word, ',. ');
					$qb->andWhere("CONCAT(
									CASE WHEN manu IS NOT NULL THEN manu.name ELSE 0 END,
									' ', pt.name)
									LIKE :search_" . $k);
					$qb->orWhere('p.code1 LIKE :search_' . $k)
						->orWhere('p.ean LIKE :search_' . $k);

					if ($searchIn['code2'])
						$qb->orWhere('p.code2 LIKE :search_' . $k);
					if ($searchIn['name2'])
						$qb->orWhere('pt.name2 LIKE :search_' . $k);
					if ($searchIn['shortDescription'])
						$qb->orWhere('pt.shortDescription LIKE :search_' . $k);
					if ($searchIn['description'])
						$qb->orWhere('pt.description LIKE :search_' . $k);

					$qb->setParameter('search_' . $k, Strings::lower("%$word%"));
				}
			}

			if (!in_array('sites', $qb->getAllAliases()) && $this->siteIdent) {
				$qb->leftJoin('p.sites', 'sites', Join::WITH, 'sites.site = :site')
					->setParameter('site', $this->siteIdent);
			} else {
				if (!in_array('siteCat', $qb->getAllAliases()))
					$qb->leftJoin('sites.category', 'siteCat', Join::WITH, 'siteCat.isPublished = 1');
				else
					$qb->andWhere('siteCat.isPublished = 1');
			}
		};

		return $this;
	}

	public function onlyInStockOrSupplier($pvsIds = null)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($pvsIds) {
			$qb->innerJoin('p.availability', 'av', Join::WITH, 'av.canShowOnList = 1');
			$arr = [];
			if ($pvsIds)
				$arr[] = 'p.id IN (' . implode(',', $pvsIds) . ')';

			if (!empty($arr))
				$qb->andWhere(implode(' OR ', $arr));
		};

		return $this;
	}

	public function availableInSearch()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->innerJoin('p.availability', 'av', Join::WITH, 'av.canShowOnSearch = 1');
		};

		return $this;
	}

	public function selectInStockBool()
	{
		$this->select[] = function(QueryBuilder $qb) {
			if (!in_array('ps', $qb->getAllAliases()))
				$qb->leftJoin('p.suppliers', 'ps');
			$qb->addSelect('CASE WHEN p.quantity > 0 THEN 1 WHEN ps.quantity > 0 THEN 1 ELSE 0 END as hidden inStockBool');// WHEN pv.quantity > 0 THEN 1
		};

		return $this;
	}

	public function useFilter($filter)
	{
		if (isset($filter['features']))
			$this->filterFeatures($filter['features']);
		if (isset($filter['priceRange']['min']))
			$this->filterPriceMin($filter['priceRange']['min']);
		if (isset($filter['priceRange']['max']))
			$this->filterPriceMax($filter['priceRange']['max']);
		if (isset($filter['manu']))
			$this->filterManu($filter['manu']);
		//		if (isset($filter['onlyStock']) && $filter['onlyStock'] === true)
		//			$this->onlyInStockOrSupplier($filter['onlyStockPVS'] ?? null);

		return $this;
	}

	public function filterPriceMin($min)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($min) {
			$qb->andWhere('p.price >= :priceMin')->setParameter('priceMin', $min);
		};

		return $this;
	}

	public function filterPriceMax($max)
	{
		$this->filter[] = function(QueryBuilder $qb) use ($max) {
			$qb->andWhere('p.price <= :priceMax')->setParameter('priceMax', $max);
		};

		return $this;
	}

	public function filterFeatures($ids)
	{
		if ($ids)
			$this->filter[] = function(QueryBuilder $qb) use ($ids) {
				if ($ids)
					$qb->andWhere('p.id IN (:featureP)')->setParameter('featureP', $ids);
			};

		return $this;
	}

	public function filterManu($ids)
	{
		if ($ids) {
			$this->filter[] = function(QueryBuilder $qb) use ($ids) {
				$qb->andWhere('p.manufacturer ' . (is_array($ids) ? 'IN (:ids)' : '= :ids'))->setParameter('ids', $ids);
			};
		}

		return $this;
	}

	public function selectIds()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->select('p.id')->groupBy('p.id');
		};

		return $this;
	}

	public function selectQuantity()
	{
		$this->filter[] = function(QueryBuilder $qb) {
			$qb->leftJoin('p.suppliers', 'ps', Join::WITH, 'ps.isActive = 1');
		};
		$this->select[] = function(QueryBuilder $qb) {
			$qb->select('p.id, p.quantity as productQ, GROUP_CONCAT(ps.quantity) as suppliersQ, GROUP_CONCAT(IDENTITY(ps.availabilityAfterSoldOut)) as suppliersA');
		};

		return $this;
	}

	/**
	 * @param int|int[] $id
	 *
	 * @return $this
	 */
	public function byId($id)
	{
		$this->select[] = function(QueryBuilder $qb) use ($id) {
			if (is_array($id))
				$qb->andWhere('p.id IN (:id)');
			else
				$qb->andWhere('p.id = :id');
			$qb->setParameter('id', $id);
		};

		return $this;
	}

	public function orderByPosition()
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->addSelect('CASE WHEN p.position IS NULL THEN 999999999 ELSE p.position END as cPosition')
				->addOrderBy('cPosition');
		};

		return $this;
	}

	public function sortBy($sort = null, $direction = null)
	{
		if ($sort) {
			$this->select[] = function(QueryBuilder $qb) use ($sort, $direction) {
				if (is_string($sort) && strpos($sort, 'pt') === 0 && !in_array('pt', $qb->getAllAliases()))
					$qb->innerJoin('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')->setParameter('lang', $this->lang);
				$qb->orderBy($sort, $direction);
			};
		}

		return $this;
	}

	public function addSortBy($sort = null, $direction = null)
	{
		if ($sort) {
			$this->select[] = function(QueryBuilder $qb) use ($sort, $direction) {
				$qb->addOrderBy($sort, $direction);
			};
		}

		return $this;
	}

	protected function doCreateQuery($repository)
	{
		$qb = $this->createBasicDql($repository);

		foreach ($this->select as $modifier) {
			$modifier($qb);
		}

		return $qb->addOrderBy('p.id', 'DESC');
	}

	protected function doCreateCountQuery($repository)
	{
		return $this->createBasicDql($repository)->select('count(p.id)');
	}

	private function createBasicDql($repository)
	{
		/** @var QueryBuilder $qb */
		$qb = $repository->createQueryBuilder('p', 'p.id')
			->addSelect('IDENTITY(p.gallery) as gallery')
			->addSelect('IDENTITY(p.manufacturer) as manufacturer');

		if (!self::$ignoreProductPublish)
			$qb = $qb->andWhere('p.isPublished = 1');

		$qb->andWhere('p.isDeleted = 0');

		foreach ($this->filter as $modifier) {
			$modifier($qb);
		}

		return $qb;
	}

	/**
	 * @param $repository
	 *
	 * @return QueryBuilder
	 */
	public function getQueryBuilder($repository)
	{
		return $this->doCreateQuery($repository);
	}
}
