<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gallery\Model\Entities\Image;
use Gedmo\Mapping\Annotation as Gedmo;
use Gallery\Model\Entities\Album;
use Core\Model\Entities\TId;
use Nette\Utils\DateTime;

/**
 * @ORM\Table(name="eshop_catalog__product", indexes={
 *     @ORM\Index(name="published", columns={"is_published", "id"}),
 *     @ORM\Index(name="published_cat", columns={"is_published", "id_category_default", "id"}),
 *     @ORM\Index(name="deleted_id", columns={"is_deleted", "id"}),
 *     @ORM\Index(name="disable_listing", columns={"disable_listing"}),
 * })
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "ProductListener"})
 */
class Product
{
	public const UNITS = ['pieces', 'kg', 'g', 'l', 'dl', 'ml'];

	use TId;
	use TTranslateListener;
	use TExtraField;

	const EXTRA_FIELD_SECTION = 'eshopCatalogProduct';

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", length=1)
	 */
	public $isPublished = 0;

	/**
	 * @var int
	 * @ORM\Column(name="in_stock", type="smallint", length=1)
	 */
	public $inStock;

	/**
	 * @var int
	 * @ORM\Column(name="quantity", type="integer", options={"default": 0}, nullable=false)
	 */
	public $quantity;

	/**
	 * @ORM\Column(name="minimum_amount", type="integer", options={"default": 1, "unsigned": true, "comment": "Minimum quantity of products per order"}, nullable=false)
	 */
	public int $minimumAmount = 1;

	/**
	 * @ORM\Column(name="maximum_amount", type="integer", options={"unsigned": true, "comment": "Maximum quantity of products per order"}, nullable=true)
	 */
	public ?int $maximumAmount = null;

	/**
	 * @var double
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $price;

	/**
	 * @var double
	 * @ORM\Column(name="retail_price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $retailPrice;

	/**
	 * @var double
	 * @ORM\Column(name="purchase_price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $purchasePrice;

	/**
	 * @var ProductPrice[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="ProductPrice", mappedBy="product", indexBy="country", cascade={"persist", "remove"})
	 */
	public $prices;

	/**
	 * @var ProductPriceLevelCountry[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="ProductPriceLevelCountry", mappedBy="product", cascade={"persist", "remove"})
	 */
	public $priceLevelCountries;

	/**
	 * @var double
	 * @ORM\Column(name="recycling_fee", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $recyclingFee;

	/**
	 * @var Manufacturer
	 * @ORM\ManyToOne(targetEntity="Manufacturer", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_manufacturer", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $manufacturer;

	/**
	 * @var ProductTexts[]
	 *
	 * @ORM\OneToMany(targetEntity="ProductTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 */
	protected $productTexts;

	/**
	 * @var CategoryProduct[]
	 * @ORM\OneToMany(targetEntity="CategoryProduct", mappedBy="product", indexBy="category", cascade={"all"})
	 */
	protected $categoryProducts;

	/**
	 * @ORM\ManyToOne(targetEntity="Category", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_category_default", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $idCategoryDefault;

	/**
	 * @var FeatureProduct[]
	 * @ORM\OneToMany(targetEntity="FeatureProduct", mappedBy="product", indexBy="featureValue", cascade={"all"})
	 */
	protected $featureProducts;

	/**
	 * @var DynamicFeatureProduct[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="DynamicFeatureProduct", mappedBy="product", indexBy="id", cascade={"all"})
	 */
	public $dynamicFeatures;

	/**
	 * @var ProductTag[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="ProductTag", mappedBy="product", indexBy="id_tag", cascade={"all"})
	 */
	public $productTags;

	/**
	 * @var ProductSupplier[]|ArrayCollection
	 *
	 * @ORM\OneToMany(targetEntity="ProductSupplier", mappedBy="product", indexBy="id_supplier", cascade={"all"})
	 */
	public $suppliers;

	/**
	 * @var VatRate
	 * @ORM\ManyToOne(targetEntity="VatRate")
	 * @ORM\JoinColumn(name="id_vat_rate", referencedColumnName="id", onDelete="SET NULL")
	 */
	public $vatRate;

	/**
	 * @var Availability
	 * @ORM\ManyToOne(targetEntity="Availability")
	 * @ORM\JoinColumn(name="id_availability", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $availability;

	/**
	 * @ORM\ManyToOne(targetEntity="Availability")
	 * @ORM\JoinColumn(name="id_availability_after_sold_out", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	public ?Availability $availabilityAfterSoldOut = null;

	/**
	 * @var string
	 * @ORM\Column(name="ean", type="string", length=20, nullable=true)
	 */
	public $ean;

	/**
	 * @var string
	 * @ORM\Column(name="code1", type="string", length=60, nullable=true)
	 */
	public $code1;

	/**
	 * @var string
	 * @ORM\Column(name="code2", type="string", length=60, nullable=true)
	 */
	public $code2;

	/**
	 * @var Album
	 *
	 * @ORM\ManyToOne(targetEntity="Gallery\Model\Entities\Album", cascade={"persist"})
	 * @ORM\JoinColumn(name="gallery_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $gallery;

	/**
	 * @var DateTime
	 * @ORM\Column(name="created", type="datetime")
	 */
	private $created;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(name="modified", type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * @var ProductPriceLevel[]
	 * @ORM\OneToMany(targetEntity="ProductPriceLevel", mappedBy="productId", indexBy="group_id")
	 */
	public $priceLevels;

	/**
	 * @var array
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected $moreData = [];

	/**
	 * @var int
	 * @ORM\Column(name="unlimited_quantity", type="smallint", options={"default": 0}, nullable=false)
	 */
	public $unlimitedQuantity;

	/**
	 * @var int
	 * @ORM\Column(name="discount_disabled", type="smallint", options={"default": 0}, nullable=false)
	 */
	public $discountDisabled;

	/**
	 * @var int
	 * @ORM\Column(name="category_gifts_allowed", type="smallint", options={"default": 1}, nullable=false)
	 */
	protected $categoryGiftsAllowed;

	/**
	 * @var ProductInSite[];
	 * @ORM\OneToMany(targetEntity="ProductInSite", mappedBy="product", indexBy="site")
	 */
	public $sites;

	/**
	 * @var ProductVariant[]|object
	 */
	public $variants;

	/**
	 * @var ProductVariant[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="ProductVariant", mappedBy="product", cascade={"persist"})
	 */
	public $isVariant;

	/**
	 * @var ArrayCollection|ProductDocument[]
	 * @ORM\OneToMany(targetEntity="ProductDocument", mappedBy="product", cascade={"all"})
	 */
	public $documents;

	/**
	 * @var ArrayCollection|ProductVideo[]
	 * @ORM\OneToMany(targetEntity="ProductVideo", mappedBy="product", cascade={"all"})
	 */
	public $videos;

	/**
	 * @var ArrayCollection|RelatedProduct[]
	 * @ORM\OneToMany(targetEntity="RelatedProduct", mappedBy="origin", indexBy="product_id")
	 */
	public $relatedProducts;

	/**
	 * @ORM\Column(name="is_assort", type="smallint", length=1, options={"default": 0})
	 */
	public int $isAssort = 0;

	/**
	 * @var string|null
	 * @ORM\Column(name="`condition`", type="string", length=255, nullable=true, options={"default": "new"})
	 */
	public ?string $condition = 'new';

	/**
	 * @var int
	 * @ORM\Column(name="disable_pick_up_spedition", type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $disablePickUpSpedition = 0;

	/**
	 * @var int
	 * @ORM\Column(name="is_oversize", type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $isOversize = 0;

	/**
	 * @ORM\Column(name="disable_stack_in_cart", type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $disableStackInCart = 0;

	/**
	 * @var int
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $isDiscount = 0;

	/**
	 * @var ProductSpedition[]
	 * @ORM\OneToMany(targetEntity="ProductSpedition", mappedBy="product", indexBy="spedition_id")
	 */
	public $speditions;

	/**
	 * In mm
	 * @ORM\Column(name="width", type="integer", options={"unsigned": true}, nullable=true)
	 */
	public ?int $width = null;

	/**
	 * In mm
	 * @ORM\Column(name="height", type="integer", options={"unsigned": true}, nullable=true)
	 */
	public ?int $height = null;

	/**
	 * In mm
	 * @ORM\Column(name="depth", type="integer", options={"unsigned": true}, nullable=true)
	 */
	public ?int $depth = null;

	/**
	 * In g
	 * @ORM\Column(name="weight", type="integer", options={"unsigned": true}, nullable=true)
	 */
	public ?int $weight = null;

	/**
	 * @ORM\Column(type="smallint", nullable=true)
	 */
	public ?int $position = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Country")
	 * @ORM\JoinColumn(name="country_of_origin", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	public ?Country $countryOfOrigin = null;

	/**
	 * @ORM\Column(name="hs_customs_code", type="integer", length=10, options={"unsigned": true}, nullable=true)
	 */
	public ?int $hsCustomsCode = null;

	/**
	 * @ORM\Column(name="is_deleted", type="integer", options={"unsgined": true, "default": 0}, nullable=false)
	 */
	public int $isDeleted = 0;

	/**
	 * @ORM\Column(type="string", length=255, options={"default": "pieces"}, nullable=false)
	 */
	public string $units = 'pieces';

	/**
	 * @ORM\Column(name="disable_listing", type="smallint", options={"unsgined": true, "default": 0}, nullable=false)
	 */
	public int $disableListing = 0;

	public function __construct()
	{
		$this->productTexts         = new ArrayCollection();
		$this->categoryProducts     = new ArrayCollection();
		$this->featureProducts      = new ArrayCollection();
		$this->suppliers            = new ArrayCollection();
		$this->productTags          = new ArrayCollection();
		$this->quantity             = 0;
		$this->inStock              = 0;
		$this->moreData             = [];
		$this->created              = new DateTime('now');
		$this->priceLevels          = new ArrayCollection();
		$this->extraFields          = new ArrayCollection();
		$this->unlimitedQuantity    = 0;
		$this->discountDisabled     = 0;
		$this->categoryGiftsAllowed = 1;
		$this->isPublished          = 1;
		$this->isVariant            = new ArrayCollection();
		$this->sites                = new ArrayCollection();
		$this->variants             = new ArrayCollection();
		$this->documents            = new ArrayCollection();
		$this->relatedProducts      = new ArrayCollection();
		$this->speditions           = new ArrayCollection();
		$this->prices               = new ArrayCollection();
		$this->priceLevelCountries  = new ArrayCollection();
		$this->dynamicFeatures      = new ArrayCollection();
		$this->position             = null;
	}

	public function __clone()
	{
		$this->id                  = null;
		$this->quantity            = 0;
		$this->gallery             = null;
		$this->isVariant           = new ArrayCollection();
		$this->productTexts        = new ArrayCollection();
		$this->categoryProducts    = new ArrayCollection();
		$this->featureProducts     = new ArrayCollection();
		$this->dynamicFeatures     = new ArrayCollection();
		$this->suppliers           = new ArrayCollection();
		$this->productTags         = new ArrayCollection();
		$this->priceLevels         = new ArrayCollection();
		$this->extraFields         = new ArrayCollection();
		$this->sites               = new ArrayCollection();
		$this->variants            = new ArrayCollection();
		$this->documents           = new ArrayCollection();
		$this->relatedProducts     = new ArrayCollection();
		$this->created             = new DateTime('now');
		$this->prices              = new ArrayCollection();
		$this->priceLevelCountries = new ArrayCollection();
		$this->isDeleted           = 0;
	}

	public function clonePrices(Product $product): array
	{
		$prices = [];
		foreach ($this->prices->toArray() as $row) {
			/** @var ProductPrice $row */
			$price              = new ProductPrice($product, $row->getCountry(), $row->currency);
			$price->price       = $row->price;
			$price->retailPrice = $row->retailPrice;
			$price->vatRate     = $row->vatRate;

			$prices[] = $price;
		}

		return $prices;
	}

	public function clonePriceLevelCounties(Product $product): array
	{
		$prices = [];
		foreach ($this->priceLevelCountries->toArray() as $row) {
			/** @var ProductPriceLevelCountry $row */
			$price        = new ProductPriceLevelCountry($product, $row->getGroup(), $row->getCountry(), $row->currency);
			$price->price = $row->price;

			$prices[] = $price;
		}

		return $prices;
	}

	public function addProductText($lang)
	{
		$this->productTexts->set($lang, new ProductTexts($this, $lang));
	}

	public function setProductTexts(array $texts): self
	{
		$this->productTexts = new ArrayCollection($texts);

		return $this;
	}

	public function setProductText(ProductTexts $productTexts)
	{
		$this->productTexts->set($productTexts->getLang(), $productTexts);
	}

	/**
	 * @deprecated use getText
	 */
	public function getProductText($lang = null)
	{
		return $this->productTexts->get($lang ?: $this->locale) ?? null;
	}

	/**
	 * @param null $lang
	 *
	 * @return ProductTexts|null
	 */
	public function getText($lang = null)
	{
		return $this->productTexts->get($lang ?: $this->locale) ?? null;
	}

	/**
	 * @deprecated use getTexts
	 */
	public function getProductTexts() { return $this->productTexts; }

	public function getTexts() { return $this->productTexts; }

	public function getFeatureProducts()
	{
		return $this->featureProducts;
	}

	public function addFeatureProduct($featureProduct)
	{
		$this->featureProducts->add($featureProduct);
	}

	/**
	 * @param FeatureProduct[] $features
	 */
	public function setFeaturesProduct(array $features)
	{
		$this->featureProducts = new ArrayCollection($features);
	}

	/**
	 * @param CategoryProduct $categoryProduct
	 *
	 * @return $this
	 */
	public function addCategoryProduct(CategoryProduct $categoryProduct)
	{
		if (!$this->categoryProducts->contains($categoryProduct))
			$this->categoryProducts->add($categoryProduct);

		return $this;
	}

	public function removeCategoryProduct(CategoryProduct $categoryProduct)
	{
		$this->categoryProducts->removeElement($categoryProduct);
	}

	/**
	 * @return ArrayCollection|CategoryProduct[]
	 */
	public function getCategoryProducts()
	{
		return $this->categoryProducts;
	}

	public function removeAllCategoryProducts()
	{
		$this->categoryProducts->clear();
	}

	public function setCategories(array $categories): self
	{
		$this->categoryProducts = new ArrayCollection($categories);

		return $this;
	}

	public function removeProductTag(ProductTag $tag): self
	{
		$this->productTags->removeElement($tag);

		return $this;
	}

	public function addProductTag(ProductTag $productTag)
	{
		$productTag->setProduct($this);
		$this->productTags->add($productTag);
	}

	public function getProductTags()
	{
		return $this->productTags;
	}

	public function setManufacturer(?Manufacturer $manufacturer = null)
	{
		$this->manufacturer = $manufacturer;
	}

	public function getManufacturer()
	{
		return $this->manufacturer;
	}

	/**
	 * @deprecated
	 */
	public function setDefaultCategory(Category $category)
	{
		$this->idCategoryDefault = $category;
	}

	/**
	 * @deprecated
	 */
	public function getDefaultCategory()
	{
		return $this->idCategoryDefault;
	}

	public function setVateRate(VatRate $vatRate)
	{
		$this->vatRate = $vatRate;
	}

	public function getVateRate(): ?VatRate
	{
		return $this->vatRate;
	}

	public function setAvailability(?Availability $availability = null): self
	{
		$this->availability = $availability;

		return $this;
	}

	public function getAvailability(): ?Availability { return $this->availability; }

	/*************************************
	 * == Gallery
	 */

	public function getGallery() { return $this->gallery; }

	public function setGallery(Album $album): self
	{
		$this->gallery = $album;

		return $this;
	}

	public function getImage(): ?Image
	{
		$img = null;
		if ($this->gallery) {
			if ($this->isVariant() && $this->isVariant()->defaultImage)
				$img = $this->gallery->getImages()[$this->isVariant()->defaultImage] ?? null;

			if (!$img)
				$img = $this->gallery->getCoverImage();
		}

		return $img;
	}

	/*************************************
	 * == ProductPriceLevel
	 */

	public function getPriceLevels()
	{
		return $this->priceLevels;
	}

	/*************************************
	 * == Supplier
	 */

	public function getSupplier($id)
	{
		return $this->suppliers->get($id);
	}

	public function getSuppliers()
	{
		return $this->suppliers;
	}

	public function setSupplier(ProductSupplier $ps)
	{
		if (!$this->suppliers->containsKey($ps->getSupplier()->getId())) {
			$this->suppliers->clear();
			$this->suppliers->add($ps);
		}
	}

	/*************************************
	 * == Dates
	 */

	public function getCreated() { return $this->created; }

	public function getModified() { return $this->modified; }

	public function validateCreated()
	{
		if ($this->created->format('y-m-d') <= 0)
			$this->created = new DateTime();
	}

	/*************************************
	 * == MoreData
	 */

	/**
	 * @param string $key
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue($key, $default = null) { return $this->moreData[$key] ?? $default; }

	/**
	 * @param string $key
	 * @param string $value
	 *
	 * @return $this
	 */
	public function setMoreDataValue($key, $value)
	{
		if (!is_array($this->moreData))
			$this->moreData = [];

		$this->moreData[$key] = $value;

		return $this;
	}

	/**
	 * @param array $data
	 *
	 * @return $this
	 */
	public function setMoreData($data)
	{
		$this->moreData = $data;

		return $this;
	}

	public function removeMoreData($key)
	{
		unset($this->moreData[$key]);

		return $this;
	}

	/*************************************
	 * == Booleans
	 */

	public function isCategoryGiftsAllowed(): int { return (int) $this->categoryGiftsAllowed; }

	public function setCategoryGiftsAllowed($allowed): self
	{
		$this->categoryGiftsAllowed = (int) $allowed;

		return $this;
	}

	public function getIsVariantParent(): bool { return $this->isVariant() && $this->isVariant()->isDefault === 1; }

	public function getVariantParent(): ?Product
	{
		if ($this->getIsVariantParent())
			return $this;

		if ($this->variants)
			foreach ($this->variants->toArray() as $v)
				if ($v->isDefault === 1)
					return $v->getProduct();

		return null;
	}

	/**
	 * @return ProductDocument[]
	 */
	public function getDocuments(): array
	{
		return $this->documents->toArray();
	}

	/**
	 * @param ProductDocument $document
	 */
	public function addDocument(ProductDocument $document): void
	{
		$this->documents->add($document);
	}

	/**
	 * @param ProductVideo $video
	 */
	public function addVideo(ProductVideo $video): void
	{
		$this->videos->add($video);
	}

	/**
	 * @return ProductVideo[]
	 */
	public function getVideos(): array
	{
		return $this->videos->toArray();
	}

	/**
	 * @return ArrayCollection|RelatedProduct[]
	 */
	public function getRelatedProducts() { return $this->relatedProducts; }

	/**
	 * @param RelatedProduct $relatedProduct
	 */
	public function addRelatedProduct(RelatedProduct $relatedProduct): void
	{
		$contains = false;
		foreach ($this->relatedProducts->getKeys() as $keys) {
			if ($keys === $relatedProduct->getProduct()->getId()) {
				$contains = true;
				break;
			}
		}

		if ($this->getId() !== $relatedProduct->getProduct()->getId() && !$contains) {
			$this->relatedProducts->set($relatedProduct->getProduct()->getId(), $relatedProduct);
		}
	}

	public function isVariant(): ?ProductVariant
	{
		return $this->isVariant->first() ?: null;
	}

	/**
	 * @return ArrayCollection|ProductSpedition[]
	 */
	public function getSpeditions() { return $this->speditions; }

	/**
	 * @return ProductPriceLevelCountry[][]
	 */
	public function getPriceLevelCountriesIndexedByGroup(): array
	{
		$result = [];

		foreach ($this->priceLevelCountries->toArray() as $row) {
			/** @var ProductPriceLevelCountry $row */
			$result[$row->getGroup()->getId()][$row->getCountry()->getId()] = $row;
		}

		return $result;
	}
}
