<?php declare(strict_types = 1);

namespace EshopCatalog\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use EshopCatalog\Model\Entities\Product;
use Tracy\Debugger;

class PseudoWarehouse
{
	protected EntityManagerDecorator $em;
	protected EventDispatcher $dispatcher;

	public function __construct(EntityManagerDecorator $em, EventDispatcher $dispatcher)
	{
		$this->em = $em;
		$this->dispatcher = $dispatcher;
	}

	protected function getUnlimitedProducts(array $ids): array
	{
		$unlimited = [];
		foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
			         ->select('p.id')
			         ->where('p.id IN (:ids)')
			         ->setParameter('ids', array_values($ids))
			         ->andWhere('p.unlimitedQuantity = 1')
			         ->getQuery()->getScalarResult() as $row)
			$unlimited[] = $row['id'];

		return $unlimited;
	}

	public function lowerQuantities(array $data): void
	{
		try {
			$this->em->beginTransaction();
			$unlimited = $this->getUnlimitedProducts(array_keys($data));

			$changes = [];
			foreach ($data as $productId => $quantity) {
				if (in_array($productId, $unlimited))
					continue;

				$product           = $this->em->getReference(Product::class, $productId);
				$product->quantity -= $quantity;
				$this->em->persist($product);
				$this->em->flush($product);

				$change = ['product' => $product, 'quantityDiff' => $quantity];
				$changes[] = $change;
			}

			$this->dispatcher->dispatch(new Event($changes), 'pseudoWarehouse.afterReduceProductsQuantity');

			$this->em->commit();
		} catch (\Exception $e) {
			Debugger::log($e);
			$this->em->rollback();
		}
	}

	public function raiseQuantities(array $data): void
	{
		try {
			$this->em->beginTransaction();
			$unlimited = $this->getUnlimitedProducts(array_keys($data));

			foreach ($data as $productId => $quantity) {
				if (in_array($productId, $unlimited))
					continue;

				$product           = $this->em->getReference(Product::class, $productId);
				$product->quantity += $quantity;
				$this->em->persist($product);
				$this->em->flush($product);

				$change = ['product' => $product, 'quantityDiff' => $quantity];
				$changes[] = $change;
			}

			$this->dispatcher->dispatch(new Event($changes), 'pseudoWarehouse.afterIncreaseProductsQuantity');

			$this->em->commit();
		} catch (\Exception $e) {
			Debugger::log($e);
			$this->em->rollback();
		}
	}
}
