<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\Entities\ExtraField;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Helpers\Strings;
use Core\Model\Parameters;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\FrontModule\Model\SeoContainer;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\FormContainers\DocumentContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductPriceLevelCountriesContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductPricesContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductSpeditionContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductSupplierContainer;
use EshopCatalog\AdminModule\Model\FormContainers\RelatedProductContainer;
use EshopCatalog\AdminModule\Model\FormContainers\VideoContainer;
use EshopCatalog\AdminModule\Model\Helpers\ProductFormHelper;
use EshopCatalog\AdminModule\Model\ProductDocuments;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\AdminModule\Model\ProductVideos;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\DynamicFeatureProduct;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\ProductListener;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\AdminModule\Model\ProductTags;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\Supplier;
use EshopCatalog\Model\Entities\Tag;
use EshopStock\DI\EshopStockExtension;
use EshopStock\Model\Entities\SupplyProduct;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\FrontModule\Model\Albums;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Import\DI\ImportExtension;
use Nette\Application\IPresenter;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureProducts;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\FeatureProduct;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use EshopCatalog\Model\Products as BaseProductsService;

class ProductForm extends BaseControl
{
	/** @var Product|null */
	public $product;

	protected Products                            $productServices;
	protected FeatureProducts                     $featureProductServices;
	protected CategoryProducts                    $categoryProductServices;
	protected Categories                          $categoryServices;
	protected CategoryContainer                   $categoryContainerService;
	protected Manufacturers                       $manufacturerServices;
	protected Suppliers                           $supplierServices;
	protected VatRates                            $vatRateServices;
	protected Features                            $featureServices;
	protected SeoContainer                        $seoContainerService;
	protected IImagesZoneFactory                  $imagesZoneFactory;
	protected Request                             $httpRequest;
	protected TagsContainer                       $tagsContainerService;
	protected ProductTags                         $productTagsServices;
	protected ProductFormHelper                   $helper;
	protected AvailabilityService                 $availabilityService;
	protected CacheService                        $cacheService;
	protected DocumentContainer                   $documentContainer;
	protected VideoContainer                      $videoContainer;
	protected RelatedProductContainer             $relatedProductContainer;
	protected ProductDocuments                    $productDocuments;
	protected ProductVideos                       $productVideos;
	protected ProductSupplierContainer            $productSupplierContainer;
	protected ProductSpeditionContainer           $speditionContainer;
	protected Storage                             $storage;
	protected ProductPricesContainer              $productPricesContainer;
	protected ProductPriceLevelCountriesContainer $productPriceLevelCountriesContainer;
	protected BaseProductsService                 $baseProductsService;

	protected ?array $suppliersToAdd = null;
	private ?array   $cFeatures      = null;
	private ?array   $cFeatureValues = null;

	public function __construct(
		Products                            $products,
		Features                            $features,
		FeatureProducts                     $featureProducts,
		Categories                          $categories,
		CategoryProducts                    $categoryProducts,
		Manufacturers                       $manufacturers,
		Suppliers                           $suppliers,
		VatRates                            $vatRates,
		SeoContainer                        $seoContainer,
		IImagesZoneFactory                  $imagesZoneFactory,
		Request                             $request,
		CategoryContainer                   $categoryContainer,
		TagsContainer                       $tagsContainer,
		ProductTags                         $productTags,
		ProductFormHelper                   $helper,
		DocumentContainer                   $documentContainer,
		ProductDocuments                    $productDocuments,
		RelatedProductContainer             $relatedProductContainer,
		ProductSpeditionContainer           $speditionContainer,
		AvailabilityService                 $availabilityService,
		ProductSupplierContainer            $productSupplierContainer,
		Storage                             $storage,
		CacheService                        $cacheService,
		VideoContainer                      $videoContainer,
		ProductVideos                       $productVideos,
		ProductPricesContainer              $productPricesContainer,
		ProductPriceLevelCountriesContainer $productPriceLevelCountriesContainer,
		BaseProductsService                 $baseProductsService
	)
	{
		$this->productServices                     = $products;
		$this->featureServices                     = $features;
		$this->featureProductServices              = $featureProducts;
		$this->categoryContainerService            = $categoryContainer;
		$this->categoryServices                    = $categories;
		$this->categoryProductServices             = $categoryProducts;
		$this->manufacturerServices                = $manufacturers;
		$this->supplierServices                    = $suppliers;
		$this->seoContainerService                 = $seoContainer;
		$this->vatRateServices                     = $vatRates;
		$this->imagesZoneFactory                   = $imagesZoneFactory;
		$this->httpRequest                         = $request;
		$this->tagsContainerService                = $tagsContainer;
		$this->productTagsServices                 = $productTags;
		$this->helper                              = $helper;
		$this->documentContainer                   = $documentContainer;
		$this->videoContainer                      = $videoContainer;
		$this->relatedProductContainer             = $relatedProductContainer;
		$this->productDocuments                    = $productDocuments;
		$this->productVideos                       = $productVideos;
		$this->speditionContainer                  = $speditionContainer;
		$this->availabilityService                 = $availabilityService;
		$this->productSupplierContainer            = $productSupplierContainer;
		$this->storage                             = $storage;
		$this->cacheService                        = $cacheService;
		$this->productPricesContainer              = $productPricesContainer;
		$this->productPriceLevelCountriesContainer = $productPriceLevelCountriesContainer;
		$this->baseProductsService                 = $baseProductsService;
	}

	public function render(): void
	{
		$this->template->thisForm         = $this['form'];
		$this->template->product          = $this->product;
		$this->template->groupCustomers   = $this->helper->getGroupCustomers();
		$this->template->enablePriceLevel = $this->helper->config->get('enablePriceLevels', false);
		$this->template->sites            = $this->helper->sitesService->getAll();

		$features                      = $this->getFeatures();
		$this->template->features      = $features;
		$this->template->featureValues = $this->getFeatureValues();

		$defaults = [];
		if ($this->product) {
			foreach ($this->product->getFeatureProducts()->toArray() as $row)
				$defaults[] = [
					'feature' => $row->getFeature()->getId(),
					'value'   => $row->getFeatureValue()->getId(),
				];

			foreach ($this->product->dynamicFeatures as $row)
				$defaults[] = [
					'feature' => $row->getFeature()->getId(),
					'value'   => $row->value,
				];
		}
		$this->template->defaultFeatures = $defaults;

		$this->template->featuresJson = Json::encode([
			'features' => $features,
			'values'   => $this->getFeatureValues(),
		]);

		$this->template->suppliersToAdd = $this->suppliersToAdd;

		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), ProductForm::class . '::render');
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ========================  HANDLE
	 */

	public function handleShowSuppliersToAdd(): void
	{
		$this->suppliersToAdd = [];
		$availabilityPairs    = [];
		$supplierPairs        = [];

		$qb = $this->em->createQueryBuilder();
		$qb->select('a.id, at.name')
			->from(Availability::class, 'a')
			->leftJoin('a.texts', 'at', 'WITH', 'at.lang = :lang')
			->orderBy('a.position')
			->setParameters([
				'lang' => $this->translator->getLocale(),
			]);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$availabilityPairs[$row['id']] = $row['name'];
		}

		$ids = array_map(fn(ProductSupplier $ps) => $ps->getSupplier()->getId(), $this->product->getSuppliers()->toArray());
		$qb  = $this->em->getRepository(Supplier::class)->createQueryBuilder('s');
		$qb->select('s.id, s.name')
			->orderBy('s.name');

		if ($ids) {
			$qb->andWhere($qb->expr()->notIn('s.id', $ids));
		}

		$container = new BaseContainer;
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$this->suppliersToAdd[$row['id']] = $row['name'];
			$c                                = $this->productSupplierContainer->getContainer();
			$c->addCustomData('name', $row['name']);
			$container->addComponent($c, 'supp_' . $row['id']);
		}
		$this['form']->addComponent($container, 'suppliersContainer');

		$this->redrawControl('suppliers-to-add-snippet');
	}

	public function handleRemoveDocument($documentId)
	{
		foreach ($this->product->getDocuments() as $key => $doc) {
			if ($doc->getId() === (int) $documentId) {
				$this->productDocuments->remove((int) $documentId);
			}
		}
	}

	public function handleRemoveVideo($videoId)
	{
		foreach ($this->product->getVideos() as $key => $v) {
			if ($v->getId() === (int) $videoId) {
				$this->productVideos->remove((int) $videoId);
			}
		}
	}

	public function handleAddFeatureValue(string $id, string $val)
	{
		if (!$val)
			$this->getPresenter()->sendJson([]);

		$exist = $this->em->getRepository(FeatureValueTexts::class)->createQueryBuilder('fvt')
			->select('IDENTITY(fvt.id) as id, fvt.name as name')
			->innerJoin('fvt.id', 'fv', Join::WITH, 'fv.feature = :feature')
			->where('fvt.name = :v')
			->setParameters([
				'feature' => $id,
				'v'       => $val,
			])
			->getQuery()->getResult()[0] ?? null;

		if (!$exist) {
			$featureValue = new FeatureValue();
			$featureValue->setFeature($this->featureServices->getReference((int) $id));
			$featureValue->isPublished = 1;
			$this->em->persist($featureValue);

			foreach ($this->langsService->getLangs(false) as $lang) {
				$featureValueText = new FeatureValueTexts($featureValue, $lang->getTag());
				$featureValueText->setName($val);
				$this->em->persist($featureValueText);
			}

			$this->em->flush();
			$exist = [
				'id'      => $featureValue->getId(),
				'name'    => $val,
				'created' => true,
			];
		}

		$this->getPresenter()->sendJson([
			'data' => $exist,
		]);
	}

	public function handleCreateVariant(int $id): void
	{
		$variant = $this->productServices->createVariant($id);

		$this->getPresenter()->flashMessageSuccess('eshopCatalog.product.variantCreated');
		$this->getPresenter()->redirect('edit', $variant->getId());
	}

	public function handleRemoveFromVariants(): void
	{
		if ($this->productServices->removeFromVariants($this->product))
			$this->getPresenter()->flashMessageSuccess('default.saved');
		else
			$this->getPresenter()->flashMessageDanger('default.error');
		$this->redirect('this');
	}

	public function handleSetAsVariantFor(): void
	{
		if ($this->productServices->setAsVariantFor($this->product->getId(), (int) $this->getPresenter()->getHttpRequest()->getPost('productId'))) {
			$this->getPresenter()->flashMessageSuccess('default.saved');
			$this->redirect('this');
		} else {
			$this->getPresenter()->flashMessageDanger('eshopCatalog.productForm.setAsVariantForError');
			$this->getPresenter()->redrawControl('flashes');
		}
	}

	public function handleSetAsMainVariant(int $id): void
	{
		if ($this->productServices->setAsMainVariant($id)) {
			$this->getPresenter()->flashMessageSuccess('default.saved');
			$this->getPresenter()->redirect('edit', $id);
		} else {
			$this->getPresenter()->flashMessageDanger('default.errorl');
			$this->getPresenter()->redrawControl('flashes');
		}
	}

	public function handleDuplicateProduct(int $id): void
	{
		$newProductId = $this->productServices->duplicateProduct($id);
		if ($newProductId) {
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.product.duplicated');
			$this->getPresenter()->redirect('edit', $newProductId);
		} else {
			$this->getPresenter()->flashMessageDanger('eshopCatalog.product.duplicateFailed');
			$this->getPresenter()->redrawControl('flashes');
		}
	}

	/*******************************************************************************************************************
	 * ========================  COMPONENT
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$this->monitor(IPresenter::class, function() use ($form) {
			if ($this->product) {
				$el = Html::el()
					->addHtml(Html::el('a', [
						'href'  => $this->link('createVariant!', $this->product->getId()),
						'class' => 'btn ajax',
					])->setText($this->t('eshopCatalog.product.createVariant')))
					->addHtml(Html::el('a', [
						'href'  => $this->link('duplicateProduct!', $this->product->getId()),
						'class' => 'btn ajax',
					])->setText($this->t('eshopCatalog.product.duplicateProduct')));
				$form->addHeadLinkEl($el);

				if ($this->product->isPublished) {
					foreach ($this->product->sites as $site) {
						if ($site->isActive()) {
							$link = Html::el('a', [
								'href'   => '//' . $site->getSite()->texts->first()->domain . '/eshopcatalog/product/' . $this->product->getId(),
								'target' => '_blank',
								'class'  => 'btn btn-success',
							])->addHtml('<i class="fa fa-eye"></i>')
								->addHtml('<span>' . $this->t('eshopCatalog.product.showProduct') . '</span>');

							$form->addHeadLinkEl($link);
						}
					}
				}
			}
		});

		$form->addBool('isPublished', 'eshopCatalog.productForm.isPublished')->setDefaultValue(1);
		$form->addText('name', 'eshopCatalog.productForm.name')->setMaxLength(255)->setIsMultilanguage();
		if (Config::load('enableProductName2')) {
			$form->addText('name2', 'eshopCatalog.productForm.name2')->setMaxLength(255)
				->setIsMultilanguage()->setNullable();
		}

		if (Config::load('product.allowModifyAlias')) {
			$form->addText('alias', 'eshopCatalog.product.alias')
				->setIsMultilanguage();
		}

		$form->addText('ean', 'eshopCatalog.productForm.ean')->setMaxLength(13);
		$form->addText('code1', 'eshopCatalog.productForm.code1')->setMaxLength(60);
		$form->addText('code2', 'eshopCatalog.productForm.code2')->setMaxLength(60);

		if (Config::load('product.allowAssort'))
			$form->addBool('isAssort', 'eshopCatalog.productForm.isAssort')->setDefaultValue(0);

		if (Config::load('product.allowNote')) {
			$form->addBool('addNote', 'eshopCatalog.product.allowAddNote')->setDefaultValue(0);
			$form->addText('noteTitle', 'eshopCatalog.product.noteTitle')->setIsMultilanguage();
		}

		if (self::allowSaleDiscountCouponsAsProducts()) {
			$form->addBool('isDiscount', 'eshopCatalog.productForm.isDiscount')->setDefaultValue(0);
		}

		$form->addText('position', 'eshopCatalog.productForm.position')
			->setNullable()
			->addCondition($form::FILLED)
			->addRule($form::INTEGER);

		$form->addSelect('condition', 'eshopCatalog.productForm.condition', [
			'new'         => 'eshopCatalog.productConditions.new',
			'refurbished' => 'eshopCatalog.productConditions.refurbished',
			'used'        => 'eshopCatalog.productConditions.used',
		]);

		if (Config::load('product.allowCountryOfOrigin')) {
			$form->addSelect('countryOfOrigin', 'eshopCatalog.product.countryOfOrigin', ['' => ''] + $this->helper->countries->getAllNameColumn());
		}

		if (Config::load('product.allowHSCustomsCode')) {
			$form->addText('hsCustomsCode', 'eshopCatalog.product.hsCustomsCode')
				->setHtmlType('number')
				->setHtmlAttribute('maxlength', 10)
				->setHtmlAttribute('oninput', 'javascript: if (this.value.length > this.maxLength) this.value = this.value.slice(0, this.maxLength);');
		}

		if (class_exists('\EshopProductionWarehouse\DI\EshopProductionWarehouseExtension')) {
			$units = [];
			foreach (Product::UNITS as $v) {
				$units[$v] = sprintf('eshopCatalog.productForm.units.items.%s', $v);
			}
			$form->addSelect('units', 'eshopCatalog.productForm.units.caption', $units)
				->setRequired();
		}
		$form->addInteger('quantity', 'eshopCatalog.productForm.quantity')
			->setDescription('eshopCatalog.productForm.quantityDescription')
			->setDefaultValue(0);
		if ($this->product && class_exists('\EshopProductionWarehouse\DI\EshopProductionWarehouseExtension')) {
			$form['quantity']->setDisabled();
		}
		$form->addInteger('minimumAmount', 'eshopCatalog.productForm.minimumAmount')
			->setDefaultValue(1);
		if (Config::load('product.allowMaximumQuantityPerOrder')) {
			$form->addInteger('maximumAmount', 'eshopCatalog.productForm.maximumAmount');
		}
		if (Config::load('pseudoWarehouse'))
			$form->addBool('unlimitedQuantity', 'eshopCatalog.productForm.unlimitedQuantity')->setDefaultValue(0);
		if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
			$form->addBool('categoryGiftsAllowed', 'eshopCatalog.productForm.categoryGiftsAllowed')->setDefaultValue(1);
		$form->addSelect('availability', $this->t('eshopCatalog.productForm.availability'), $this->helper->availabilityService->getOptionsForSelect())
			->setTranslator(null);
		$form->addSelect('availabilityAfterSoldOut', $this->t('eshopCatalog.productForm.availabilityAfterSoldOut'), [null => ''] + $this->helper->availabilityService->getOptionsForSelect())
			->setTranslator(null);

		if (Config::load('product.allowPreorder'))
			$form->addText('preorderText', 'eshopCatalog.productForm.preorderText')
				->setIsMultilanguage();

		$form->addText('price', 'eshopCatalog.productForm.price' . (Config::load('product.priceIsWithoutVat', false) ? 'WithoutVat' : ''))
			->setHtmlType('number')
			->setHtmlAttribute('step', .01)
			->setHtmlAttribute('min', 0)
			->setRequired();

		if (Config::load('product.allowRetailPrice')) {
			$form->addText('retailPrice', 'eshopCatalog.productForm.retailPrice' . (Config::load('product.priceIsWithoutVat', false) ? 'WithoutVat' : ''))
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0)
				->setNullable();
		}

		// DPH
		$vatRates = [];
		foreach ($this->vatRateServices->getAll() as $vrs)
			$vatRates[$vrs->getId()] = $vrs->name;
		$form->addSelect('vatRate', 'eshopCatalog.productForm.vatRate', $vatRates)
			->setRequired();

		if (Config::load('enablePurchasePrice'))
			$form->addText('purchasePrice', 'eshopCatalog.productForm.purchasePrice')
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0)
				->setNullable();
		$form->addEditor('shortDesc', 'eshopCatalog.productForm.shortDesc')->setToolbar('Text')->setDisableAutoP()->setHeight(100)
			->setMaxLength(250)->setIsMultilanguage();
		$form->addEditor('desc', 'eshopCatalog.productForm.desc')->setHeight(300)->setIsMultilanguage();

		$form->addText('recyclingFee', 'eshopCatalog.productForm.recyclingFee')
			->setHtmlType('number')
			->setHtmlAttribute('step', .01)
			->setHtmlAttribute('min', 0);

		if (class_exists(ImportExtension::class)) {
			$form->addBool('stopImportPrice', 'eshopCatalog.productForm.stopImportPrice')->setDefaultValue(1);
		}

		if (Config::load('product.allowStopImportQuantity')) {
			$form->addBool('stopImportQuantity', 'eshopCatalog.productForm.stopImportQuantity')->setDefaultValue(0);
		}

		if (Config::load('product.loadQuantityOnStock') && class_exists(EshopStockExtension::class)) {
			$form->addText('quantityOnStock', 'eshopCatalog.productForm.quantityOnStock')
				->setDefaultValue(0)
				->setHtmlAttribute('readonly', true)
				->setOmitted();
		}

		$form->addBool('discountDisabled', 'eshopCatalog.productForm.discountDisabled')->setDefaultValue(1);

		if (Config::load('product.allowDisablePickUpSpedition'))
			$form->addBool('disablePickUpSpedition', 'eshopCatalog.product.disablePickUpSpedition')->setDefaultValue(0);
		if (Config::load('product.allowMarkAsOversize'))
			$form->addBool('isOversize', 'eshopCatalog.product.isOversize')->setDefaultValue(0);
		if (Config::load('product.allowDisableStackInCart'))
			$form->addBool('disableStackInCart', 'eshopCatalog.product.disableStackInCart')->setDefaultValue(0);

		$form->addSelect('manufacturer', $this->t('eshopCatalog.productForm.manufacturer'), ['' => ''] + $this->manufacturerServices->getOptionsForSelect())
			->setTranslator(null);

		//Suppliers - Dodavatele
		// Aktuálně pouze zobrazení z importu

		//PriceLevel
		if ($this->helper->config->get('enablePriceLevels', false)) {
			foreach ($this->helper->getGroupCustomers() as $groupCustomers) {
				$nameSuffix = '';
				if (Config::load('product.priceIsWithoutVat', false))
					$nameSuffix = ' ' . $this->t('eshopCatalog.withoutVat');
				$form->addText('pricelevel' . $groupCustomers->getId(), $groupCustomers->name . $nameSuffix)
					->setHtmlType('number')
					->setHtmlAttribute('step', .01)
					->setHtmlAttribute('min', 0)
					->setNullable();
			}
		}

		// Ceny podle zeme
		if (Config::load('enableCountryPrices')) {
			$form->addComponent($this->productPricesContainer->getContainer(), 'prices');
			$form->addComponent($this->productPriceLevelCountriesContainer->getContainer(), 'priceLevelCountries');
		}

		// Kategorie a Eshopy
		$form->addComponent($this->categoryContainerService->getContainers(), 'categories');
		if (Config::load('product.setDefaultTrueVisibleInEshop') && !$this->product) {
			foreach ($form->getComponent('categories')->getComponents() as $catSiteComponent) {
				$catSiteComponent->setDefaults([
					'isActive' => 1
				]);
			}
		}

		//Features - Vlastnosti
		$form->addContainer('features');

		//Variants
		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$form->addComponent($this->tagsContainerService->getContainer(), 'tags');

		// Doprava
		if (Config::load('product.allowModifySpedition', false)) {
			$form->addComponent($this->speditionContainer->getContainer(), 'speditions');
		}

		// Rozměry
		$form->addInteger('width', $this->t('eshopCatalog.productForm.width'))
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setNullable();
		$form->addInteger('height', $this->t('eshopCatalog.productForm.height'))
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setNullable();
		$form->addInteger('depth', $this->t('eshopCatalog.productForm.depth'))
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setNullable();
		$form->addInteger('weight', $this->t('eshopCatalog.productForm.weight'))
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setNullable();

		$form->addHidden('preparedAlbumId');

		$event          = new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null);
		$event->control = $this;
		$this->eventDispatcher->dispatch($event, ProductForm::class . '::createForm');

		$form->addSaveCancelControl('saveControl');

		// Nastaveni pro varianty
		if ($this->product && $this->product->isVariant()) {
			if ($this->product->getIsVariantParent()) {
				$form->addText('variantName', 'eshopCatalog.product.variantName')
					->setMaxLength(255)
					->setNullable();
			} else {
				$variantUse    = $form->addContainer('variantUse');
				$parentProduct = $this->product->getVariantParent();

				foreach (get_object_vars($this->product->isVariant()) as $k => $v) {
					if (in_array($k, ['defaultImage', 'createdDefault']))
						continue;
					$variantUse->addBool($k, 'eshopCatalog.product.useForVariant')->setDefaultValue($v ? 1 : 0);
				}

				$form['manufacturer']->setDisabled()
					->setDefaultValue($parentProduct->getManufacturer() ? $parentProduct->getManufacturer()->getId() : null);
				$form['vatRate']->setDisabled()
					->setDefaultValue($parentProduct->getVateRate() ? $parentProduct->getVateRate()->getId() : null);
			}
		}
		// Dokumenty
		$form->addComponent($this->documentContainer->getContainer(), 'documents');

		// Dokumenty
		$form->addComponent($this->videoContainer->getContainer(), 'videos');

		// Souvisejici produkty
		$form->addComponent($this->relatedProductContainer->getContainer(), 'relatedProducts');

		$this->monitor(IPresenter::class, function(AbstractPresenter $presenter) use ($form) {
			$form['relatedProducts']['product']->getControlPrototype()->setAttribute('data-link', $presenter->link(':EshopCatalog:Cron:Products:loadAll', [
				'excluded' => $this->product ? [$this->product->getId()] : [],
			]));

			$this->template->renderUseVariant = function() { return ''; };
			if ($this->product) {
				// vzkresleni zapnuti hodnot pro varianty
				$this->template->isVariant        = $this->product->isVariant() && !$this->product->getIsVariantParent();
				$this->template->renderUseVariant = function(string $c) use ($form) {
					if (isset($form['variantUse']) && in_array($c, ['vatRate']))
						return Html::el('div', ['style' => 'height: 2.8rem;']);

					$c = 'use' . ucfirst($c);
					if (isset($form['variantUse'][$c]))
						return Html::el('div', [
							'class'             => 'variant-use-bool ' . ($form['variantUse'][$c]->getValue() ? 'active' : ''),
							'data-variant-bool' => $c,
						])->addHtml($form->getRenderer()->renderPair($form['variantUse'][$c]));

					return '';
				};
			}
		});

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		$nameFilled = false;
		foreach ($values->name as $v) {
			if ($v) {
				$nameFilled = true;
				break;
			}
		}

		if (!$nameFilled) {
			$form->addError('eshopCatalog.productForm.nameMissing');
		}

		// Duplikace aliasu
		if (Config::load('allowValidateDuplicateTextsProduct')) {
			$skipCheck = Config::load('disableValidateDuplicateTextForLanguage') ?? [];
			foreach (['name', 'alias'] as $col) {
				if ($col === 'alias' && !Config::load('product.allowModifyAlias')) {
					continue;
				}

				$data = $values->{$col};

				if (!$data) {
					continue;
				}

				$qb = $this->productServices->em->getRepository(ProductTexts::class)->createQueryBuilder('pt')
					->select('IDENTITY(pt.id) as id, pt.lang, pt.' . $col)
					->innerJoin('pt.id', 'p', Join::WITH, 'p.isPublished = 1');

				$orWhere = [];
				foreach ($this->langsService->getLangs(false) as $langKey => $lang) {
					if (!isset($data[$langKey]) || in_array($langKey, $skipCheck, true)) {
						continue;
					}

					$orWhere[] = "(pt.lang = '{$langKey}' AND pt.{$col} = '" . ($col === 'alias' ? Strings::webalize($data[$langKey]) : $data[$langKey]) . "')";
				}

				if (!empty($orWhere)) {
					$qb->andWhere(implode(' OR ', $orWhere));
				}

				foreach ($qb->getQuery()->getArrayResult() as $row) {
					if (
						!$form[$col]->getErrors()
						&& $row[$col] && (
							!$this->product
							|| ($this->product->getId() !== $row['id'])
						)
					) {
						$form[$col]->addError($this->t('eshopCatalog.productForm.duplicateField', [
							'lang'  => Strings::upper($row['lang']),
							'field' => $this->translator->translate('eshopCatalog.productForm.' . $col),
						]));
					}
				}
			}
		}

		$event                   = new FormValidateEvent($form, $values,
			$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
		$event->custom['entity'] = $this->product;
		if ($this->getPresenter(false))
			$event->presenter = $this->getPresenter();
		$event->control = $this;
		$this->eventDispatcher->dispatch($event, ProductForm::class . '::formValidate');

		if ($form->hasErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$beforeFormSuccessEvent = new FormSuccessEvent($form, $values,
			$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null
		);
		$beforeFormSuccessEvent->control = $this;
		$this->eventDispatcher->dispatch($beforeFormSuccessEvent, self::class . '::beforeFormSuccess');

		$this->em->beginTransaction();
		try {
			ProductListener::$updateVariant = false;
			$langValues                     = $form->convertMultilangValuesToArray();
			if ($this->product) {
				$product = $this->product;
			} else {
				$product = new Product();

				$this->em->persist($product);
				$this->em->flush();
			}

			// Ulozeni hodnot pouziti pro variantu
			if ($product->isVariant() && !$product->getIsVariantParent()) {
				foreach ($values->variantUse as $k => $v) {
					if (in_array($k, ['product']) || $product->isVariant()->isDefault == 0 && $k == 'variantName')
						continue;

					$product->isVariant()->$k = $v;
				}

				$this->em->persist($product->isVariant());
				$this->em->flush($product->isVariant());
			}

			/** @var ProductVariant|null $va */
			$va = $product->isVariant();

			if (!$form['quantity']->isDisabled()) {
				$product->quantity = $values->quantity;
				$product->inStock  = $values->quantity > 0 ? 1 : 0;
			}

			$product->minimumAmount    = $values->minimumAmount;
			$product->price            = $values->price;

			if (Config::load('product.allowRetailPrice')) {
				$product->retailPrice = $values->retailPrice ?: null;
			}

			$product->purchasePrice    = $values->purchasePrice ?: null;
			$product->recyclingFee     = $values->recyclingFee ?: null;
			$product->isPublished      = $values->isPublished;
			$product->isDiscount       = (int) (self::allowSaleDiscountCouponsAsProducts() ? $values->isDiscount : 0);
			$product->position         = Validators::isNone($values->position) ? null : ((int) $values->position);
			$product->ean              = $values->ean;
			$product->code1            = $values->code1;
			$product->code2            = $values->code2;
			$product->discountDisabled = $values->discountDisabled == 1 ? 0 : 1;
			$product->isAssort         = Config::load('product.allowAssort')
				? ($values->isAssort ? 1 : 0)
				: 0;
			$product->condition        = $values->condition;
			$product->setAvailability($this->helper->availabilityService->getReference($values->availability));
			$product->availabilityAfterSoldOut = $values->availabilityAfterSoldOut ? $this->helper->availabilityService->getReference($values->availabilityAfterSoldOut) : null;

			if ($form->getComponent('maximumAmount', false)) {
				$product->maximumAmount = is_numeric($values->maximumAmount) && $values->maximumAmount > 0 ? (int) $values->maximumAmount : null;
			}

			$product->width  = $values->width;
			$product->height = $values->height;
			$product->depth  = $values->depth;
			$product->weight = $values->weight;

			if (isset($values->disablePickUpSpedition))
				$product->disablePickUpSpedition = $values->disablePickUpSpedition ? 1 : 0;
			if (isset($values->isOversize))
				$product->isOversize = $values->isOversize ? 1 : 0;
			if (isset($values->disableStackInCart))
				$product->disableStackInCart = $values->disableStackInCart ? 1 : 0;

			if (isset($values['units'])) {
				$product->units = $values['units'];
			}

			if (Config::load('product.allowCountryOfOrigin')) {
				$product->countryOfOrigin = $values->countryOfOrigin
					? $this->helper->countries->getReference($values->countryOfOrigin)
					: null;
			}

			if (Config::load('product.allowHSCustomsCode')) {
				$product->hsCustomsCode = $values->hsCustomsCode ? (int) $values->hsCustomsCode : null;
			}

			if (Config::load('pseudoWarehouse')) {
				$product->unlimitedQuantity = $values->unlimitedQuantity;
				$avs                        = $this->helper->availabilityService->getAllByIdent();

				$tmp = [$avs[Availability::IN_STOCK]->getId()];
				if (isset($avs[Availability::PREORDER]))
					$tmp[] = $avs[Availability::PREORDER]->getId();

				if ($product->unlimitedQuantity && !in_array($values['availability'], $tmp)) {
					$product->setAvailability($this->helper->availabilityService->getAllByIdent()['inStock']);
				}
			}

			if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
				$product->setCategoryGiftsAllowed($values->categoryGiftsAllowed);

			$this->em->persist($product);
			$this->em->flush();

			if ($values->preparedAlbumId)
				$product->setGallery($this->em->getRepository(Album::class)->getReference($values->preparedAlbumId));

			//vyrobce
			if (!$product->isVariant() || $product->getIsVariantParent())
				$product->setManufacturer($values->manufacturer ? $this->manufacturerServices->get($values->manufacturer) : null);

			//vatRate
			if ($values->vatRate) {
				$vatRate = $this->vatRateServices->get($values->vatRate);
				$product->setVateRate($vatRate);
			}

			// Texty
			/** @var ProductTexts[] $productTexts */
			$productTexts = $product->getTexts()->toArray();
			$seoData      = $this->seoContainerService->getFormData($values->seo, true);

			// Extra fields
			$ef = $product->getExtraFieldsByKey();

			if (Config::load('product.allowNote')) {
				$addNote = $ef['addNote'] ?? null;

				if (!$addNote) {
					$addNote = new ExtraField('addNote', $values->addNote);
					$product->fillExtraField($addNote);
				}

				$addNote->value = $values->addNote;
				$this->em->persist($addNote);

				foreach ($langValues as $l => $v) {
					$noteTitle = $ef['noteTitle'][$l] ?? null;

					if (!$noteTitle) {
						$noteTitle       = new ExtraField('noteTitle', $v['noteTitle']);
						$noteTitle->lang = $l;
						$product->fillExtraField($noteTitle);
					}

					$noteTitle->value = $v['noteTitle'];
					$this->em->persist($noteTitle);
				}
			}

			foreach ($langValues as $l => $v) {
				if (!$productTexts[$l])
					$productTexts[$l] = new ProductTexts($product, $l);

				$alias = $v['name'];
				if (Config::load('product.allowModifyAlias') && $v['alias']) {
					$alias = $v['alias'];
				}

				$productTexts[$l]->setName($v['name'])
					->setAlias(Strings::webalize((string) $alias))
					->setDescription($v['desc'])
					->setShortDescription($v['shortDesc'])
					->setSeo($seoData[$l] ?? []);

				if (Config::load('enableProductName2'))
					$productTexts[$l]->setName2($v['name2']);
				if (Config::load('product.allowPreorder'))
					$productTexts[$l]->preorderText = $v['preorderText'] ?: null;

				$this->em->persist($productTexts[$l]);
			}

			$product->setProductTexts($productTexts);

			$this->em->persist($product);
			$this->em->flush();

			//Vlastnosti
			$currentFeatures = [];
			$formFeatures    = [];
			foreach ($product->getFeatureProducts() as $fp) {
				if (isset($currentFeatures[(int) $fp->getFeatureValue()->getId()])) {
					$this->product->getFeatureProducts()->removeElement($fp);
					$this->em->remove($fp);
					continue;
				}
				$currentFeatures[(int) $fp->getFeatureValue()->getId()] = $fp;
			}

			foreach ($form->getHttpData()['features'] as $v) {
				$formFeatures[$v] = $v;
			}

			$newFeatures = array_diff_key($formFeatures, $currentFeatures);
			$oldFeatures = array_diff_key($currentFeatures, $formFeatures);

			foreach ($newFeatures as $value) {
				$fp = new FeatureProduct(
					$product,
					$this->em->getReference(FeatureValue::class, $value),
				);

				unset($formFeatures[$value]);
				$this->em->persist($fp);
			}

			foreach ($oldFeatures as $value => $fp) {
				$this->product->getFeatureProducts()->removeElement($fp);
				$this->em->remove($fp);
			}

			/** @var DynamicFeatureProduct[][] $currentDynamicFeatures */
			$currentDynamicFeatures = [];
			$formDynamicFeatures    = $form->getHttpData()['dynamicFeatures'] ?? [];
			foreach ($product->dynamicFeatures as $df) {
				$currentDynamicFeatures[$df->getFeature()->getId()][(string) $df->value] = $df;
			}

			foreach ($formDynamicFeatures as $k => $v) {
				foreach ($v as $v2) {
					$dfp        = $currentDynamicFeatures[$k][(string) $v2] ?: new DynamicFeatureProduct($product, $this->em->getReference(Feature::class, $k), (string) $v);
					$dfp->value = (string) $v2;
					$this->em->persist($dfp);
				}
			}

			$toRemove = [];
			foreach ($currentDynamicFeatures as $featureId => $vals) {
				foreach ($vals as $k => $v) {
					if (!in_array($k, $formDynamicFeatures[$featureId] ?? [])) {
						$toRemove[] = $v->getId();
					}
				}
			}

			if (!empty($toRemove)) {
				$this->em->createQueryBuilder()->delete(DynamicFeatureProduct::class, 'dfp')
					->where('dfp.id IN (' . implode(',', $toRemove) . ')')
					->getQuery()->execute();
			}

			//PriceLevels
			if ($this->helper->config->get('enablePriceLevels', false) && (!$va || $va->usePriceLevels || $va->isDefault)) {
				foreach ($this->helper->getGroupCustomers() as $gc) {
					$remove = true;
					if (isset($values['pricelevel' . $gc->getId()])) {
						$v = $values['pricelevel' . $gc->getId()];
						if ($v != '' && $v !== null) {
							$remove     = false;
							$ppl        = $product->getPriceLevels()->get($gc->getId()) ?: new ProductPriceLevel($product, $gc);
							$ppl->price = $v;
							$this->em->persist($ppl);
							if (!$product->getPriceLevels()->containsKey($gc->getId()))
								$product->getPriceLevels()->add($ppl);
						}
					}

					if ($remove) {
						$ppl = $product->getPriceLevels()->get($gc->getId());
						if ($ppl) {
							$product->getPriceLevels()->remove($gc->getId());
							$this->em->getConnection()->delete('eshop_catalog__product_price_level', [
								'product_id' => $product->getId(),
								'group_id'   => $gc->getId(),
							]);
						}
					}
				}
			}

			// Prices
			if (Config::load('enableCountryPrices')) {
				if ($va && !$va->isDefault) {
					foreach ($values->prices as &$row) {
						if (!$va->usePrice) {
							unset($row['price']);
						}

						if (Config::load('product.allowRetailPrice') && !$va->useRetailPrice) {
							unset($row['retailPrice']);
						}
					}
				}

				$this->productPricesContainer->saveData($product, (array) $values->prices);

				if (!$va || $va->isDefault || $va->usePriceLevels)
					$this->productPriceLevelCountriesContainer->saveData($product, (array) $values->priceLevelCountries);
			}

			// Kategorie
			if (!$va || $va->isDefault)
				$this->categoryContainerService->saveDataMultipleSite([$product->getId()], (array) $values->categories);

			// Dodavatele
			if ($values->suppliers) {
				$suppliers = $product->getSuppliers();

				foreach ($values->suppliers as $k => $v) {
					$suppId = substr($k, 1);

					if (isset($suppliers[$suppId])) {
						$suppliers[$suppId]->isActive = (int) $v;
						$this->em->persist($suppliers[$suppId]);
					}
				}
			}

			//Tags
			$tagsForRemove = [];
			foreach ($product->getProductTags()->toArray() as $tag)
				$tagsForRemove[$tag->getTag()->type] = $tag;

			foreach ($values->tags as $tagType => $vals) {
				if (!$vals->active)
					continue;

				if (!isset($tagsForRemove[$tagType])) {
					$productTag = new ProductTag($product, $this->em->getReference(Tag::class, $vals['id']));
					$product->addProductTag($productTag);
				} else
					$productTag = $tagsForRemove[$tagType];

				unset($tagsForRemove[$tagType]);
				$productTag->validFrom = $vals['dateFrom'];
				$productTag->validTo   = $vals['dateTo'];

				$this->em->persist($productTag);
			}

			foreach ($tagsForRemove as $tagType => $tag) {
				$product->removeProductTag($tag);
				$this->em->remove($tag);
			}

			if (isset($values->stopImportPrice)) {
				$values->stopImportPrice = $values->stopImportPrice == 1 ? 0 : 1;
				$product->setMoreDataValue('stopImportPrice', $values->stopImportPrice);
			}

			if (isset($values->stopImportQuantity))
				$product->setMoreDataValue('stopImportQuantity', $values->stopImportQuantity);

			if (isset($values->stopImportQuantity))
				$product->setMoreDataValue('stopImportQuantity', $values->stopImportQuantity);

			// dokumenty
			$this->documentContainer->saveData($values['documents'], $product);

			// videa
			$this->videoContainer->saveData($values['videos'], $product);

			// souvisejici produkty
			if (Config::load('allowRelatedProducts')) {
				$this->relatedProductContainer->saveData($form->getHttpData(), $product);
			}

			// Doprava
			if (Config::load('product.allowModifySpedition')) {
				$this->speditionContainer->saveData($values->speditions, $product);
			}

			// Pridani dodavatelů
			$suppliersContainer = $form->getHttpData()['suppliersContainer'];
			$suppliersToAdd     = isset($suppliersContainer) && is_array($suppliersContainer) && count($suppliersContainer) > 0 ? $suppliersContainer : [];
			foreach ($suppliersToAdd as $k => $row) {
				if (isset($row['selected']) || $row['selected'] === 'on') {
					$supplier     = $this->supplierServices->get((int) explode('_', $k)[1]);
					$availability = $this->availabilityService->getEr()->find((int) $row['availability']);
					if (!$supplier) {
						continue;
					}
					$this->productSupplierContainer->saveData($row, $supplier, $product, $availability);
				}
			}

			$this->em->persist($product);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $product;
			if ($this->getPresenter(false))
				$event->presenter = $this->getPresenter();
			$event->control = $this;
			$this->eventDispatcher->dispatch($event, ProductForm::class . '::formSuccess');

			$this->availabilityService->updateAvailabilityByQuantity($product);
			$this->em->flush();
			$this->em->commit();

			$this->baseProductsService->prepareUpdateVariant($product);

			// Ulozeni dat pro hlavni variant ua nastaveni aktegorii variantam
			if ($product->isVariant()) {
				if ($product->getIsVariantParent()) {
					$parent                            = $product;
					$product->isVariant()->variantName = $values->variantName;
					$this->em->persist($product->isVariant());
				} else {
					$parent = $product->getVariantParent();
				}

				if ($parent->getId()) {
					$this->productServices->updateVariantCategories($parent, (int) $parent->isVariant()->getVariantId());
				}

				$this->em->flush();
			}

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $product;
			if ($this->getPresenter(false))
				$event->presenter = $this->getPresenter();
			$event->control = $this;
			$this->eventDispatcher->dispatch($event, ProductForm::class . '::formSuccessAfterSave');

			$form->addCustomData('productId', $product->getId());
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.productForm.productSaved');
			$this->getPresenter()->redrawControl('flashes');

			if (Config::load('useOldUrls')) {
				$oldIdsCache = new Cache($this->storage, 'oldZuriel');
				foreach ($this->translator->getAvailableLocales() as $l) {
					$oldIdsCache->remove('prodIds_' . $l);
				}
			}

			if (Config::load('allowRelatedProducts'))
				$this->cacheService->productCache->remove('related/' . $product->getId());
			$this->cacheService->defaultCache->clean([Cache::TAGS => ['productsByTag']]);
			foreach ($this->helper->sitesService->getOptionsForSelect() as $k => $v) {
				$this->cacheService->defaultCache->remove($k . '_all');
			}

			foreach (array_keys($langValues) as $l) {
				$this->cacheService->defaultCache->remove('productsExtraFields_' . $l);
				$this->cacheService->productCache->remove('product/' . $product->getId() . '/' . $l);
				$this->cacheService->productCache->remove('link/' . $product->getId() . '/' . $l);
			}

			if ($product->getGallery()) {
				$albumCache = new Cache($this->storage, Albums::CACHE_NAMESPACE);
				$albumCache->clean([Cache::TAGS => ['album/' . $product->getGallery()->getId()]]);
			}
			$cache = new Cache($this->storage, Tags::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [Tags::CACHE_NAMESPACE]]);
		} catch (\Exception $e) {
			bdump($e->getMessage());
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			$form->addError($e->getMessage());
			$this->getPresenter()->redrawControl('flashes');
			$this->redrawControl('formErrors');

			return false;
		}

		return true;
	}

	protected function createComponentImagesZone()
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->product && $this->product->isVariant()) {
			$control->onBeforeUpdate[] = function($id, $v) {
				$variant = $this->product->isVariant();

				$variant->defaultImage = (int) $id;
				$this->em->persist($variant);
			};

			$control->onPrepareDefaultImage[] = function(Image $image, &$arr) {
				$arr['variantImage'] = $image->getId() === $this->product->isVariant()->defaultImage ? 1 : 0;

				return $arr;
			};

			$control->addCustomFieldLatteBefore(__DIR__ . '/ProductFormImageZoneCustomFields.latte');
		}

		if ($this->product && $this->product->getGallery())
			$control->setAlbum($this->product->getGallery()->getId());

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */
			$dataRaw = $this->httpRequest->getPost('formData');
			$data    = [];

			if (!$dataRaw)
				return false;

			foreach ($dataRaw as $row) {
				$data[$row['name']] = $row['value'];
			}

			if (!$this->product || !$this->product->getGallery()) {
				$album = new Album(UPLOADS_PATH . '/products');
				$this->em->persist($album)->flush();
				$control->setAlbum($album->getId());
			}

			if ($this->product && isset($album)) {
				$this->product->setGallery($album);
				$this->em->persist($this->product)->flush();
			}
		};

		return $control;
	}

	/*******************************************************************************************************************
	 * ========================  GET / SET
	 */

	public function setProduct($id)
	{
		$this->product = $this->productServices->get($id);

		if ($this->product) {
			/** @var Product $prod */
			$prod       = $this->product;
			$parentProd = null;
			$variant    = null;

			if ($prod->isVariant()) {
				$variants = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
					->where('pv.variantId = :variantId')->setParameter('variantId', $prod->isVariant()->getVariantId())
					->getQuery()->getResult();

				$prod->variants = new ArrayCollection($variants);
				$this->product  = $prod;

				// kontrola hlavniho produktu
				if (!$prod->getVariantParent()) {
					$prod->isVariant()->isDefault = 1;
					$this->em->persist($prod->isVariant());
					$this->em->flush($prod->isVariant());
					$this->getPresenter()->redirect('this');
				}
			}

			if ($prod->isVariant() && !$prod->getIsVariantParent()) {
				$parentProd = $prod->getVariantParent();
				$variant    = $prod->isVariant();
			}

			$form = $this['form'];
			$d    = [
				'isPublished'      => $prod->isPublished,
				'ean'              => $prod->ean,
				'quantity'         => $prod->quantity,
				'minimumAmount'    => $prod->minimumAmount,
				'recyclingFee'     => $prod->recyclingFee,
				'price'            => $variant && !$variant->usePrice ? $parentProd->price : $prod->price,
				'retailPrice'      => Config::load('product.allowRetailPrice') && $variant && !$variant->useRetailPrice ? $parentProd->retailPrice : $prod->retailPrice,
				'purchasePrice'    => $variant && !$variant->usePurchasePrice ? $parentProd->purchasePrice : $prod->purchasePrice,
				'code1'            => $prod->code1,
				'code2'            => $prod->code2,
				'discountDisabled' => ($variant && !$variant->useDiscountDisabled ? $parentProd->discountDisabled : $prod->discountDisabled) == 1 ? 0 : 1,
				'condition'        => $prod->condition,
				'width'            => $prod->width,
				'height'           => $prod->height,
				'depth'            => $prod->depth,
				'weight'           => $prod->weight,
				'position'         => $prod->position,
			];

			if ($form->getComponent('maximumAmount', false)) {
				$d['maximumAmount'] = $prod->maximumAmount;
			}

			if ($form->getComponent('units', false)) {
				$d['units'] = $prod->units;
			}

			if ($prod->isVariant() && $prod->getIsVariantParent()) {
				$d['variantName'] = $prod->isVariant()->variantName;
			}

			if (isset($this['form']['disablePickUpSpedition']))
				$d['disablePickUpSpedition'] = $prod->disablePickUpSpedition;
			if (isset($this['form']['isOversize']))
				$d['isOversize'] = $prod->isOversize;
			if (isset($this['form']['disableStackInCart']))
				$d['disableStackInCart'] = $prod->disableStackInCart;
			if (isset($this['form']['isDiscount']))
				$d['isDiscount'] = $prod->isDiscount;

			if (Config::load('product.allowCountryOfOrigin')) {
				$d['countryOfOrigin'] = $prod->countryOfOrigin ? $prod->countryOfOrigin->getId() : null;
			}

			if (Config::load('product.allowHSCustomsCode')) {
				$d['hsCustomsCode'] = $prod->hsCustomsCode;
			}

			if ($prod->getAvailability() && array_key_exists($prod->getAvailability()->getId(), $form['availability']->getItems()))
				$form['availability']->setDefaultValue($prod->getAvailability()->getId());

			if ($prod->availabilityAfterSoldOut && array_key_exists($prod->availabilityAfterSoldOut->getId(), $form['availabilityAfterSoldOut']->getItems()))
				$form['availabilityAfterSoldOut']->setDefaultValue($prod->availabilityAfterSoldOut->getId());

			if (Config::load('product.allowAssort'))
				$d['isAssort'] = $prod->isAssort ? 1 : 0;

			if (Config::load('pseudoWarehouse'))
				$form['unlimitedQuantity']->setDefaultValue($variant && !$variant->useUnlimitedQuantity ? $parentProd->unlimitedQuantity : $prod->unlimitedQuantity);
			if (class_exists('EshopGifts\DI\EshopGiftsExtension'))
				$form['categoryGiftsAllowed']->setDefaultValue($variant && !$variant->useCategoryGiftsAllowed ? $parentProd->isCategoryGiftsAllowed() : $prod->isCategoryGiftsAllowed());

			if (class_exists(ImportExtension::class)) {
				$stopImportPrice = ($variant && !$variant->useImportPrice ? $parentProd : $prod)->getMoreDataValue('stopImportPrice') == 1 ? 0 : 1;
				$form['stopImportPrice']->setValue($stopImportPrice);
			}

			if (Config::load('product.allowStopImportQuantity')) {
				$stopImportQuantity = ($variant ? $parentProd : $prod)->getMoreDataValue('stopImportQuantity');
				$form['stopImportQuantity']->setValue($stopImportQuantity);
			}

			if (Config::load('product.loadQuantityOnStock') && class_exists(EshopStockExtension::class)) {
				$qb = $this->em->getRepository(SupplyProduct::class)->createQueryBuilder('sp')
					->select('COUNT(sp)')
					->join('sp.product', 'p')
					->where('p.id = :id AND sp.order IS NULL AND sp.writtenOffDate IS NULL')
					->setParameter('id', (int) $id);
				$form['quantityOnStock']->setValue($qb->getQuery()->getSingleResult());
			}

			// Manufacturer
			if ($prod->getManufacturer() && array_key_exists($prod->getManufacturer()->getId(), $form['manufacturer']->getItems()))
				$form['manufacturer']->setDefaultValue($prod->getManufacturer()->getId());

			//Suppliers
			if ($prod->getSuppliers()->count()) {
				$formSupp = $form->addContainer('suppliers');

				foreach ($prod->getSuppliers()->toArray() as $sup) {
					$formSupp->addBool('s' . $sup->getSupplier()->getId(), '')
						->setDefaultValue($sup->isActive);
				}
			}

			$this->onAnchor[] = function() use ($prod) {
				$this->template->suppliers = [];
				foreach ($prod->getSuppliers()->toArray() as $productSupplier) {
					/** @var ProductSupplier $productSupplier */
					$this->template->suppliers[$productSupplier->getSupplier()->getId()] = [
						'name'       => $productSupplier->getSupplier()->name,
						'code'       => $productSupplier->code,
						'quantity'   => $productSupplier->quantity,
						'supplierId' => $productSupplier->getSupplier()->getId(),
					];
				}
			};

			// Extra Fields
			$ef = $prod->getExtraFieldsValues();
			if (Config::load('product.allowNote')) {
				$d['addNote']   = $ef['addNote'] ?? false;
				$d['noteTitle'] = $ef['noteTitle'] ?? null;
			}

			$form->setDefaults($d);

			// Texty
			$mTexts  = [];
			$seoData = [];

			$prodTexts   = $prod->getProductTexts()->toArray();
			$parentTexts = [];
			if ($variant) {
				$parentTexts = $this->em->getRepository(ProductTexts::class)->createQueryBuilder('pt', 'pt.lang')
					->where('pt.id = :id')->setParameter('id', $parentProd->getId())
					->getQuery()->getResult();
			}

			foreach ($prodTexts as $lang => $texts) {
				/** @var ProductTexts $texts */
				foreach (['name' => 'name', 'shortDesc' => 'shortDescription', 'desc' => 'description'] as $k => $v) {
					$mTexts[$k][$lang] = $variant && !$variant->{'use' . ucfirst($v)} ? $parentTexts[$lang]->$v : $texts->$v;
				}

				if (Config::load('product.allowModifyAlias')) {
					$mTexts['alias'][$lang] = $texts->alias ? Strings::webalize((string) $texts->alias) : Strings::webalize((string) $texts->name);
				}

				if (Config::load('enableProductName2'))
					$mTexts['name2'][$lang] = $variant && !$variant->useName2 ? $parentTexts[$lang]->name2 : $texts->name2;
				if (Config::load('product.allowPreorder'))
					$mTexts['preorderText'][$lang] = $texts->preorderText;

				foreach ($texts->getSeo() as $k => $v)
					$seoData[$k][$lang] = $v;
			}
			$form->setDefaults($mTexts);
			$this->seoContainerService->setDefaults($form['seo'], $seoData);

			if ($prod->getVateRate() && array_key_exists($prod->getVateRate()->getId(), $form['vatRate']->getItems()))
				$form['vatRate']->setDefaultValue($prod->getVateRate()->getId());

			// Vlastnosti
			//			$fd = [
			//				'features'     => [],
			//				'featuresList' => [],
			//			];
			//
			//			foreach ($prod->getFeatureProducts()->toArray() as $fp) {
			//				/** @var FeatureProduct $fp */
			//				$featureId                        = $fp->getFeature()->getId();
			//				$featureValueId                   = $fp->getFeatureValue()->getId();
			//				$fd['features']['f' . $featureId] = $featureValueId;
			//				$fd['featuresList'][]             = $featureId;
			//			}
			//			$form->setDefaults($fd);

			//set hodnoty variant

			// set Prices
			if (Config::load('enableCountryPrices')) {
				$this->productPricesContainer->setDefaults($form['prices'], $prod);
				$this->productPriceLevelCountriesContainer->setDefaults($form['priceLevelCountries'], $prod);
			}

			//set kategorie
			$allCategoriesId = array_map(function($c) { return $c->getCategory()->getId(); }, $prod->getCategoryProducts()->toArray());
			$this->categoryContainerService->setDefaultsMultipleSite($form['categories'], $prod->sites->toArray(), $allCategoriesId);

			//set tags
			$tags = [];
			foreach ($prod->getProductTags()->toArray() as $t) {
				$tags[$t->getTag()->type] = [
					'active'   => 1,
					'dateFrom' => $t->validFrom,
					'dateTo'   => $t->validTo,
					'id'       => $t->getTag()->getId(),
				];
			}
			$this->tagsContainerService->setDefaults($form['tags'], $tags);

			// priceLevel
			$prodPriceLevels = $prod->getPriceLevels()->toArray();
			if ($variant && !$variant->usePriceLevels) {
				$prodPriceLevels = [];
				foreach ($this->em->getRepository(ProductPriceLevel::class)->createQueryBuilder('pl',)
					         ->where('pl.productId = :id')->setParameter('id', $parentProd->getId())
					         ->getQuery()->getResult() as $row)
					$prodPriceLevels[$row->getGroupId()->getId()] = $row;
			}
			if ($this->helper->config->get('enablePriceLevels', false)) {
				foreach ($this->helper->getGroupCustomers() as $gc) {
					if (isset($form['pricelevel' . $gc->getId()]) && isset($prodPriceLevels[$gc->getId()])) {
						$form['pricelevel' . $gc->getId()]->setDefaultValue($prodPriceLevels[$gc->getId()]->price);
					}
				}
			}

			$this->monitor(IPresenter::class, function() use ($prod, $form) {
				$documentsData = [];
				foreach ($this->productDocuments->getByProduct($prod->getId()) as $document) {
					$documentsData[] = [
						'id'         => $document->getId(),
						'fileName'   => $document->name,
						'filePath'   => $document->file,
						'lang'       => $document->lang,
						'removeLink' => $this->link('removeDocument', $document->getId()),
					];
				}
				$this->documentContainer->setDefaults($form['documents'], $documentsData);

				$videos = [];
				foreach ($this->productVideos->getByProduct($prod->getId()) as $video) {
					$arr = [
						'id'         => $video->getId(),
						'title'      => $video->title,
						'lang'       => $video->lang,
						'thumbnail'  => $video->thumbnail,
						'removeLink' => $this->link('removeVideo', $video->getId()),
					];

					if ($video->url && Strings::startsWith($video->url, 'http')) {
						$arr['url'] = $video->url;
					} else if ($video->url) {
						$arr['file'] = $video->url;
					}

					$videos[] = $arr;
				}
				$this->videoContainer->setDefaults($form['videos'], $videos);

				if (Config::load('allowRelatedProducts')) {
					$this->relatedProductContainer->setDefaults($form['relatedProducts'], $prod->getRelatedProducts()->toArray());
				}

				if (Config::load('product.allowModifySpedition', false))
					$this->speditionContainer->setDefaults($form['speditions'], $prod->getSpeditions()->toArray());
			});

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->product), ProductForm::class . '::setProduct');
		} else
			$this->getPresenter()->error();
	}

	protected function getFeatures(): array
	{
		if ($this->cFeatures === null) {

			$this->cFeatures = [];
			foreach ($this->featureServices->getEr()->createQueryBuilder('f')
				         ->select('f.id, ft.name, f.type, f.unit, f.decimals')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->where('f.isPublished = 1')
				         ->orderBy('f.position')
				         ->getQuery()->getResult() as $v) {
				$v['name'] = trim((string) $v['name']);

				if ($v['type'] === Feature::TYPE_RANGE) {
					$v['step'] = Feature::getStep((int) $v['decimals']);
				}

				$this->cFeatures[$v['id']] = $v;
			}
		}

		return $this->cFeatures;
	}

	protected function getFeatureValues(?int $id = null): array
	{
		if ($this->cFeatureValues === null) {
			$this->cFeatureValues = [];

			foreach ($this->featureServices->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
				         ->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->andWhere('fv.isPublished = 1')
				         ->orderBy('fv.position')
				         ->getQuery()->getResult() as $v)
				$this->cFeatureValues[$v['featureId']][] = [
					'id'   => $v['id'],
					'name' => trim($v['name']),
				];
		}

		return $id ? $this->cFeatureValues[$id] ?? [] : $this->cFeatureValues;
	}

	protected static function allowSaleDiscountCouponsAsProducts(): bool
	{
		return Parameters::load('eshopSales.autoGenerateDiscountCoupon.enable', false) && class_exists('EshopSales\DI\EshopSalesExtension');
	}
}
