<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Import;

use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\AdminModule\Model\Suppliers;
use Import\AdminModule\Model\ImportType\ImportType;
use Nette\Localization\ITranslator;
use Nette\Http\Request;

abstract class ATypeProduct extends ImportType
{
	/** @var ITranslator */
	protected ITranslator $translator;

	/** @var Request */
	protected Request $httpRequest;

	/** @var ImportHelper */
	protected ImportHelper $importHelper;

	protected Suppliers $suppliers;

	protected AvailabilityService $availabilityService;

	protected array $formContainerFields = [];

	public function __construct(ITranslator $translator, Request $request, ImportHelper $importHelper, Suppliers $suppliers,
	                            AvailabilityService $availabilityService)
	{
		$this->translator          = $translator;
		$this->httpRequest         = $request;
		$this->importHelper        = $importHelper;
		$this->suppliers           = $suppliers;
		$this->availabilityService = $availabilityService;
	}

	/**
	 * @param $request
	 */
	public function setHttpRequest($request) { $this->httpRequest = $request; }

	public function getFormSyncOpts(array $importKeys, array $importValues, $defaults)
	{
		$container = new BaseContainer();

		$arr = [];
		foreach (['ean', 'code1', 'code2', 'supplierCode'] as $v)
			$arr[$v] = 'eshopCatalog.import.' . $v;
		$default = null;
		if (isset($defaults['syncBy']) && array_key_exists($defaults['syncBy'], $arr))
			$default = $defaults['syncBy'];
		$container->addSelect('syncBy', 'import.importForm.syncBy', $arr)->setDefaultValue($default)->setRequired();

		$container->addText('xPath', 'import.import.xPath')->setDefaultValue($defaults['xPath'] ?? null);
		$container->addSelect('onlyUpdate', 'import.import.onlyUpdate', [
			0 => 'default.no',
			1 => 'default.yes',
		])->setDefaultValue($defaults['onlyUpdate'] ?? 0);

		$suppliers = [];
		foreach ($this->suppliers->getOptionsForSelect() as $v) {
			$suppliers[$v] = $v;
		}
		$container->addSelect('supplier', 'eshopCatalog.import.supplier', $suppliers)->setRequired();
		if ($defaults['supplier'] && array_key_exists($defaults['supplier'], $container['supplier']->getItems()))
			$container['supplier']->setDefaultValue($defaults['supplier']);

		$default = null;
		$arr     = $this->importHelper->categoriesList();
		if (isset($defaults['baseCategory']) && array_key_exists($defaults['baseCategory'], $arr))
			$default = $defaults['baseCategory'];

		$container->addSelect('baseCategory', 'import.import.baseCategory', $this->importHelper->categoriesList())
			->setDefaultValue($default);
		$container->addText('username', 'eshopCatalog.import.username')
			->setDefaultValue($defaults['username']);
		$container->addText('password', 'eshopCatalog.import.password')
			->setDefaultValue($defaults['password']);

		foreach (['Ean', 'Code1', 'Code2'] as $col) {
			$container->addText('only' . $col . 'Start', 'eshopCatalog.import.only' . $col . 'Start')
				->setDefaultValue($defaults['only' . $col . 'Start']);
		}

		return $container;
	}

	public function getFormContainer(array $importKeys, array $importValues, $defaults)
	{
		$importKeys   = [null => null] + $importKeys;
		$importValues = [null => null] + $importValues;
		$container    = new BaseContainer();
		$postData     = $this->httpRequest->getPost('data');
		/** @var BaseContainer[] $containers */
		$containers = [];

		$fields = $this->formContainerFields;

		foreach ($fields as $k => $opts) {
			$containers[] = $c = $container->addContainer($k, 'eshopCatalog.import.' . $k);

			if ($opts['type'] == 'sel') {
				// Select
				$c->addSelect('key', 'import.importForm.select', $importKeys);

				if (isset($opts['multilang']) && $opts['multilang'] === true)
					$c['key']->setIsMultilanguage();
				$c['key']->setValue($defaults[$k]['key'] ?? null);

				if (!isset($opts['fixed']) || $opts['fixed'])
					$c->addText('fixedValue', 'import.importForm.orFixedValue')
						->setDefaultValue($defaults[$c->getName()]['fixedValue'] ?? null);

				if (isset($opts['canModifyInt']) && $opts['canModifyInt'] === true) {
					$c->addText('modifyInt', 'import.importForm.modifyInt')
						->setDefaultValue($defaults[$c->getName()]['modifyInt'] ?? null);
				}

				if (in_array($k, ['priceVat', 'retailPrice'])) {
					$c->addCheckbox('withoutVat', 'eshopCatalog.import.priceWithoutVat')
						->setDefaultValue($defaults[$c->getName()]['withoutVat'] ?? 0);
				}
			} else if ($opts['type'] == 'bol') {
				// Bolean
				$c->addSelect('key', 'import.importForm.select', $importKeys)
					->setAttribute('onchange', "importFormUpdateSelect('$k')")
					->setDefaultValue($defaults[$k]['key'] ?? null);

				$tmp   = [];
				$items = [];
				if (isset($importValues[$postData[$k]['key']]))
					$tmp = $importValues[$postData[$k]['key']] ?? [];
				else if (isset($defaults[$k]['key']))
					$tmp = $importValues[$defaults[$k]['key']] ?? [];

				foreach ($tmp as $v) {
					$items[$v] = $v;
				};

				$secondType = $opts['secondType'] ?? 'addSelect';

				if ($secondType === 'addSelect' && $items)
					$items = [null => null] + $items;

				$c->{$secondType}('value', 'import.importForm.valueIs', $items)
					->setDefaultValue($defaults[$k]['value']);

				if (!isset($opts['fixed']) || $opts['fixed']) {
					$arr = [
						''   => $this->translator->translate('import.importForm.noUse'),
						'v1' => $this->translator->translate('default.yes'),
						'v0' => $this->translator->translate('default.no'),
					];

					$default = '';
					if (isset($defaults[$c->getName()]['fixedValue']) && array_key_exists('v' . $defaults[$c->getName()]['fixedValue'], $arr))
						$default = 'v' . $defaults[$c->getName()]['fixedValue'];

					$c->addBool('fixedValue', $this->translator->translate('import.importForm.orFixedValue'), $arr)->setDefaultValue($default);
				}
			} else if ($opts['type'] == 'slist') {
				// Sortable list
				$arr = $importKeys;
				array_shift($arr);
				$default = [];
				foreach ($defaults[$c->getName()]['key'] as $k)
					if (array_key_exists($k, $arr))
						$default[] = $k;
				$c->addSortableCheckboxList('key', 'import.importForm.select', $arr)->setDefaultValue($default);
			} else if ($opts['type'] == 'comparison') {
				$c->addSelect('key', 'import.importForm.select', $importKeys)
					->setHtmlAttribute('onchange', "importFormUpdateSelect('$k')")
					->setDefaultValue($defaults[$k]['key'] ?? null);

				$items = ['' => ''];
				if (isset($importValues[$postData[$k]['key']]))
					$items += $importValues[$postData[$k]['key']] ?? [];
				else if (isset($defaults[$k]['key']))
					$items += $importValues[$defaults[$k]['key']] ?? [];

				foreach ($items as $ik => $iv) {
					if ($iv === 'true') {
						unset($items[$ik]);
						$items['true'] = $iv;
					} else if ($iv === 'false') {
						unset($items[$ik]);
						$items['false'] = $iv;
					}
				}

				$comparison = [
					'<'  => '<',
					'<=' => '<=',
					'='  => '=',
					'>=' => '>=',
					'>'  => '>',
				];

				$c->addSelect('valueIs', 'import.importForm.valueIs', $comparison)
					->setDefaultValue($defaults[$k]['valueIs']);
				$c->addSelect('value', 'import.importForm.value', $items)
					->setValue($defaults[$k]['value']);
				$c->addText('custom', 'import.importForm.value')
					->setDefaultValue($defaults[$k]['custom']);

				if ($k === 'idAvailability') {
					$availabilities = ['' => ''] + $this->availabilityService->getOptionsForSelect();
					$c->addSelect('inStock', 'eshopCatalog.import.availabilities.inStock', $availabilities)
						->setValue($defaults[$k]['inStock']);
					$c->addSelect('soldOut', 'eshopCatalog.import.availabilities.soldOut', $availabilities)
						->setValue($defaults[$k]['soldOut']);
				}
			}

			if ($k == 'category') {
				$c->addText('separator', 'import.importForm.categorySeparator')
					->setDefaultValue($defaults[$c->getName()]['separator'] ?? null);
				$c->addText('deepSeparator', 'import.importForm.categoryDeepSeparator')
					->setDefaultValue($defaults[$c->getName()]['deepSeparator'] ?? null);

				$default = null;
				$arr     = $this->importHelper->categoriesList();
				if (isset($defaults[$c->getName()]['fixedValue']) && array_key_exists($defaults[$c->getName()]['fixedValue'], $arr))
					$default = $defaults[$c->getName()]['fixedValue'];
				$c->addSelect('fixedValue', 'import.importForm.orFixedValue', ['' => ''] + $arr)
					->setDefaultValue($default);
			}
		}

		foreach ($containers as $c) {
			$arr     = [
				'all' => [
					'title'    => $this->translator->translate('import.importForm.applyOnAll'),
					'btnClass' => 'success',
				],
				'new' => [
					'title'    => $this->translator->translate('import.importForm.applyOnNew'),
					'btnClass' => 'danger',
				],
			];
			$default = 'new';
			if (isset($defaults[$c->getName()]['applyOn']) && array_key_exists($defaults[$c->getName()]['applyOn'], $arr))
				$default = $defaults[$c->getName()]['applyOn'];
			$c->addBool('applyOn', 'import.importForm.applyOn', $arr)->setDefaultValue($default);
		}

		return $container;
	}

	public function processSubmitSyncOpts($values)
	{
		return $values;
	}

	public function processSubmitData($values)
	{
		$data = [];
		foreach ($values as $k => $v) {
			if ($v->key === '' && $v->fixedValue === '')
				continue;

			if ($v->key !== '') {
				unset($v->fixedValue);
			} else if ($v->fixedValue !== '') {
				unset($v->key);
				unset($v->value);

				if (in_array($v->fixedValue, ['v0', 'v1']))
					$v->fixedValue = ltrim($v->fixedValue, 'v');
			}

			$data[$k] = $v;
			if (is_array($v->key)) {
				foreach ($v->key as $k2 => $v2) {
					$data[$k]['key'][$k2] = $v2;
				}
			}
		}

		return $data;
	}
}
