<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Subscribers;

use Core\Model\Event\FormSuccessEvent;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\PseudoWarehouse;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\AdminModule\Components\Order\OrderStatusForm;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderStatusSubscriber implements EventSubscriberInterface
{
	protected PseudoWarehouse $pseudoWarehouse;

	public function __construct(PseudoWarehouse $pseudoWarehouse)
	{
		$this->pseudoWarehouse = $pseudoWarehouse;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderStatusForm::class . '::formSuccess'    => 'formSuccess',
			OrderStatusesGrid::class . '::beforeDelete' => 'beforeDelete',
			OrdersGrid::class . '::changeStatus'        => 'changeStatus',
		];
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderStatus $entity */
		$entity = $event->custom['entity'];
		$order  = $entity->getOrder();
		/** @var string $value */
		$value = $event->values->status;

		$this->onOrderSetCanceled($order, $value);
	}

	public function beforeDelete(OrderStatusEvent $event): void
	{
		if ($event->orderStatus->getStatus()->getId() !== OrderStatus::STATUS_CANCELED)
			return;

		// pocet statusů canceled
		$canceledStatusCount = 0;
		foreach ($event->orderStatus->getOrder()->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$orderStatus->isDeleted()) {
				$canceledStatusCount++;
			}
		}

		// produkty do objednavky se vraceji az pri poslednim ruseni storna objednavky
		if ($canceledStatusCount === 1) {
			$data = [];
			foreach ($event->orderStatus->getOrder()->getOrderItems()->toArray() as $item) {
				/** @var OrderItem $item */
				$data[$item->getProductId()] = $item->getQuantity();
			}

			$this->pseudoWarehouse->lowerQuantities($data, $event->orderStatus->getOrder());
		}
	}

	public function changeStatus(\EshopOrders\Model\Event\OrderStatusEvent $event): void
	{
		$this->onOrderSetCanceled($event->order, $event->status);
	}

	protected function onOrderSetCanceled(Order $order, string $value): void
	{
		if (!Config::load('pseudoWarehouse')) {
			return;
		}

		$isCanceled = false;
		/** @var OrderStatus $orderStatus */
		foreach ($order->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$orderStatus->isDeleted()) {
				$isCanceled = true;
			}
		}

		if ($value === OrderStatus::STATUS_CANCELED && !$isCanceled) {
			$data = [];

			foreach ($order->getOrderItems()->toArray() as $item) {
				/** @var OrderItem $item */
				$data[$item->getProductId()] = $item->getQuantity();
			}

			$this->pseudoWarehouse->raiseQuantities($data, $order);
		}
	}

}
