<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImagePipe;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\AvailabilityService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Event\ProductsFeedEvent;
use Nette\DI\Container;
use Nette\Http\Request;
use Nette\Utils\Image;
use Nette\Utils\Json;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class AdsProductsFeedSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected Categories             $categories;
	protected Sites                  $sites;
	protected Request                $httpRequest;
	protected AvailabilityService    $availabilityService;
	protected Currencies             $currencies;
	protected Exchange               $exchange;
	protected ImagePipe              $imagePipe;
	protected Container              $container;

	public function __construct(
		EntityManagerDecorator $em,
		Categories             $categories,
		Sites                  $sites,
		Request                $httpRequest,
		AvailabilityService    $availabilityService,
		Currencies             $currencies,
		Exchange               $exchange,
		ImagePipe              $imagePipe,
		Container              $container
	)
	{
		$this->em                  = $em;
		$this->categories          = $categories;
		$this->sites               = $sites;
		$this->httpRequest         = $httpRequest;
		$this->availabilityService = $availabilityService;
		$this->currencies          = $currencies;
		$this->exchange            = $exchange;
		$this->imagePipe           = $imagePipe;
		$this->container           = $container;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopCatalog.productsFeed'             => 'productsFeed',
			'eshopCatalog.productsFeedAvailability' => 'productsFeedAvailability',
			'eshopCatalog.categoriesFeed'           => 'categoriesFeed',
		];
	}

	public function productsFeed(ProductsFeedEvent $event): void
	{
		if ($event->customerName === 'ads') {
			$event->outputFormat = 'csv';
			$availabilityFilter  = Config::load('productsFeed.adsAvailabilityFilter');

			$fp = fopen('php://memory', 'wb+');
			fputcsv($fp, ['Page URL', 'Custom label'], ',');
			foreach ($event->data as $row) {
				if (empty($row->categoryId)) {
					continue;
				}

				if ($availabilityFilter === 'quantity' && $row->availabilityIdent !== Availability::IN_STOCK) {
					continue;
				}

				if ($availabilityFilter === 'canAddToCart') {
					$av = $this->availabilityService->getByIdent($row->availabilityIdent);

					if (!$av || !$av->canAddToCart()) {
						continue;
					}
				}

				$label = [
					Strings::webalize((string) $row->category[0]),
					(string) $row->categoryId[0],
				];

				fputcsv($fp, [
					$row->link,
					implode('-', $label),
				], ',');
			}

			rewind($fp);
			$event->data = [
				'output' => stream_get_contents($fp),
			];

			fclose($fp);
		} else if ($event->customerName === 'ads-lp') {
			$event->outputFormat = 'csv';

			$fp = fopen('php://memory', 'wb+');

			$storeCode = $this->httpRequest->getQuery('store');

			if (!$this->container->hasService('eshopProductsComparison.front.export.data')) {
				throw new \Exception('No data');
			}

			$this->em->getConfiguration()->setSQLLogger(null);
			fputcsv($fp, ['Rank', 'Product Item Id', 'Title', 'Description', 'Item URL', 'Image URL', 'Price',
				'Store code', 'Sale Price'], ',');

			$file = WWW_DIR . '/exportproducts/google-' . $event->siteIdent . '-' . $event->lang . '.xml';

			if (!file_exists($file)) {
				throw new \Exception('No data');
			}

			$quantities = [];
			foreach ($this->em->createQueryBuilder()->select('IDENTITY(pis.product) as prodId, p.quantity')
				         ->from(ProductInSite::class, 'pis')
				         ->andWhere('pis.site = :siteIdent')
				         ->setParameters([
					         'siteIdent' => $event->siteIdent,
				         ])
				         ->innerJoin('pis.product', 'p')
				         ->getQuery()->getArrayResult() as $row) {
				$quantities[$row['prodId']] = $row['quantity'];
			}

			$xml = simplexml_load_string(file_get_contents($file));

			foreach ($xml->channel->item as $p) {
				$row = [];
				foreach ($p->children('http://base.google.com/ns/1.0') as $name => $item) {
					$row[$name] = (string) $item;
				}

				if ($row['availability'] === 'in_stock') {
					$quantity = $quantities[$row['id']] ?? 1;
				} else {
					$quantity = 0;
				}

				if ($quantity <= 0) {
					continue;
				}

				$imgTmp = explode('/uploads', (string) $row['image_link']);
				$img    = isset($imgTmp[1]) ? '/uploads' . $imgTmp[1] : null;

				if ($img) {
					try {
						$img = $this->imagePipe->request($img, '720x720', Image::FIT, false, false, 'jpg');
					} catch (\Exception $e) {
					}
				}

				fputcsv($fp, [
					'1',
					(string) $row['id'],
					Strings::truncate($row['title'], 35, ''),
					Strings::truncate($row['description'], 127, ''),
					$row['link'],
					$img ?: $row['image_link'],
					(string) $row['price'],
					(string) $storeCode,
					(string) $row['sale_price'],
				], ',');
			}

			rewind($fp);
			$event->data = [
				'output' => stream_get_contents($fp),
			];

			fclose($fp);
		}
	}

	public function productsFeedAvailability(ProductsFeedEvent $event): void
	{
		if ($event->customerName !== 'ads') {
			return;
		}

		$event->outputFormat = 'csv';

		$fp = fopen('php://memory', 'wb+');

		$storeCode          = $this->httpRequest->getQuery('store');
		$targetCurrency     = $this->httpRequest->getQuery('currency');
		$defaultCurrency    = $this->currencies->getDefaultCode();
		$availabilityFilter = Config::load('productsFeed.adsAvailabilityFilter');

		fputcsv($fp, ['store_code', 'id', 'quantity', 'price', 'sale_price', 'availability', 'pickup_method',
			'pickup_sla'], ',');
		foreach ($event->data as $row) {
			if ($availabilityFilter === 'quantity' && $row['available'] <= 0) {
				continue;
			}

			$av = $this->availabilityService->getByIdent($row['availabilityIdent']);
			if (!$av || ($availabilityFilter === 'canAddToCart' && !$av->canAddToCart())) {
				continue;
			}

			if (Config::load('product.allowRetailPrice')) {
				$price     = $row['retailPrice'] ?: $row['price'];
				$salePrice = $row['retailPrice'] ? $row['price'] : '';
			} else {
				$price     = $row['price'];
				$salePrice = $row['price'];
			}

			if ($targetCurrency && $targetCurrency !== $defaultCurrency) {
				$price = $this->exchange->change((float) $price, $targetCurrency);

				if ($salePrice) {
					$salePrice = $this->exchange->change((float) $salePrice, $targetCurrency);
				}
			}

			$outputCurrency = $targetCurrency ?: $defaultCurrency;

			fputcsv($fp, [
				(string) $storeCode,
				$row['id'],
				$row['available'],
				$price . ' ' . $outputCurrency,
				$salePrice ? $salePrice . ' ' . $outputCurrency : '',
				$av->getOpenGraphText(),
				'buy',
				'same day',
			], ',');
		}

		rewind($fp);
		$event->data = [
			'output' => stream_get_contents($fp),
		];

		fclose($fp);
	}

	public function categoriesFeed(ProductsFeedEvent $event): void
	{
		if ($event->customerName !== 'ads') {
			return;
		}

		$event->outputFormat = 'csv';

		$currentSite = $this->sites->getCurrentSite();
		$site        = $currentSite->getIdent();
		$file        = WWW_DIR . '/feed-' . $site . '-' . $currentSite->getCurrentDomain()->getLang() . '.json';
		$hasProducts = [];

		try {
			if (file_exists($file)) {
				foreach ((array) Json::decode(file_get_contents($file)) as $row) {
					foreach ($row->categoryId as $catId) {
						$hasProducts[$catId] = true;
					}
				}
			}
		} catch (\Exception $e) {
		}

		$fp = fopen('php://memory', 'wb+');
		fputcsv($fp, ['Page URL', 'Custom label'], ',');
		foreach ($event->data as $row) {
			if (!isset($hasProducts[$row['category_id']])) {
				continue;
			}

			fputcsv($fp, [
				$row['link'],
				Strings::webalize($row['title']) . '-' . $row['category_id'],
			], ',');
		}

		rewind($fp);
		$event->data = [
			'output' => stream_get_contents($fp),
		];

		fclose($fp);
	}
}
