<?php declare(strict_types = 1);

namespace EshopCatalog\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductSupplier;

class AvailabilityService extends BaseFrontEntityService
{
	protected $entityClass = Availability::class;

	protected ?array $cAvs = null;

	protected static $updated = [];

	/**
	 * Kontrola stavu produktu pokud je vyprodán nebo skladem
	 *
	 * @param Product $product
	 */
	public function updateAvailabilityByQuantity(Product &$product): void
	{
		$avs           = $this->getAvailability();
		$allowExternal = (int) $this->settings->get('eshopCatalogDisableExternalStorage', 0) === 0;

		if ($product->unlimitedQuantity) {
			if (!$product->getAvailability() || $product->getAvailability()->getIdent() !== Availability::PREORDER) {
				$product->setAvailability($this->em->getReference(Availability::class, $avs[Availability::IN_STOCK]));
				self::$updated[$product->getId()] = $product->getId();
			}
		} else if ($product->quantity <= 0) {
			$supplierPos = null;
			$supplierAv  = null;

			if ($allowExternal) {
				foreach ($product->getSuppliers()->toArray() as $supp) {
					/** @var ProductSupplier $supp */
					if (!$supp->isActive)
						continue;

					if ($supp->availabilityAfterSoldOut && $supp->quantity > 0 && ($supplierPos === null || $supplierPos > $supp->availabilityAfterSoldOut->getPosition())) {
						$supplierAv  = $supp->availabilityAfterSoldOut;
						$supplierPos = $supp->availabilityAfterSoldOut->getPosition();
					}
				}
			}

			if (!$supplierAv && $product->availabilityAfterSoldOut)
				$supplierAv = $product->availabilityAfterSoldOut;

			$product->setAvailability($supplierAv ?: $this->em->getReference(Availability::class, $avs[Availability::SOLD_OUT]));
			self::$updated[$product->getId()] = $product->getId();
		} else if ($product->quantity > 0
			&& (!$product->getAvailability() || !in_array($product->getAvailability()->getIdent(), [
					Availability::IN_STOCK,
					Availability::PREORDER,
				]))) {

			if (isset($avs[Availability::PREORDER]) && $product->getMoreDataValue('lastStatusPreorder', false)) {
				$product->setAvailability($this->em->getReference(Availability::class, $avs[Availability::PREORDER]));
				$product->removeMoreData('lastStatusPreorder');
			} else {
				$product->setAvailability($this->em->getReference(Availability::class, $avs[Availability::IN_STOCK]));
			}
			self::$updated[$product->getId()] = $product->getId();
		}
	}

	public function updateProductsWithoutSupplier()
	{
		$avs = $this->getAvailability();
		/** @var Availability $inStock */
		$inStock = $avs[Availability::IN_STOCK];
		/** @var Availability $soldOut */
		$soldOut = $avs[Availability::SOLD_OUT];
		/** @var Availability $preorder */
		$preorder = $avs[Availability::PREORDER] ?? null;

		$isIn   = [];
		$isOut  = [];
		$others = [];

		foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
			         ->select('p.id, p.quantity, IDENTITY(p.availability) as av, IDENTITY(p.availabilityAfterSoldOut) as avAfterSoldOut, p.unlimitedQuantity')
			         ->andWhere('p.isDeleted = 0')
			         ->leftJoin('p.suppliers', 'ps')
			         ->andWhere('ps.product IS NULL')
			         ->getQuery()->getResult() as $row) {
			if (isset(self::$updated[$row['id']]))
				continue;

			if ($row['unlimitedQuantity']) {
				if ($row['av'] == $soldOut)
					$others[$row['avAfterSoldOut']][] = $row['id'];
			}
			if ($row['avAfterSoldOut'] && $row['quantity'] <= 0)
				$others[$row['avAfterSoldOut']][] = $row['id'];
			else if ($row['quantity'] > 0 && $row['av'] == $soldOut)
				$isIn[] = $row['id'];
			else if ($row['quantity'] <= 0)
				$isOut[] = $row['id'];
		}

		foreach (array_chunk($isIn, 100) as $chunk) {
			$this->em->getConnection()
				->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$inStock} WHERE id IN (" . implode(',', $chunk) . ")");
		}

		foreach (array_chunk($isOut, 100) as $chunk) {
			$this->em->getConnection()
				->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$soldOut} WHERE id IN (" . implode(',', $chunk) . ")");
		}

		foreach ($others as $avId => $ids) {
			foreach (array_chunk($ids, 100) as $chunk) {
				$this->em->getConnection()
					->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$avId} WHERE id IN (" . implode(',', $chunk) . ")");
			}
		}
	}

	protected function getAvailability(): array
	{
		if ($this->cAvs === null) {
			$this->cAvs = [];

			foreach ($this->getEr()->createQueryBuilder('a')->select('a.id, a.ident')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cAvs[$row['ident']] = (int) $row['id'];
			}
		}

		return $this->cAvs;
	}
}
