import Vue from 'vue'
import Vuex from 'vuex'
import {fetchProductsFromCategory, getCategories, searchProducts} from "../api/api";
import {deepcopy} from "../utils/utils";

Vue.use(Vuex);

export default new Vuex.Store({
	state: {
		visibleProducts: [],
		checkoutProducts: [],
		searchTerm: '',
		visibleCategories: [],
		categories: [],
		editedProductArrayId: -1,
		editedSaleArrayId: -1,
		totalPrice: 0,
		activeCategory: {},
		activeProduct: {},
		breadcrumb: [],
		editingSale: false,
		showDetailProduct: false,
		showDetailSale: false,
		detailProduct: {
			name: '',
			price: 0,
			count: 0,
			sale: {
				one: {
					name: '',
					staticSale: 0,
					percentSale: 0,
					note: '',
					saleCode: ''
				},
				all: {
					name: '',
					staticSale: 0,
					percentSale: 0,
					note: '',
					saleCode: ''
				}
			}
		},
		detailSale: {
			name: '',
			staticSale: 0,
			percentSale: 0,
			note: '',
			saleCode: ''
		},
		globalSales: [],
		staticSales: 0,
		percentSales: 0,
		rawPrice: 0
	},
	mutations: {
		SET_VISIBLE_PRODUCTS(state, items) {
			state.visibleProducts = items.map(product => {
				product['count'] = 0;
				product['sale'] = {
					all: {
						name: '',
						staticSale: 0,
						percentSale: 0,
						note: '',
						saleCode: ''
					},
					one: {
						name: '',
						staticSale: 0,
						percentSale: 0,
						note: '',
						saleCode: ''
					}
				};
				return product
			});
		},

		ADD_ITEM_TO_CHECKOUT(state, item) {
			const foundProduct = state.checkoutProducts.filter(product => (product.id === item.id));
			if (foundProduct.length === 0) {
				state.checkoutProducts.push(item);
				item.count = 0;
			}
			item.count++;
		},

		REMOVE_ITEM_FROM_CHECKOUT(state, item) {
			item.count--;
		},
		REMOVE_ALL_PIECES_OF_ITEM_FROM_CHECKOUT(state, arrayId) {
			state.checkoutProducts.splice(arrayId, 1);
		},
		SET_VISIBLE_CATEGORIES(state, categories) {
			state.visibleCategories = categories;
		},
		SET_CATEGORIES(state, categories) {
			state.categories = categories;
		},
		RESET(state) {
			state.checkoutProducts = [];
			state.visibleProducts = [];
		},
		RECALCULATE(state) {
			startLoader();
			let sum = 0;
			let sumWithoutSales = 0;
			let staticSales = 0;
			let percentSales = 0;
			state.checkoutProducts.forEach(product => {

				// Standartni pocet kusu * cena
				sum += (parseFloat(product.price) * product.count);
				sumWithoutSales = sum;

				// Pevna sleva na vsechny
				if (product.sale.all.staticSale > 0) {
					sum -= product.sale.all.staticSale;
					staticSales += product.sale.all.staticSale;
				}

				// Procentuelni sleva na vsechny
				if (product.sale.all.percentSale > 0) {
					const sale = ((parseFloat(product.price) * product.count) / 100) * product.sale.all.percentSale;
					sum -= sale;
					staticSales += sale;
				}

				// Pevna sleva na kus
				if (product.sale.one.staticSale) {
					const sale = product.sale.one.staticSale * product.count;
					sum -= sale;
					staticSales += sale;
				}

				// Procentuelni sleva na kus
				if (product.sale.one.percentSale) {
					const sale = ((product.price / 100) * product.sale.one.percentSale) * product.count;
					sum -= sale;
					staticSales += sale;
				}

			});
			state.rawPrice = sumWithoutSales;
			state.globalSales.forEach(sale => {
				staticSales += parseFloat(sale.staticSale);
				percentSales += parseFloat(sale.percentSale);
				sum -= parseFloat(sale.staticSale);
			});

			state.staticSales = staticSales;
			state.percentSales = percentSales;

			state.totalPrice = sum - ((sum / 100) * percentSales);
			stopLoader();
		},

		// PRODUCT
		SET_PRODUCT_DETAIL(state, product) {
			state.detailProduct = product
		},
		EDIT_PRODUCT(state, product) {
			Vue.set(state.checkoutProducts, state.editedProductArrayId, product);
		},

		// SALES
		ADD_GLOBAL_SALE(state, sale) {
			state.globalSales.push(sale);
		},
		SET_DETAIL_SALE(state, sale) {
			state.detailSale = sale;
		},
		REMOVE_GLOBAL_SALE(state, sale) {
			let index = state.globalSales.indexOf(sale);
			if (index !== -1) state.globalSales.splice(index, 1);
		},
		EDIT_GLOBAL_SALE(state, sale) {
			console.log(sale);
			Vue.set(state.globalSales, state.editedSaleArrayId, sale);
		},

		// Breadcrumb
		ADD_TO_BREADCRUMB(state, breadcrumb) {
			if (state.breadcrumb.filter(item => item.id === breadcrumb.id))
				state.breadcrumb.push(breadcrumb);
		},

		// Modals
		SET_PRODUCT_DETAIL_MODAL(state, active) {
			state.showDetailProduct = active;
		},
		SET_SALE_DETAIL_MODAL(state, active) {
			state.showDetailSale = active;
			if (!active) state.editingSale = false;
		}
	},
	actions: {
		removeItemFromCheckout({commit}, item) {
			commit('REMOVE_ITEM_FROM_CHECKOUT', item);
			commit('RECALCULATE');
		},
		removeAllPiecesOfItemFromCheckout({commit}, arrayID) {
			commit('REMOVE_ALL_PIECES_OF_ITEM_FROM_CHECKOUT', arrayID);
			commit('RECALCULATE');
		},
		async getProductsFromCategory({commit, state}, categoryId) {
			startLoader();
			if (categoryId)
				commit('SET_VISIBLE_PRODUCTS', await fetchProductsFromCategory(categoryId))
			stopLoader();
		},
		async addItemToCheckout({commit}, item) {
			commit('ADD_ITEM_TO_CHECKOUT', item);
			commit('RECALCULATE');
		},
		checkoutCash({commit}) {

		},
		checkoutCreditCard({commit}) {

		},
		reset({commit}) {
			commit('RESET');
			commit('RECALCULATE');
		},
		async search({commit}, term = "") {
			commit('SET_VISIBLE_PRODUCTS', await searchProducts(term));
		},
		async getRootCategories({commit}) {
			startLoader();
			const categories = await getCategories();
			commit('SET_VISIBLE_CATEGORIES', categories);
			commit('SET_CATEGORIES', categories);
			commit('ADD_TO_BREADCRUMB', {index: 0, id: null, name: 'Eshop'});
			stopLoader();
		},
		getChildrenOfCategory({commit}, {category, index}) {
			if (category.child) {
				commit('SET_VISIBLE_CATEGORIES', category.child);
			} else {
				commit('SET_VISIBLE_CATEGORIES', []);
			}
			commit('ADD_TO_BREADCRUMB', {index, id: category.id, name: category.name})
		},

		// Products
		setProductToEdit({commit, state}, {arrayId, product}) {
			state.editedProductArrayId = arrayId;
			commit('SET_PRODUCT_DETAIL', deepcopy(product));
			commit('SET_PRODUCT_DETAIL_MODAL', true)
		},

		editProduct({commit}, product) {
			commit('EDIT_PRODUCT', deepcopy(product));
			commit('SET_PRODUCT_DETAIL_MODAL', false);
			commit('RECALCULATE');
		},

		// Sales
		setGlobalSaleToEdit({commit, state}, {sale, arrayId}) {
			state.editingSale = true;
			state.editedSaleArrayId = arrayId;
			commit('SET_DETAIL_SALE', deepcopy(sale));
			commit('SET_SALE_DETAIL_MODAL', true)
		},

		addGlobalSale({commit}, sale) {
			commit('ADD_GLOBAL_SALE', deepcopy(sale));
			commit('SET_SALE_DETAIL_MODAL', false);
			commit('RECALCULATE');
		},

		editGlobalSale({commit}, sale) {
			commit('EDIT_GLOBAL_SALE', deepcopy(sale));
			commit('SET_SALE_DETAIL_MODAL', false);
			commit('RECALCULATE');
		},
		removeGlobalSale({commit}, sale) {
			commit('REMOVE_GLOBAL_SALE', sale);
			commit('RECALCULATE');
		},

		// Breadcrumb
		refreshByBreadcrumbId({commit, state, dispatch}, breadcrumbId) {
			let categories = state.categories;
			if (breadcrumbId > 0) {
				categories = categories[state.breadcrumb[breadcrumbId].index];
				for (let i = 0; i < breadcrumbId; i++)
					categories = categories.child;
				dispatch('getProductsFromCategory', state.breadcrumb[breadcrumbId].id);
			} else {
				commit('SET_VISIBLE_PRODUCTS', []);
			}
			state.breadcrumb.splice(breadcrumbId + 1);
			commit('SET_VISIBLE_CATEGORIES', categories);
		},

		// Modals
		setProductModal({commit}, state) {
			commit('SET_PRODUCT_DETAIL_MODAL', state)
		},
		setSaleModal({commit}, state) {
			commit('SET_SALE_DETAIL_MODAL', state);
		}
	},
	getters: {}
})

const startLoader = () => {
	const body = document.querySelector('body');
	body.insertAdjacentHTML('beforeend', `<div id="loading-overlay" data-count="1">
		<div class="spinner">
		<div class="rect1"></div>
		<div class="rect2"></div>
		<div class="rect3"></div>
		<div class="rect4"></div>
		</div>
		</div>`);
};
const stopLoader = () => {
	document.querySelector('#loading-overlay').remove();
};
