<?php declare(strict_types = 1);

namespace EshopCheckout\ApiModule\Api\V1\Controllers;

use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestBody;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Core\Model\Entities\EntityManagerDecorator;
use EshopCheckout\ApiModule\Api\V1\Model\Terminals;
use EshopCheckout\Model\Entities\Terminal;
use Exception;
use Tracy\Debugger;

/**
 * @Path("/terminals")
 */
class TerminalsController extends BaseController
{
	protected EntityManagerDecorator $em;
	protected Terminals $terminals;

	public function __construct(EntityManagerDecorator $em, Terminals $terminals)
	{
		$this->em = $em;
		$this->terminals = $terminals;
	}

	/**
	 * @Path("/save")
	 * @Method("POST")
	 */
	public function save(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data = $request->getJsonBody();

		try {
			$this->em->getConnection()->beginTransaction();

			$terminal = null;
			if ($data['id']) {
				$terminal = $this->em->getReference(Terminal::class, (int) $data['id']);
			}

			$terminal = $terminal ?? new Terminal($data['ip'], $data['port'], $data['secureString'], $data['terminalId']);
			$terminal->ip = $data['ip'];
			$terminal->port = $data['port'];
			$terminal->secureString = $data['secureString'];
			$terminal->terminalId = $data['terminalId'];

			$this->em->persist($terminal)->flush();

			$this->em->getConnection()->commit();

		} catch (Exception $e) {
			Debugger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
				'status' => 'error',
				'data'   => [],
			]));
		}

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
			'status' => 'ok',
			'data'   => [
				'id'           => $terminal->getId(),
				'ip'           => $terminal->ip,
				'port'         => $terminal->port,
				'secureString' => $terminal->secureString,
				'isActive'     => $terminal->isActive,
				'terminalId'   => $terminal->terminalId,
			],
		]));
	}

	/**
	 * @Path("/get-all")
	 * @Method("GET")
	 */
	public function getAll(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->terminals->getAllAssoc();
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($result));
	}

	/**
	 * @Path("/set-active-terminal")
	 * @Method("POST")
	 */
	public function setActiveTerminal(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data = $request->getJsonBody();
		$terminalId = $data['terminalId'];

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
			'status' => $this->terminals->setActive((int) $terminalId) ? 'ok' : 'error'
		]));
	}

	/**
	 * @Path("/remove")
	 * @Method("POST")
	 */
	public function remove(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data = $request->getJsonBody();
		$terminalId = $data['terminalId'];

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
			'status' => $this->terminals->remove((int) $terminalId) ? 'ok' : 'error'
		]));
	}

}